<?php

namespace App\Support;

use App\Models\MessengerSetting;
use App\Models\User;

class MessengerAccess
{
    /**
     * @var array<string, string>
     */
    public const PROVIDER_OPTIONS = [
        'telegram' => 'Telegram',
        'whatsapp_business' => 'WhatsApp Business',
    ];

    /**
     * @var array<string, string>
     */
    public const NEW_CONTACT_ACTIONS = [
        'none' => 'Do nothing',
        'task' => 'Create task',
        'deal' => 'Create deal',
    ];

    /**
     * @return array<string, string>
     */
    public static function providerOptions(): array
    {
        return self::PROVIDER_OPTIONS;
    }

    /**
     * @return array<string, string>
     */
    public static function newContactActionOptions(): array
    {
        return self::NEW_CONTACT_ACTIONS;
    }

    public static function normalizeProvider(mixed $value): string
    {
        $normalized = strtolower(trim((string) $value));
        $normalized = str_replace([' ', '-', '.'], ['_', '_', ''], $normalized);

        if (in_array($normalized, ['telegram', 'tg'], true)) {
            return 'telegram';
        }

        if (in_array($normalized, ['whatsapp', 'wa', 'waba', 'whatsappbusiness', 'whatsapp_business'], true)) {
            return 'whatsapp_business';
        }

        return array_key_exists($normalized, self::PROVIDER_OPTIONS) ? $normalized : 'telegram';
    }

    /**
     * @return list<int>
     */
    public static function allowedOperatorIds(?MessengerSetting $settings): array
    {
        $source = $settings?->allowed_operator_ids;
        if (! is_array($source)) {
            return [];
        }

        $ids = [];
        foreach ($source as $value) {
            $id = (int) $value;
            if ($id > 0) {
                $ids[$id] = $id;
            }
        }

        return array_values($ids);
    }

    public static function firstAllowedOperatorId(?MessengerSetting $settings): ?int
    {
        $ids = self::allowedOperatorIds($settings);

        return $ids[0] ?? null;
    }

    public static function canAcceptIncoming(User $user, ?MessengerSetting $settings): bool
    {
        if (! AccessControl::allows($user, 'messengers', 'update')) {
            return false;
        }

        if (AccessControl::isElevated($user)) {
            return true;
        }

        return in_array($user->id, self::allowedOperatorIds($settings), true);
    }

    /**
     * @param  array<int, mixed>|null  $rawIds
     * @return list<int>
     */
    public static function sanitizeOperatorIds(?array $rawIds): array
    {
        if (! is_array($rawIds)) {
            return [];
        }

        $ids = [];
        foreach ($rawIds as $value) {
            $id = (int) $value;
            if ($id > 0) {
                $ids[$id] = $id;
            }
        }

        return array_values($ids);
    }
}

