<?php

namespace App\Http\Controllers;

use App\Models\Activity;
use App\Models\Deal;
use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Carbon\CarbonImmutable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;
use Illuminate\Support\Str;

class CalendarController extends Controller
{
    /**
     * @var array<string, string>
     */
    private const STATUS_COLORS = [
        'todo' => '#64748B',
        'in_progress' => '#2563EB',
        'review' => '#D97706',
        'done' => '#059669',
    ];

    /**
     * @var array<string, string>
     */
    private const STATUS_LABELS = [
        'todo' => 'To be carried out',
        'in_progress' => 'At work',
        'review' => 'Examination',
        'done' => 'Ready',
    ];

    /**
     * @var array<string, string>
     */
    private const PRIORITY_LABELS = [
        'low' => 'Short',
        'medium' => 'Average',
        'high' => 'High',
        'urgent' => 'Urgent',
    ];

    /**
     * @var array<string, string>
     */
    private const DEAL_STATUS_COLORS = [
        'open' => '#0EA5E9',
        'won' => '#10B981',
        'lost' => '#EF4444',
    ];

    /**
     * @var array<string, string>
     */
    private const DEAL_STATUS_LABELS = [
        'open' => 'Open',
        'won' => 'Won',
        'lost' => 'Lost',
    ];

    /**
     * @var array<string, string>
     */
    private const EVENT_TYPE_COLORS = [
        'task' => '#2563EB',
        'activity' => '#0EA5E9',
        'deal' => '#F59E0B',
    ];

    /**
     * @var array<string, string>
     */
    private const EVENT_TYPE_LABELS = [
        'task' => 'Tasks',
        'activity' => 'Activities',
        'deal' => 'Deals',
    ];

    public function index(Request $request): View
    {
        $this->authorize('viewAny', Task::class);

        $users = User::query()->orderBy('name')->get(['id', 'name', 'email', 'job_title']);
        $projects = Project::query()->orderBy('name')->get(['id', 'name']);

        $filters = [
            'q' => trim((string) $request->input('q', '')),
            'status' => trim((string) $request->input('status', '')),
            'priority' => trim((string) $request->input('priority', '')),
            'assignee_id' => $request->input('assignee_id', ''),
            'project_id' => $request->input('project_id', ''),
            'mine' => $request->boolean('mine'),
        ];

        $initialView = (string) $request->input('view', 'dayGridMonth');
        if (! in_array($initialView, ['dayGridMonth', 'timeGridWeek', 'timeGridDay', 'listWeek'], true)) {
            $initialView = 'dayGridMonth';
        }

        return view('calendar.index', [
            'users' => $users,
            'projects' => $projects,
            'filters' => $filters,
            'initialView' => $initialView,
            'statusLabels' => self::STATUS_LABELS,
            'priorityLabels' => self::PRIORITY_LABELS,
            'statusColors' => self::STATUS_COLORS,
            'eventTypeColors' => self::EVENT_TYPE_COLORS,
            'eventTypeLabels' => self::EVENT_TYPE_LABELS,
        ]);
    }

    public function events(Request $request): JsonResponse
    {
        $this->authorize('viewAny', Task::class);
        /** @var User $user */
        $user = $request->user();

        $validated = $request->validate([
            'start' => ['nullable', 'date'],
            'end' => ['nullable', 'date'],
            'q' => ['nullable', 'string', 'max:255'],
            'status' => ['nullable', 'string', 'max:80'],
            'priority' => ['nullable', 'string', 'max:80'],
            'assignee_id' => ['nullable', 'integer', 'exists:users,id'],
            'project_id' => ['nullable', 'integer', 'exists:projects,id'],
            'mine' => ['nullable', 'boolean'],
        ]);

        $rangeStart = isset($validated['start'])
            ? CarbonImmutable::parse((string) $validated['start'])
            : now()->startOfMonth()->subMonth()->toImmutable();
        $rangeEnd = isset($validated['end'])
            ? CarbonImmutable::parse((string) $validated['end'])
            : now()->endOfMonth()->addMonth()->toImmutable();

        $query = Task::query()
            ->with(['assignee:id,name', 'project:id,name'])
            ->where(function (Builder $builder) use ($rangeStart, $rangeEnd): void {
                $builder
                    ->whereBetween('starts_at', [$rangeStart, $rangeEnd])
                    ->orWhereBetween('due_at', [$rangeStart, $rangeEnd])
                    ->orWhere(function (Builder $sub) use ($rangeStart, $rangeEnd): void {
                        $sub->whereNotNull('starts_at')
                            ->whereNotNull('due_at')
                            ->where('starts_at', '<=', $rangeEnd)
                            ->where('due_at', '>=', $rangeStart);
                    })
                    ->orWhere(function (Builder $sub) use ($rangeStart, $rangeEnd): void {
                        $sub->whereNull('starts_at')
                            ->whereNull('due_at')
                            ->whereBetween('created_at', [$rangeStart, $rangeEnd]);
                    });
            });

        $search = trim((string) ($validated['q'] ?? ''));
        if ($search !== '') {
            $query->where(function (Builder $builder) use ($search): void {
                $builder->where('title', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%")
                    ->orWhereHas('project', fn (Builder $project) => $project->where('name', 'like', "%{$search}%"))
                    ->orWhereHas('assignee', fn (Builder $assignee) => $assignee->where('name', 'like', "%{$search}%"));
            });
        }

        $status = trim((string) ($validated['status'] ?? ''));
        if ($status !== '') {
            $query->where('status', $status);
        }

        $priority = trim((string) ($validated['priority'] ?? ''));
        if ($priority !== '') {
            $query->where('priority', $priority);
        }

        if (! empty($validated['assignee_id'])) {
            $query->where('assignee_id', (int) $validated['assignee_id']);
        }

        if (! empty($validated['project_id'])) {
            $query->where('project_id', (int) $validated['project_id']);
        }

        if ((bool) ($validated['mine'] ?? false)) {
            $userId = (int) $request->user()->id;
            $query->where(function (Builder $builder) use ($userId): void {
                $builder->where('creator_id', $userId)
                    ->orWhere('assignee_id', $userId);
            });
        }

        $tasks = $query
            ->orderByRaw('coalesce(starts_at, due_at, created_at)')
            ->orderBy('id')
            ->limit(3000)
            ->get();

        $events = $tasks->map(fn (Task $task): array => $this->taskToCalendarEvent($task))->values();

        if ($user->can('viewAny', Activity::class)) {
            $activityQuery = Activity::query()
                ->with(['user:id,name', 'deal:id,title'])
                ->whereBetween('occurred_at', [$rangeStart, $rangeEnd]);

            if ($search !== '') {
                $activityQuery->where(function (Builder $builder) use ($search): void {
                    $builder->where('subject', 'like', "%{$search}%")
                        ->orWhere('details', 'like', "%{$search}%")
                        ->orWhereHas('deal', fn (Builder $deal) => $deal->where('title', 'like', "%{$search}%"))
                        ->orWhereHas('user', fn (Builder $member) => $member->where('name', 'like', "%{$search}%"));
                });
            }

            if (! empty($validated['assignee_id'])) {
                $activityQuery->where('user_id', (int) $validated['assignee_id']);
            }

            if ((bool) ($validated['mine'] ?? false)) {
                $activityQuery->where('user_id', (int) $user->id);
            }

            $activities = $activityQuery
                ->orderBy('occurred_at')
                ->orderBy('id')
                ->limit(1500)
                ->get();

            $events = $events
                ->concat($activities->map(fn (Activity $activity): array => $this->activityToCalendarEvent($activity)));
        }

        if ($user->can('viewAny', Deal::class)) {
            $dealQuery = Deal::query()
                ->with(['owner:id,name', 'stage:id,name'])
                ->whereNotNull('expected_close_at')
                ->whereBetween('expected_close_at', [$rangeStart->toDateString(), $rangeEnd->toDateString()]);

            if ($search !== '') {
                $dealQuery->where(function (Builder $builder) use ($search): void {
                    $builder->where('title', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%");
                });
            }

            if (! empty($validated['assignee_id'])) {
                $dealQuery->where('owner_id', (int) $validated['assignee_id']);
            }

            if ((bool) ($validated['mine'] ?? false)) {
                $dealQuery->where('owner_id', (int) $user->id);
            }

            $dealStatus = trim((string) ($validated['status'] ?? ''));
            if (in_array($dealStatus, array_keys(self::DEAL_STATUS_LABELS), true)) {
                $dealQuery->where('status', $dealStatus);
            }

            $dealPriority = trim((string) ($validated['priority'] ?? ''));
            if (in_array($dealPriority, ['low', 'medium', 'high'], true)) {
                $dealQuery->where('priority', $dealPriority);
            }

            $deals = $dealQuery
                ->orderBy('expected_close_at')
                ->orderBy('id')
                ->limit(1500)
                ->get();

            $events = $events
                ->concat($deals->map(fn (Deal $deal): array => $this->dealToCalendarEvent($deal)));
        }

        $events = $events
            ->sortBy(fn (array $event): string => (string) ($event['start'] ?? ''))
            ->values();

        return response()->json([
            'events' => $events,
        ]);
    }

    public function updateTaskTiming(Request $request, Task $task): JsonResponse
    {
        $this->authorize('update', $task);

        $validated = $request->validate([
            'start' => ['nullable', 'date'],
            'end' => ['nullable', 'date'],
        ]);

        $hasStart = array_key_exists('start', $validated) && $validated['start'] !== null && $validated['start'] !== '';
        $hasEnd = array_key_exists('end', $validated) && $validated['end'] !== null && $validated['end'] !== '';

        if (! $hasStart && ! $hasEnd) {
            throw ValidationException::withMessages([
                'start' => 'Pass start and/or end.',
            ]);
        }

        $start = $hasStart ? CarbonImmutable::parse((string) $validated['start']) : null;
        $end = $hasEnd ? CarbonImmutable::parse((string) $validated['end']) : null;

        if ($start === null) {
            $start = $task->starts_at?->toImmutable()
                ?? $task->due_at?->toImmutable()
                ?? now()->toImmutable();
        }

        if ($end === null) {
            $end = $task->due_at?->toImmutable() ?? $start->addHour();
        }

        if ($end->lessThanOrEqualTo($start)) {
            $end = $start->addHour();
        }

        $task->update([
            'starts_at' => $start,
            'due_at' => $end,
        ]);

        $task->refresh()->load(['assignee:id,name', 'project:id,name']);

        return response()->json([
            'message' => 'Task deadlines have been updated.',
            'event' => $this->taskToCalendarEvent($task),
        ]);
    }

    /**
     * @return array<string, mixed>
     */
    private function taskToCalendarEvent(Task $task): array
    {
        $start = $task->starts_at?->toImmutable()
            ?? $task->due_at?->toImmutable()
            ?? $task->created_at?->toImmutable()
            ?? now()->toImmutable();

        $end = $task->due_at?->toImmutable() ?? $start->addHour();
        if ($end->lessThanOrEqualTo($start)) {
            $end = $start->addHour();
        }

        $status = (string) ($task->status ?: 'todo');
        $priority = (string) ($task->priority ?: 'medium');
        $statusLabel = self::STATUS_LABELS[$status] ?? Str::headline(str_replace('_', ' ', $status));
        $priorityLabel = self::PRIORITY_LABELS[$priority] ?? Str::headline(str_replace('_', ' ', $priority));
        $color = self::STATUS_COLORS[$status] ?? '#475569';

        return [
            'id' => 'task-'.$task->id,
            'title' => '#'.$task->id.' · '.$task->title,
            'start' => $start->toIso8601String(),
            'end' => $end->toIso8601String(),
            'allDay' => false,
            'editable' => true,
            'startEditable' => true,
            'durationEditable' => true,
            'url' => route('tasks.show', $task),
            'backgroundColor' => $color,
            'borderColor' => $color,
            'textColor' => '#ffffff',
            'extendedProps' => [
                'entity_type' => 'task',
                'task_id' => $task->id,
                'task_title' => $task->title,
                'status' => $status,
                'status_label' => $statusLabel,
                'priority' => $priority,
                'priority_label' => $priorityLabel,
                'assignee_name' => $task->assignee?->name,
                'project_name' => $task->project?->name,
            ],
        ];
    }

    /**
     * @return array<string, mixed>
     */
    private function activityToCalendarEvent(Activity $activity): array
    {
        $start = $activity->occurred_at?->toImmutable() ?? now()->toImmutable();
        $end = $start->addHour();
        $typeLabel = Str::headline((string) $activity->type);
        $color = self::EVENT_TYPE_COLORS['activity'];

        return [
            'id' => 'activity-'.$activity->id,
            'title' => $typeLabel.' · '.$activity->subject,
            'start' => $start->toIso8601String(),
            'end' => $end->toIso8601String(),
            'allDay' => false,
            'editable' => false,
            'startEditable' => false,
            'durationEditable' => false,
            'url' => $activity->deal
                ? route('deals.show', $activity->deal)
                : route('activities.show', $activity),
            'backgroundColor' => $color,
            'borderColor' => $color,
            'textColor' => '#ffffff',
            'extendedProps' => [
                'entity_type' => 'activity',
                'activity_id' => $activity->id,
                'status_label' => self::EVENT_TYPE_LABELS['activity'],
                'priority_label' => $typeLabel,
                'assignee_name' => $activity->user?->name,
                'project_name' => $activity->deal?->title,
            ],
        ];
    }

    /**
     * @return array<string, mixed>
     */
    private function dealToCalendarEvent(Deal $deal): array
    {
        $start = CarbonImmutable::parse((string) $deal->expected_close_at)->startOfDay();
        $status = (string) ($deal->status ?: 'open');
        $statusLabel = self::DEAL_STATUS_LABELS[$status] ?? Str::headline($status);
        $color = self::DEAL_STATUS_COLORS[$status] ?? self::EVENT_TYPE_COLORS['deal'];

        return [
            'id' => 'deal-'.$deal->id,
            'title' => '#'.$deal->id.' · '.$deal->title,
            'start' => $start->toDateString(),
            'end' => $start->addDay()->toDateString(),
            'allDay' => true,
            'editable' => false,
            'startEditable' => false,
            'durationEditable' => false,
            'url' => route('deals.show', $deal),
            'backgroundColor' => $color,
            'borderColor' => $color,
            'textColor' => '#ffffff',
            'extendedProps' => [
                'entity_type' => 'deal',
                'deal_id' => $deal->id,
                'status_label' => self::EVENT_TYPE_LABELS['deal'],
                'priority_label' => $statusLabel,
                'assignee_name' => $deal->owner?->name,
                'project_name' => $deal->stage?->name,
            ],
        ];
    }
}
