<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\ProductResource;
use App\Models\Product;
use App\Support\CrmModuleManager;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class ProductController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Product::class, 'product');
    }

    public function index(Request $request)
    {
        $search = trim((string) $request->input('q', ''));
        $status = trim((string) $request->input('status', ''));
        $companyId = $request->integer('company_id');
        $ownerId = $request->integer('owner_id');

        $products = Product::query()
            ->with(['company', 'owner'])
            ->when($search !== '', function ($query) use ($search): void {
                $query->where(function ($sub) use ($search): void {
                    $sub->where('name', 'like', "%{$search}%")
                        ->orWhere('sku', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%");
                });
            })
            ->when(in_array($status, ['active', 'archived'], true), fn ($query) => $query->where('status', $status))
            ->when($companyId > 0, fn ($query) => $query->where('company_id', $companyId))
            ->when($ownerId > 0, fn ($query) => $query->where('owner_id', $ownerId))
            ->latest('id')
            ->paginate(20)
            ->withQueryString();

        return ProductResource::collection($products);
    }

    public function store(Request $request, CrmModuleManager $moduleManager)
    {
        $payload = $this->validatedData($request);
        $payload = $moduleManager->applyPayloadHooks('products.store', $payload, [
            'hook' => 'products.store',
            'user_id' => $request->user()->id,
            'source' => 'api',
        ], array_keys($payload));

        $product = Product::create($payload);

        return ProductResource::make($product->load(['company', 'owner']))
            ->response()
            ->setStatusCode(201);
    }

    public function show(Product $product): ProductResource
    {
        return ProductResource::make($product->load(['company', 'owner']));
    }

    public function update(Request $request, Product $product, CrmModuleManager $moduleManager): ProductResource
    {
        $payload = $this->validatedData($request, $product);
        $payload = $moduleManager->applyPayloadHooks('products.update', $payload, [
            'hook' => 'products.update',
            'user_id' => $request->user()->id,
            'product_id' => $product->id,
            'source' => 'api',
        ], array_keys($payload));

        $product->update($payload);

        return ProductResource::make($product->load(['company', 'owner']));
    }

    public function destroy(Product $product)
    {
        $product->delete();

        return response()->noContent();
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedData(Request $request, ?Product $product = null): array
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'sku' => [
                'nullable',
                'string',
                'max:100',
                Rule::unique('products', 'sku')->ignore($product?->id),
            ],
            'price' => ['nullable', 'numeric', 'min:0'],
            'currency' => ['required', 'string', 'size:3'],
            'unit' => ['required', 'string', 'max:32'],
            'stock' => ['nullable', 'numeric', 'min:0'],
            'status' => ['required', Rule::in(['active', 'archived'])],
            'company_id' => ['nullable', 'exists:companies,id'],
            'owner_id' => ['nullable', 'exists:users,id'],
            'description' => ['nullable', 'string'],
        ]);

        $validated['price'] = (float) ($validated['price'] ?? 0);
        $validated['stock'] = (float) ($validated['stock'] ?? 0);
        $validated['currency'] = strtoupper((string) $validated['currency']);

        return $validated;
    }
}
