@php
    $mapRows = max(1, (int) ($warehouse->map_rows ?? 8));
    $mapColumns = max(1, (int) ($warehouse->map_columns ?? 12));
    $addressLookup = $addressLookup ?? collect();
    $addressStatuses = $addressStatuses ?? [
        'free' => __('Free'),
        'reserved' => __('Reserved'),
        'blocked' => __('Blocked'),
    ];
@endphp

<x-app-layout>
    <x-slot name="header">
        <div class="flex items-center justify-between gap-4">
            <div>
                <h2 class="font-semibold text-xl text-gray-800 leading-tight">{{ $warehouse->name }}</h2>
                <p class="text-sm text-gray-500">{{ $warehouse->code ?: '—' }}</p>
            </div>
            <a href="{{ route('warehouses.edit', $warehouse) }}" class="inline-flex items-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white hover:bg-indigo-500">{{ __('Edit') }}</a>
        </div>
    </x-slot>

    <div class="pb-12">
        <div class="w-full px-6 space-y-6">
            @if ($errors->any())
                <div class="rounded-xl border border-red-200 bg-red-50 px-4 py-3 text-sm text-red-700">
                    <ul class="list-disc pl-5 space-y-1">
                        @foreach ($errors->all() as $error)
                            <li>{{ $error }}</li>
                        @endforeach
                    </ul>
                </div>
            @endif

            <section class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div class="bg-white rounded-xl border border-gray-200 p-5 lg:col-span-2">
                    <h3 class="font-semibold text-gray-900">{{ __('Details') }}</h3>
                    <dl class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                        <div>
                            <dt class="text-gray-500">{{ __('Location') }}</dt>
                            <dd class="text-gray-900">{{ $warehouse->location ?: '—' }}</dd>
                        </div>
                        <div>
                            <dt class="text-gray-500">{{ __('Capacity') }}</dt>
                            <dd class="text-gray-900">{{ number_format((float) $warehouse->capacity, 2, '.', ' ') }}</dd>
                        </div>
                        <div>
                            <dt class="text-gray-500">{{ __('Responsible') }}</dt>
                            <dd class="text-gray-900">{{ $warehouse->manager?->name ?: '—' }}</dd>
                        </div>
                        <div>
                            <dt class="text-gray-500">{{ __('Status') }}</dt>
                            <dd class="text-gray-900">{{ $warehouse->status === 'inactive' ? __('Inactive') : __('Active') }}</dd>
                        </div>
                        <div>
                            <dt class="text-gray-500">{{ __('Map rows') }}</dt>
                            <dd class="text-gray-900">{{ $mapRows }}</dd>
                        </div>
                        <div>
                            <dt class="text-gray-500">{{ __('Map columns') }}</dt>
                            <dd class="text-gray-900">{{ $mapColumns }}</dd>
                        </div>
                    </dl>

                    <div class="mt-4 text-sm">
                        <p class="text-gray-500">{{ __('Description') }}</p>
                        <p class="mt-1 text-gray-900 whitespace-pre-line">{{ $warehouse->description ?: '—' }}</p>
                    </div>
                </div>

                <div class="bg-white rounded-xl border border-gray-200 p-5 space-y-4">
                    <h3 class="font-semibold text-gray-900">{{ __('Address storage') }}</h3>
                    <p class="text-xs text-gray-500">{{ __('Build warehouse map and place address cells by coordinates.') }}</p>

                    @can('update', $warehouse)
                        <form method="POST" action="{{ route('warehouses.map.update', $warehouse) }}" class="space-y-3">
                            @csrf
                            @method('PATCH')
                            <div class="grid grid-cols-2 gap-2">
                                <div>
                                    <label for="map_rows_quick" class="block text-xs font-medium text-gray-600">{{ __('Rows') }}</label>
                                    <input id="map_rows_quick" name="map_rows" type="number" min="1" max="200" value="{{ old('map_rows', $mapRows) }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                </div>
                                <div>
                                    <label for="map_columns_quick" class="block text-xs font-medium text-gray-600">{{ __('Columns') }}</label>
                                    <input id="map_columns_quick" name="map_columns" type="number" min="1" max="200" value="{{ old('map_columns', $mapColumns) }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                </div>
                            </div>
                            <button type="submit" class="w-full rounded-md bg-slate-800 px-3 py-2 text-sm font-medium text-white hover:bg-slate-700">
                                {{ __('Save map settings') }}
                            </button>
                        </form>
                    @endcan
                </div>
            </section>

            <section class="bg-white rounded-xl border border-gray-200 p-5 space-y-4">
                <div class="flex items-center justify-between gap-3">
                    <h3 class="font-semibold text-gray-900">{{ __('Warehouse map') }}</h3>
                    <span class="text-xs text-gray-500">{{ __('Rows') }}: {{ $mapRows }} | {{ __('Columns') }}: {{ $mapColumns }}</span>
                </div>

                @can('update', $warehouse)
                    <form method="POST" action="{{ route('warehouses.addresses.generate', $warehouse) }}" class="grid grid-cols-1 gap-3 rounded-lg border border-gray-200 bg-gray-50 p-3 md:grid-cols-[minmax(0,1fr)_minmax(0,1fr)_130px_auto]">
                        @csrf
                        <div>
                            <label for="generate_prefix" class="block text-xs font-medium text-gray-600">{{ __('Code prefix') }}</label>
                            <input id="generate_prefix" type="text" name="prefix" value="A" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label for="generate_zone" class="block text-xs font-medium text-gray-600">{{ __('Zone') }}</label>
                            <input id="generate_zone" type="text" name="zone" placeholder="{{ __('Main') }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label for="generate_color" class="block text-xs font-medium text-gray-600">{{ __('Color') }}</label>
                            <input id="generate_color" type="color" name="color" value="#0EA5E9" class="mt-1 h-10 w-full rounded-md border-gray-300 bg-white shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div class="flex items-end gap-3">
                            <label class="inline-flex items-center gap-2 text-xs text-gray-700">
                                <input type="checkbox" name="clear_existing" value="1" class="rounded border-gray-300 text-indigo-600 shadow-sm focus:ring-indigo-500">
                                {{ __('Clear existing') }}
                            </label>
                            <button type="submit" class="rounded-md bg-indigo-600 px-3 py-2 text-sm font-medium text-white hover:bg-indigo-500">
                                {{ __('Generate map') }}
                            </button>
                        </div>
                    </form>
                @endcan

                <div class="overflow-x-auto rounded-lg border border-gray-200 bg-gray-50 p-3">
                    <div
                        class="grid gap-1"
                        style="grid-template-columns: repeat({{ $mapColumns }}, minmax(72px, 1fr)); min-width: {{ max(680, $mapColumns * 74) }}px;"
                    >
                        @for ($y = 1; $y <= $mapRows; $y++)
                            @for ($x = 1; $x <= $mapColumns; $x++)
                                @php
                                    $address = $addressLookup->get($x.'-'.$y);
                                @endphp

                                @if ($address)
                                    <div class="h-16 rounded-md border border-black/10 px-2 py-1 text-[11px] leading-tight text-white shadow-sm" style="background: {{ $address->color ?: '#0EA5E9' }}">
                                        <p class="truncate font-semibold">{{ $address->code }}</p>
                                        <p class="truncate opacity-90">{{ $addressStatuses[$address->status] ?? ucfirst((string) $address->status) }}</p>
                                        <p class="truncate opacity-90">{{ __('Load') }}: {{ number_format((float) $address->current_load, 3, '.', ' ') }}</p>
                                    </div>
                                @else
                                    <div class="h-16 rounded-md border border-dashed border-slate-300 bg-white/80"></div>
                                @endif
                            @endfor
                        @endfor
                    </div>
                </div>
            </section>

            <section class="bg-white rounded-xl border border-gray-200 p-5 space-y-4">
                <div class="flex items-center justify-between gap-3">
                    <h3 class="font-semibold text-gray-900">{{ __('Storage addresses') }}</h3>
                    <span class="text-xs text-gray-500">{{ __('Total') }}: {{ $warehouse->addresses->count() }}</span>
                </div>

                @can('update', $warehouse)
                    <form method="POST" action="{{ route('warehouses.addresses.store', $warehouse) }}" class="grid grid-cols-1 gap-3 rounded-lg border border-gray-200 bg-gray-50 p-3 lg:grid-cols-4">
                        @csrf
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Code') }}</label>
                            <input type="text" name="code" required class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Zone') }}</label>
                            <input type="text" name="zone" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Aisle') }}</label>
                            <input type="text" name="aisle" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Rack') }}</label>
                            <input type="text" name="rack" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Shelf') }}</label>
                            <input type="text" name="shelf" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Cell') }}</label>
                            <input type="text" name="cell" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('X') }}</label>
                            <input type="number" name="x" min="1" max="{{ $mapColumns }}" required class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Y') }}</label>
                            <input type="number" name="y" min="1" max="{{ $mapRows }}" required class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Capacity') }}</label>
                            <input type="number" name="capacity" min="0" step="0.001" value="0" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Current load') }}</label>
                            <input type="number" name="current_load" min="0" step="0.001" value="0" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Status') }}</label>
                            <select name="status" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                @foreach ($addressStatuses as $statusKey => $statusLabel)
                                    <option value="{{ $statusKey }}">{{ $statusLabel }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div>
                            <label class="block text-xs font-medium text-gray-600">{{ __('Color') }}</label>
                            <input type="color" name="color" value="#0EA5E9" class="mt-1 h-10 w-full rounded-md border-gray-300 bg-white shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div class="lg:col-span-4">
                            <label class="block text-xs font-medium text-gray-600">{{ __('Note') }}</label>
                            <input type="text" name="note" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div class="lg:col-span-4 flex justify-end">
                            <button type="submit" class="rounded-md bg-slate-800 px-4 py-2 text-sm font-medium text-white hover:bg-slate-700">{{ __('Add address') }}</button>
                        </div>
                    </form>
                @endcan

                <div class="space-y-3">
                    @forelse ($warehouse->addresses as $address)
                        <form method="POST" action="{{ route('warehouses.addresses.update', [$warehouse, $address]) }}" class="grid grid-cols-1 gap-3 rounded-lg border border-gray-200 p-3 lg:grid-cols-6">
                            @csrf
                            @method('PATCH')

                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Code') }}</label>
                                <input type="text" name="code" value="{{ $address->code }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Zone') }}</label>
                                <input type="text" name="zone" value="{{ $address->zone }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Aisle') }}</label>
                                <input type="text" name="aisle" value="{{ $address->aisle }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Rack') }}</label>
                                <input type="text" name="rack" value="{{ $address->rack }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Shelf') }}</label>
                                <input type="text" name="shelf" value="{{ $address->shelf }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Cell') }}</label>
                                <input type="text" name="cell" value="{{ $address->cell }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('X') }}</label>
                                <input type="number" name="x" min="1" max="{{ $mapColumns }}" value="{{ $address->x }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Y') }}</label>
                                <input type="number" name="y" min="1" max="{{ $mapRows }}" value="{{ $address->y }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Capacity') }}</label>
                                <input type="number" name="capacity" min="0" step="0.001" value="{{ number_format((float) $address->capacity, 3, '.', '') }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Current load') }}</label>
                                <input type="number" name="current_load" min="0" step="0.001" value="{{ number_format((float) $address->current_load, 3, '.', '') }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Status') }}</label>
                                <select name="status" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                                    @foreach ($addressStatuses as $statusKey => $statusLabel)
                                        <option value="{{ $statusKey }}" @selected($address->status === $statusKey)>{{ $statusLabel }}</option>
                                    @endforeach
                                </select>
                            </div>
                            <div>
                                <label class="block text-xs font-medium text-gray-600">{{ __('Color') }}</label>
                                <input type="color" name="color" value="{{ $address->color ?: '#0EA5E9' }}" class="mt-1 h-10 w-full rounded-md border-gray-300 bg-white shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>
                            <div class="lg:col-span-6">
                                <label class="block text-xs font-medium text-gray-600">{{ __('Note') }}</label>
                                <input type="text" name="note" value="{{ $address->note }}" class="mt-1 w-full rounded-md border-gray-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" @disabled(!auth()->user()->can('update', $warehouse))>
                            </div>

                            @can('update', $warehouse)
                                <div class="lg:col-span-6 flex items-center justify-end gap-3">
                                    <button type="submit" class="rounded-md bg-indigo-600 px-3 py-2 text-sm font-medium text-white hover:bg-indigo-500">{{ __('Save') }}</button>
                                </div>
                            @endcan
                        </form>

                        @can('update', $warehouse)
                            <form method="POST" action="{{ route('warehouses.addresses.destroy', [$warehouse, $address]) }}" class="mt-2 flex justify-end" onsubmit="return confirm('{{ __('Delete storage address?') }}');">
                                @csrf
                                @method('DELETE')
                                <button type="submit" class="rounded-md bg-red-50 px-3 py-2 text-sm font-medium text-red-700 hover:bg-red-100">
                                    {{ __('Delete') }}
                                </button>
                            </form>
                        @endcan
                    @empty
                        <div class="rounded-lg border border-dashed border-gray-300 bg-gray-50 px-4 py-6 text-sm text-gray-500">
                            {{ __('No storage addresses yet.') }}
                        </div>
                    @endforelse
                </div>
            </section>
        </div>
    </div>
</x-app-layout>
