@php
    $settingsSection = isset($settingsSection) && is_string($settingsSection) ? $settingsSection : null;
    $organizationSetting = $organizationSetting ?? null;
    $organizationCompaniesCollection = ($organizationCompanies ?? collect()) instanceof \Illuminate\Support\Collection
        ? $organizationCompanies
        : collect($organizationCompanies ?? []);
    $companiesById = $organizationCompaniesCollection->keyBy('id');
    $oldCompanies = old('companies');
    $maxCompanies = 30;

    if (is_array($oldCompanies)) {
        $companies = collect($oldCompanies)->map(function ($row) use ($companiesById): array {
            $id = (int) ($row['id'] ?? 0);
            $existing = $id > 0 ? $companiesById->get($id) : null;

            return [
                'id' => $id > 0 ? $id : null,
                'name' => trim((string) ($row['name'] ?? '')),
                'requisites' => is_string($row['requisites'] ?? null) ? $row['requisites'] : '',
                'address' => is_string($row['address'] ?? null) ? $row['address'] : '',
                'phones' => is_string($row['phones'] ?? null) ? $row['phones'] : '',
                'logo_path' => (string) ($existing?->logo_path ?? ''),
                'remove_logo' => (bool) ($row['remove_logo'] ?? false),
                'delete' => (bool) ($row['delete'] ?? false),
                'parent_id' => isset($row['parent_id']) ? (int) $row['parent_id'] : null,
                'parent_index' => isset($row['parent_index']) && $row['parent_index'] !== '' ? (int) $row['parent_index'] : null,
            ];
        })->values()->all();
    } else {
        $companies = $organizationCompaniesCollection->map(function ($company): array {
            return [
                'id' => $company->id,
                'name' => (string) $company->name,
                'requisites' => (string) ($company->requisites ?? ''),
                'address' => (string) ($company->address ?? ''),
                'phones' => method_exists($company, 'phonesAsText') ? $company->phonesAsText() : '',
                'logo_path' => (string) ($company->logo_path ?? ''),
                'remove_logo' => false,
                'delete' => false,
                'parent_id' => $company->parent_id,
                'parent_index' => null,
            ];
        })->values()->all();
    }

    if ($companies === []) {
        $companies = [[
            'id' => null,
            'name' => '',
            'requisites' => '',
            'address' => '',
            'phones' => '',
            'logo_path' => '',
            'remove_logo' => false,
            'delete' => false,
            'parent_id' => null,
            'parent_index' => null,
        ]];
    }

    $groupCompaniesEnabled = (bool) old(
        'group_companies_enabled',
        (bool) ($organizationSetting?->group_companies_enabled ?? false)
    );
@endphp

<section class="space-y-6" data-company-settings-form data-next-index="{{ count($companies) }}" data-max-companies="{{ $maxCompanies }}">
    <header>
        <h2 class="text-lg font-medium text-gray-900">{{ __('Company settings') }}</h2>
        <p class="mt-1 text-sm text-gray-600">
            {{ __('Maintain internal company profile data used in the CRM workspace.') }}
        </p>
    </header>

    @if (session('status') === 'company-settings-updated')
        <div class="rounded-lg border border-emerald-200 bg-emerald-50 px-4 py-3 text-sm text-emerald-800">
            {{ __('Company settings have been updated.') }}
        </div>
    @endif

    @if ($errors->has('group_companies_enabled') || $errors->has('companies') || $errors->has('companies.*'))
        <div class="rounded-lg border border-rose-200 bg-rose-50 px-4 py-3 text-sm text-rose-800">
            {{ __('Please check the company settings form fields.') }}
        </div>
    @endif

    <form method="POST" action="{{ route('profile.company-settings.update') }}" enctype="multipart/form-data" class="space-y-4">
        @csrf
        @method('PATCH')
        @if ($settingsSection)
            <input type="hidden" name="_settings_section" value="{{ $settingsSection }}">
        @endif

        <label class="inline-flex items-center gap-2 rounded-md border border-slate-200 bg-slate-50 px-3 py-2 text-sm text-slate-700">
            <input
                type="checkbox"
                name="group_companies_enabled"
                value="1"
                data-group-toggle
                class="rounded border-slate-300 text-indigo-600 shadow-sm focus:ring-indigo-500"
                @checked($groupCompaniesEnabled)
            >
            <span>{{ __('Group of companies') }}</span>
        </label>

        <div class="space-y-3" data-company-list>
            @foreach ($companies as $index => $company)
                @php
                    $isDeleted = (bool) ($company['delete'] ?? false);
                    $logoPath = (string) ($company['logo_path'] ?? '');
                    $logoUrl = $logoPath !== '' ? \Illuminate\Support\Facades\Storage::disk('public')->url($logoPath) : null;
                @endphp
                <div class="rounded-lg border border-slate-200 bg-white p-4 shadow-sm {{ $isDeleted ? 'hidden' : '' }}" data-company-row data-row-index="{{ $index }}">
                    <input type="hidden" name="companies[{{ $index }}][id]" value="{{ $company['id'] }}" data-company-id>
                    <input type="hidden" name="companies[{{ $index }}][delete]" value="{{ $isDeleted ? '1' : '0' }}" data-delete-flag>
                    <input type="hidden" name="companies[{{ $index }}][parent_index]" value="{{ $company['parent_index'] }}" data-parent-index>

                    <div class="grid grid-cols-1 gap-4 lg:grid-cols-2">
                        <div>
                            <label class="block text-sm font-medium text-slate-700">{{ __('Company name') }}</label>
                            <input type="text" name="companies[{{ $index }}][name]" value="{{ $company['name'] }}" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" required data-company-name>
                            <x-input-error class="mt-1" :messages="$errors->get('companies.'.$index.'.name')" />
                        </div>

                        <div data-parent-select-wrap class="hidden">
                            <label class="block text-sm font-medium text-slate-700">{{ __('Parent company') }}</label>
                            <select name="companies[{{ $index }}][parent_id]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" data-parent-select>
                                <option value="">{{ __('Main company') }}</option>
                                @foreach ($organizationCompaniesCollection as $parentOption)
                                    @if ((int) $parentOption->id !== (int) ($company['id'] ?? 0))
                                        <option value="{{ $parentOption->id }}" @selected((int) ($company['parent_id'] ?? 0) === (int) $parentOption->id)>
                                            {{ $parentOption->name }}
                                        </option>
                                    @endif
                                @endforeach
                            </select>
                        </div>

                        <div data-parent-badge class="hidden">
                            <label class="block text-sm font-medium text-slate-700">{{ __('Parent company') }}</label>
                            <div class="mt-1 rounded-md border border-slate-200 bg-slate-50 px-3 py-2 text-sm text-slate-700">
                                <span data-parent-label>{{ __('Not selected') }}</span>
                            </div>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-slate-700">{{ __('Logo') }}</label>
                            <input type="file" name="companies[{{ $index }}][logo]" accept="image/*" class="mt-1 block w-full text-sm text-slate-700 file:mr-3 file:rounded-md file:border-0 file:bg-indigo-50 file:px-3 file:py-2 file:text-sm file:font-medium file:text-indigo-700 hover:file:bg-indigo-100">
                            <x-input-error class="mt-1" :messages="$errors->get('companies.'.$index.'.logo')" />
                            @if ($logoUrl)
                                <div class="mt-2 flex items-center gap-3">
                                    <img src="{{ $logoUrl }}" alt="{{ $company['name'] }}" class="h-10 w-10 rounded-md border border-slate-200 object-cover">
                                    <label class="inline-flex items-center gap-2 text-xs text-slate-600">
                                        <input type="checkbox" name="companies[{{ $index }}][remove_logo]" value="1" class="rounded border-slate-300 text-indigo-600 shadow-sm focus:ring-indigo-500" @checked((bool) ($company['remove_logo'] ?? false))>
                                        <span>{{ __('Delete current photo') }}</span>
                                    </label>
                                </div>
                            @endif
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-slate-700">{{ __('Requisites') }}</label>
                            <textarea name="companies[{{ $index }}][requisites]" rows="4" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">{{ $company['requisites'] }}</textarea>
                            <x-input-error class="mt-1" :messages="$errors->get('companies.'.$index.'.requisites')" />
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-slate-700">{{ __('Address') }}</label>
                            <textarea name="companies[{{ $index }}][address]" rows="4" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">{{ $company['address'] }}</textarea>
                            <x-input-error class="mt-1" :messages="$errors->get('companies.'.$index.'.address')" />
                        </div>

                        <div class="lg:col-span-2">
                            <label class="block text-sm font-medium text-slate-700">{{ __('Phones (one per line)') }}</label>
                            <textarea name="companies[{{ $index }}][phones]" rows="3" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">{{ $company['phones'] }}</textarea>
                            <x-input-error class="mt-1" :messages="$errors->get('companies.'.$index.'.phones')" />
                        </div>
                    </div>

                    <div class="mt-3 flex flex-wrap items-center justify-end gap-2">
                        <button type="button" class="inline-flex items-center rounded-md border border-indigo-200 px-3 py-1.5 text-sm font-medium text-indigo-700 hover:bg-indigo-50" data-add-child>
                            {{ __('Add child company') }}
                        </button>
                        <button type="button" class="inline-flex items-center rounded-md border border-rose-200 px-3 py-1.5 text-sm font-medium text-rose-700 hover:bg-rose-50" data-remove-company>
                            {{ __('Delete company') }}
                        </button>
                    </div>
                </div>
            @endforeach
        </div>

        <p class="hidden rounded-md border border-slate-200 bg-slate-50 px-3 py-2 text-sm text-slate-500" data-empty-note>
            {{ __('There are no companies in settings yet. Add at least one card.') }}
        </p>

        <div class="flex flex-wrap items-center gap-3">
            <button type="button" class="inline-flex items-center rounded-md border border-slate-300 px-3 py-2 text-sm font-medium text-slate-700 hover:bg-slate-50" data-add-company data-label-default="{{ __('Add company') }}" data-label-group="{{ __('Add one more') }}">
                {{ __('Add company') }}
            </button>
            <x-primary-button>{{ __('Save') }}</x-primary-button>
        </div>
    </form>

    <template data-company-template>
        <div class="rounded-lg border border-slate-200 bg-white p-4 shadow-sm" data-company-row>
            <input type="hidden" name="companies[__INDEX__][id]" value="" data-company-id>
            <input type="hidden" name="companies[__INDEX__][delete]" value="0" data-delete-flag>
            <input type="hidden" name="companies[__INDEX__][parent_index]" value="" data-parent-index>

            <div class="grid grid-cols-1 gap-4 lg:grid-cols-2">
                <div>
                    <label class="block text-sm font-medium text-slate-700">{{ __('Company name') }}</label>
                    <input type="text" name="companies[__INDEX__][name]" value="" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" required data-company-name>
                </div>

                <div data-parent-select-wrap class="hidden">
                    <label class="block text-sm font-medium text-slate-700">{{ __('Parent company') }}</label>
                    <select name="companies[__INDEX__][parent_id]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" data-parent-select>
                        <option value="">{{ __('Main company') }}</option>
                        @foreach ($organizationCompaniesCollection as $parentOption)
                            <option value="{{ $parentOption->id }}">{{ $parentOption->name }}</option>
                        @endforeach
                    </select>
                </div>

                <div data-parent-badge class="hidden">
                    <label class="block text-sm font-medium text-slate-700">{{ __('Parent company') }}</label>
                    <div class="mt-1 rounded-md border border-slate-200 bg-slate-50 px-3 py-2 text-sm text-slate-700">
                        <span data-parent-label>{{ __('Not selected') }}</span>
                    </div>
                </div>

                <div>
                    <label class="block text-sm font-medium text-slate-700">{{ __('Logo') }}</label>
                    <input type="file" name="companies[__INDEX__][logo]" accept="image/*" class="mt-1 block w-full text-sm text-slate-700 file:mr-3 file:rounded-md file:border-0 file:bg-indigo-50 file:px-3 file:py-2 file:text-sm file:font-medium file:text-indigo-700 hover:file:bg-indigo-100">
                </div>

                <div>
                    <label class="block text-sm font-medium text-slate-700">{{ __('Requisites') }}</label>
                    <textarea name="companies[__INDEX__][requisites]" rows="4" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"></textarea>
                </div>

                <div>
                    <label class="block text-sm font-medium text-slate-700">{{ __('Address') }}</label>
                    <textarea name="companies[__INDEX__][address]" rows="4" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"></textarea>
                </div>

                <div class="lg:col-span-2">
                    <label class="block text-sm font-medium text-slate-700">{{ __('Phones (one per line)') }}</label>
                    <textarea name="companies[__INDEX__][phones]" rows="3" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"></textarea>
                </div>
            </div>

            <div class="mt-3 flex flex-wrap items-center justify-end gap-2">
                <button type="button" class="inline-flex items-center rounded-md border border-indigo-200 px-3 py-1.5 text-sm font-medium text-indigo-700 hover:bg-indigo-50" data-add-child>
                    {{ __('Add child company') }}
                </button>
                <button type="button" class="inline-flex items-center rounded-md border border-rose-200 px-3 py-1.5 text-sm font-medium text-rose-700 hover:bg-rose-50" data-remove-company>
                    {{ __('Delete company') }}
                </button>
            </div>
        </div>
    </template>
</section>

<script>
    (function () {
        const root = document.querySelector('[data-company-settings-form]');
        if (!root || root.dataset.initialized === '1') {
            return;
        }
        root.dataset.initialized = '1';

        const list = root.querySelector('[data-company-list]');
        const template = root.querySelector('template[data-company-template]');
        const addButton = root.querySelector('[data-add-company]');
        const groupToggle = root.querySelector('[data-group-toggle]');
        const emptyNote = root.querySelector('[data-empty-note]');
        let nextIndex = Number(root.dataset.nextIndex || 0);
        const maxCompanies = Number(root.dataset.maxCompanies || 30);
        const labelDefault = addButton?.dataset.labelDefault || '';
        const labelGroup = addButton?.dataset.labelGroup || labelDefault;

        const activeRows = () => Array.from(list.querySelectorAll('[data-company-row]'))
            .filter((row) => !row.classList.contains('hidden'));

        const refreshControls = () => {
            const rowsCount = activeRows().length;
            const canAdd = groupToggle.checked
                ? rowsCount < maxCompanies
                : rowsCount === 0;
            addButton.disabled = !canAdd;
            addButton.classList.toggle('cursor-not-allowed', !canAdd);
            addButton.classList.toggle('opacity-50', !canAdd);
            emptyNote.classList.toggle('hidden', rowsCount > 0);
            if (addButton) {
                addButton.textContent = groupToggle.checked ? labelGroup : labelDefault;
            }

            list.querySelectorAll('[data-add-child]').forEach((button) => {
                button.classList.toggle('hidden', !groupToggle.checked);
                button.disabled = !groupToggle.checked || rowsCount >= maxCompanies;
                button.classList.toggle('opacity-50', button.disabled);
                button.classList.toggle('cursor-not-allowed', button.disabled);
            });

            list.querySelectorAll('[data-company-row]').forEach((row) => {
                const hasParentIndex = (row.dataset.parentIndex || '') !== '';
                const selectWrap = row.querySelector('[data-parent-select-wrap]');
                const badge = row.querySelector('[data-parent-badge]');

                if (selectWrap) {
                    selectWrap.classList.toggle('hidden', !groupToggle.checked || hasParentIndex);
                }
                if (badge) {
                    badge.classList.toggle('hidden', !groupToggle.checked || !hasParentIndex);
                }
            });
        };

        const updateParentLabel = (row, parentIndex) => {
            const label = row.querySelector('[data-parent-label]');
            if (!label) {
                return;
            }
            const parentRow = list.querySelector(`[data-row-index="${parentIndex}"]`);
            const parentName = parentRow?.querySelector('[data-company-name]')?.value?.trim();
            label.textContent = parentName || label.textContent || '';
        };

        const applyParentIndex = (row, parentIndex) => {
            const parentIndexInput = row.querySelector('[data-parent-index]');
            if (parentIndexInput) {
                parentIndexInput.value = String(parentIndex);
            }
            row.dataset.parentIndex = String(parentIndex);
            updateParentLabel(row, parentIndex);

            const parentBadge = row.querySelector('[data-parent-badge]');
            if (parentBadge) {
                parentBadge.classList.remove('hidden');
            }

            const parentSelect = row.querySelector('[data-parent-select]');
            if (parentSelect) {
                parentSelect.value = '';
                parentSelect.disabled = true;
                parentSelect.classList.add('opacity-50', 'cursor-not-allowed');
            }
        };

        function addRow({ parentIndex = null } = {}) {
            if (addButton?.disabled || !template) {
                return;
            }

            const html = template.innerHTML.replace(/__INDEX__/g, String(nextIndex++));
            const wrapper = document.createElement('div');
            wrapper.innerHTML = html.trim();

            const row = wrapper.firstElementChild;
            if (!row) {
                return;
            }

            row.dataset.rowIndex = String(nextIndex - 1);
            list.appendChild(row);
            wireRow(row);

            if (parentIndex !== null) {
                applyParentIndex(row, parentIndex);
            }

            refreshControls();
        }

        function wireRow(row) {
            const removeButton = row.querySelector('[data-remove-company]');
            if (!removeButton) {
                return;
            }

            removeButton.addEventListener('click', () => {
                const idInput = row.querySelector('[data-company-id]');
                const deleteInput = row.querySelector('[data-delete-flag]');

                if (idInput && idInput.value !== '') {
                    if (deleteInput) {
                        deleteInput.value = '1';
                    }
                    row.classList.add('hidden');
                } else {
                    row.remove();
                }

                refreshControls();
            });

            const addChildButton = row.querySelector('[data-add-child]');
            if (addChildButton) {
                addChildButton.addEventListener('click', () => {
                    if (addChildButton.disabled || !template) {
                        return;
                    }
                    const parentIndex = row.dataset.rowIndex;
                    if (!parentIndex) {
                        return;
                    }
                    addRow({ parentIndex });
                });
            }

            const nameInput = row.querySelector('[data-company-name]');
            if (nameInput) {
                nameInput.addEventListener('input', () => {
                    const parentIndex = row.dataset.rowIndex;
                    if (!parentIndex) {
                        return;
                    }
                    list.querySelectorAll(`[data-parent-index="${parentIndex}"]`).forEach((childRow) => {
                        updateParentLabel(childRow, parentIndex);
                    });
                });
            }

            const parentIndexInput = row.querySelector('[data-parent-index]');
            if (parentIndexInput && parentIndexInput.value !== '') {
                applyParentIndex(row, parentIndexInput.value);
            }
        }

        list.querySelectorAll('[data-company-row]').forEach(wireRow);

        addButton.addEventListener('click', () => addRow());

        if (groupToggle) {
            groupToggle.addEventListener('change', refreshControls);
        }

        refreshControls();
    })();
</script>
