@php
    $fieldRows = old('fields', is_array($form->fields ?? null) ? $form->fields : []);
    if (!is_array($fieldRows) || $fieldRows === []) {
        $fieldRows = [
            ['label' => 'Name', 'name' => 'name', 'type' => 'text', 'required' => true, 'placeholder' => '', 'target' => 'name', 'options' => []],
        ];
    }

    $selectedBehavior = old('behavior', (string) ($form->behavior ?? 'create_task'));
    $behaviorSettings = old('behavior_settings', is_array($form->behavior_settings ?? null) ? $form->behavior_settings : []);
    $submitResultAction = (string) ($behaviorSettings['submit_result_action'] ?? 'reload');
    if (!in_array($submitResultAction, ['reload', 'hide_form', 'redirect'], true)) {
        $submitResultAction = 'reload';
    }
    $submitRedirectUrl = (string) ($behaviorSettings['redirect_url'] ?? '');
    $selectedPipelineId = (int) ($behaviorSettings['pipeline_id'] ?? 0);
    $selectedStageId = (int) ($behaviorSettings['stage_id'] ?? 0);
    $pipelineStageMap = $pipelines->mapWithKeys(fn ($pipeline) => [
        $pipeline->id => $pipeline->stages->map(fn ($stage) => ['id' => $stage->id, 'name' => $stage->name])->values(),
    ]);

    $fieldNameSuggestions = collect($fieldRows)
        ->pluck('name')
        ->filter(fn ($name) => is_string($name) && trim($name) !== '')
        ->map(fn ($name) => \Illuminate\Support\Str::snake(trim((string) $name)))
        ->unique()
        ->values()
        ->all();

    $storedFieldMap = is_array($behaviorSettings['field_map'] ?? null) ? $behaviorSettings['field_map'] : [];
    $oldFieldMap = old('behavior_settings.field_map');
    $fieldMapRowsByBehavior = [];

    foreach ($behaviorFieldMapOptions as $behaviorKey => $crmFields) {
        $rows = [];
        if (is_array($oldFieldMap) && isset($oldFieldMap[$behaviorKey]) && is_array($oldFieldMap[$behaviorKey])) {
            $rows = $oldFieldMap[$behaviorKey];
        } elseif ($behaviorKey === $selectedBehavior && is_array($oldFieldMap) && array_is_list($oldFieldMap)) {
            $rows = $oldFieldMap;
        } elseif ($behaviorKey === $selectedBehavior) {
            $rows = $storedFieldMap;
        }

        $rows = collect($rows)
            ->filter(fn ($row) => is_array($row))
            ->map(function (array $row): array {
                return [
                    'crm_field' => trim((string) ($row['crm_field'] ?? '')),
                    'form_field' => trim((string) ($row['form_field'] ?? '')),
                ];
            })
            ->filter(fn (array $row): bool => $row['crm_field'] !== '' || $row['form_field'] !== '')
            ->values()
            ->all();

        if ($rows === []) {
            $rows = [['crm_field' => '', 'form_field' => '']];
        }

        $fieldMapRowsByBehavior[$behaviorKey] = $rows;
    }

    $styleSettingsInput = old('style_settings', is_array($form->style_settings ?? null) ? $form->style_settings : []);
    $styleSettings = \App\Support\FormStyleManager::normalize(is_array($styleSettingsInput) ? $styleSettingsInput : []);
    $stylePalette = $styleSettings['palette'] ?? [];
    $styleLayout = $styleSettings['layout'] ?? [];
    $styleTypography = $styleSettings['typography'] ?? [];
    $styleCustomCss = (string) ($styleSettings['custom_css'] ?? '');
@endphp

<form method="POST" action="{{ $action }}" class="space-y-5">
    @csrf
    @if (($httpMethod ?? 'POST') !== 'POST')
        @method($httpMethod)
    @endif

    <datalist id="form-field-name-suggestions">
        @foreach ($fieldNameSuggestions as $fieldName)
            <option value="{{ $fieldName }}"></option>
        @endforeach
    </datalist>

    <section class="rounded-lg border border-slate-200 bg-white p-4 shadow-sm">
        <h3 class="text-sm font-semibold text-slate-900">Main settings</h3>

        <div class="mt-4 grid grid-cols-1 gap-4 md:grid-cols-2">
            <div>
                <label for="form-name" class="block text-sm font-medium text-slate-700">Name</label>
                <input id="form-name" type="text" name="name" value="{{ old('name', $form->name) }}" required class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                <x-input-error class="mt-2" :messages="$errors->get('name')" />
            </div>

            <div>
                <label for="form-slug" class="block text-sm font-medium text-slate-700">Link (slug)</label>
                <input id="form-slug" type="text" name="slug" value="{{ old('slug', $form->slug) }}" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" placeholder="generated-automatically-if-empty">
                <p class="mt-1 text-xs text-slate-500">Only lowercase Latin letters, numbers and dashes.</p>
                <x-input-error class="mt-2" :messages="$errors->get('slug')" />
            </div>
        </div>

        <div class="mt-4">
            <label for="form-description" class="block text-sm font-medium text-slate-700">Description</label>
            <textarea id="form-description" name="description" rows="2" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" placeholder="Optional text above the form">{{ old('description', $form->description) }}</textarea>
            <x-input-error class="mt-2" :messages="$errors->get('description')" />
        </div>

        <div class="mt-4 inline-flex items-center gap-2">
            <input type="hidden" name="is_active" value="0">
            <input id="form-active" type="checkbox" name="is_active" value="1" @checked(old('is_active', $form->is_active ?? true)) class="rounded border-slate-300 text-indigo-600 shadow-sm focus:ring-indigo-500">
            <label for="form-active" class="text-sm text-slate-700">Form is active</label>
        </div>
    </section>

    <section class="rounded-lg border border-slate-200 bg-white p-4 shadow-sm">
        <h3 class="text-sm font-semibold text-slate-900">Buttons</h3>

        <div class="mt-4 grid grid-cols-1 gap-4 md:grid-cols-3">
            <div>
                <label for="submit-label" class="block text-sm font-medium text-slate-700">Submit button text</label>
                <input id="submit-label" type="text" name="submit_label" value="{{ old('submit_label', $form->submit_label) }}" required class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                <x-input-error class="mt-2" :messages="$errors->get('submit_label')" />
            </div>

            <div class="md:col-span-2">
                <label for="success-message" class="block text-sm font-medium text-slate-700">Success message</label>
                <input id="success-message" type="text" name="success_message" value="{{ old('success_message', $form->success_message) }}" required class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                <x-input-error class="mt-2" :messages="$errors->get('success_message')" />
            </div>
        </div>

        <div class="mt-4 flex flex-wrap items-center gap-4">
            <label class="inline-flex items-center gap-2 text-sm text-slate-700">
                <input type="hidden" name="show_reset_button" value="0">
                <input type="checkbox" name="show_reset_button" value="1" @checked(old('show_reset_button', $form->show_reset_button ?? false)) class="rounded border-slate-300 text-indigo-600 shadow-sm focus:ring-indigo-500">
                Show reset button
            </label>

            <div>
                <label for="reset-label" class="block text-xs font-medium uppercase tracking-wide text-slate-500">Reset button text</label>
                <input id="reset-label" type="text" name="reset_label" value="{{ old('reset_label', $form->reset_label) }}" class="mt-1 block w-56 rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                <x-input-error class="mt-2" :messages="$errors->get('reset_label')" />
            </div>
        </div>
    </section>

    <section class="rounded-lg border border-slate-200 bg-white p-4 shadow-sm">
        <div class="flex flex-wrap items-center justify-between gap-3">
            <div>
                <h3 class="text-sm font-semibold text-slate-900">{{ __('Form styles') }}</h3>
                <p class="mt-1 text-xs text-slate-500">{{ __('Style settings apply to the public form link.') }}</p>
            </div>
        </div>

        <div class="mt-4 grid grid-cols-1 gap-4 md:grid-cols-3">
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Page background') }}</label>
                <input type="color" name="style_settings[palette][page_bg]" value="{{ $stylePalette['page_bg'] ?? '#F1F5F9' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Form card background') }}</label>
                <input type="color" name="style_settings[palette][card_bg]" value="{{ $stylePalette['card_bg'] ?? '#FFFFFF' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Border color') }}</label>
                <input type="color" name="style_settings[palette][border]" value="{{ $stylePalette['border'] ?? '#E2E8F0' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Text color') }}</label>
                <input type="color" name="style_settings[palette][text]" value="{{ $stylePalette['text'] ?? '#0F172A' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Muted text') }}</label>
                <input type="color" name="style_settings[palette][muted]" value="{{ $stylePalette['muted'] ?? '#64748B' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Input background') }}</label>
                <input type="color" name="style_settings[palette][input_bg]" value="{{ $stylePalette['input_bg'] ?? '#FFFFFF' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Input text') }}</label>
                <input type="color" name="style_settings[palette][input_text]" value="{{ $stylePalette['input_text'] ?? '#0F172A' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Input border') }}</label>
                <input type="color" name="style_settings[palette][input_border]" value="{{ $stylePalette['input_border'] ?? '#CBD5E1' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Primary button') }}</label>
                <input type="color" name="style_settings[palette][primary]" value="{{ $stylePalette['primary'] ?? '#4F46E5' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Button hover') }}</label>
                <input type="color" name="style_settings[palette][primary_hover]" value="{{ $stylePalette['primary_hover'] ?? '#6366F1' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Button text') }}</label>
                <input type="color" name="style_settings[palette][button_text]" value="{{ $stylePalette['button_text'] ?? '#FFFFFF' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Reset button background') }}</label>
                <input type="color" name="style_settings[palette][reset_bg]" value="{{ $stylePalette['reset_bg'] ?? '#FFFFFF' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Reset button text') }}</label>
                <input type="color" name="style_settings[palette][reset_text]" value="{{ $stylePalette['reset_text'] ?? '#0F172A' }}" class="mt-1 h-10 w-full rounded-md border border-slate-300 bg-white">
            </div>
        </div>

        <div class="mt-6 grid grid-cols-1 gap-4 md:grid-cols-3">
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Max width') }}</label>
                <select name="style_settings[layout][max_width]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    @foreach (\App\Support\FormStyleManager::widthOptions() as $width)
                        <option value="{{ $width }}" @selected((int) ($styleLayout['max_width'] ?? 768) === (int) $width)>
                            {{ $width }}px
                        </option>
                    @endforeach
                </select>
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Card radius') }}</label>
                <input type="number" min="0" max="32" name="style_settings[layout][card_radius]" value="{{ $styleLayout['card_radius'] ?? 16 }}" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Input radius') }}</label>
                <input type="number" min="0" max="24" name="style_settings[layout][input_radius]" value="{{ $styleLayout['input_radius'] ?? 8 }}" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Button radius') }}</label>
                <input type="number" min="0" max="32" name="style_settings[layout][button_radius]" value="{{ $styleLayout['button_radius'] ?? 10 }}" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            </div>
        </div>

        <div class="mt-6 grid grid-cols-1 gap-4 md:grid-cols-2">
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Font family') }}</label>
                <input type="text" name="style_settings[typography][font_family]" value="{{ $styleTypography['font_family'] ?? 'Figtree, ui-sans-serif, system-ui' }}" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            </div>
            <div>
                <label class="block text-sm font-medium text-slate-700">{{ __('Font stylesheet URL') }}</label>
                <input type="text" name="style_settings[typography][font_url]" value="{{ $styleTypography['font_url'] ?? '' }}" placeholder="https://fonts.bunny.net/css?family=inter:400,600&display=swap" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                <p class="mt-1 text-xs text-slate-500">{{ __('Optional. Provide a URL to a CSS file that loads the font.') }}</p>
            </div>
        </div>

        <div class="mt-6">
            <label class="block text-sm font-medium text-slate-700">{{ __('Custom CSS') }}</label>
            <textarea name="style_settings[custom_css]" rows="4" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" placeholder=".public-form-card { box-shadow: 0 12px 24px rgba(0,0,0,.08); }">{{ $styleCustomCss }}</textarea>
            <p class="mt-1 text-xs text-slate-500">{{ __('Optional. Add extra CSS rules for the public form.') }}</p>
        </div>
    </section>

    <section class="rounded-lg border border-slate-200 bg-white p-4 shadow-sm" data-behavior-scope>
        <h3 class="text-sm font-semibold text-slate-900">Behavior after submit</h3>

        <div class="mt-4 grid grid-cols-1 gap-4 md:grid-cols-2">
            <div>
                <label for="behavior" class="block text-sm font-medium text-slate-700">Action</label>
                <select id="behavior" name="behavior" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" data-behavior-select>
                    @foreach ($behaviorOptions as $value => $label)
                        <option value="{{ $value }}" @selected($selectedBehavior === $value)>{{ $label }}</option>
                    @endforeach
                </select>
                <x-input-error class="mt-2" :messages="$errors->get('behavior')" />
            </div>
        </div>

        <div class="mt-4 rounded-md border border-slate-200 bg-slate-50 p-3">
            <h4 class="text-xs font-semibold uppercase tracking-wide text-slate-500">{{ __('Result for visitor') }}</h4>
            <div class="mt-3 grid grid-cols-1 gap-3">
                <div>
                    <label for="submit-result-action" class="block text-sm font-medium text-slate-700">{{ __('After successful submit') }}</label>
                    <select id="submit-result-action" name="behavior_settings[submit_result_action]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" data-submit-result-action-select>
                        <option value="reload" @selected($submitResultAction === 'reload')>{{ __('Reload page and show message') }}</option>
                        <option value="hide_form" @selected($submitResultAction === 'hide_form')>{{ __('Show message and hide form (single submit)') }}</option>
                        <option value="redirect" @selected($submitResultAction === 'redirect')>{{ __('Redirect to URL') }}</option>
                    </select>
                    <x-input-error class="mt-2" :messages="$errors->get('behavior_settings.submit_result_action')" />
                </div>

                <div data-redirect-url-block @class(['hidden' => $submitResultAction !== 'redirect'])>
                    <label for="submit-redirect-url" class="block text-sm font-medium text-slate-700">{{ __('Redirect URL') }}</label>
                    <input
                        id="submit-redirect-url"
                        type="text"
                        name="behavior_settings[redirect_url]"
                        value="{{ old('behavior_settings.redirect_url', $submitRedirectUrl) }}"
                        class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                        placeholder="/thank-you or https://example.com/thanks"
                    >
                    <p class="mt-1 text-xs text-slate-500">{{ __('Allowed formats: relative path (/thanks) or full http(s) URL.') }}</p>
                    <x-input-error class="mt-2" :messages="$errors->get('behavior_settings.redirect_url')" />
                </div>
            </div>
        </div>

        <div class="mt-4 space-y-4">
            <div data-behavior-settings="create_task" class="rounded-md border border-slate-200 bg-slate-50 p-3">
                <h4 class="text-xs font-semibold uppercase tracking-wide text-slate-500">Task settings</h4>
                <div class="mt-3 grid grid-cols-1 gap-3 md:grid-cols-2">
                    <div>
                        <label class="block text-sm font-medium text-slate-700">Assignee</label>
                        <select name="behavior_settings[assignee_id]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            <option value="">Auto (form owner)</option>
                            @foreach ($users as $user)
                                <option value="{{ $user->id }}" @selected((int) ($behaviorSettings['assignee_id'] ?? 0) === $user->id)>{{ $user->name }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-slate-700">Default priority</label>
                        <select name="behavior_settings[priority]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            @foreach (['low' => 'Low', 'medium' => 'Medium', 'high' => 'High', 'urgent' => 'Urgent'] as $value => $label)
                                <option value="{{ $value }}" @selected(($behaviorSettings['priority'] ?? 'medium') === $value)>{{ $label }}</option>
                            @endforeach
                        </select>
                    </div>
                </div>
            </div>

            <div data-behavior-settings="create_deal" class="rounded-md border border-slate-200 bg-slate-50 p-3">
                <h4 class="text-xs font-semibold uppercase tracking-wide text-slate-500">Lead/deal settings</h4>
                <div class="mt-3 grid grid-cols-1 gap-3 md:grid-cols-2">
                    <div>
                        <label class="block text-sm font-medium text-slate-700">Owner</label>
                        <select name="behavior_settings[owner_id]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            <option value="">Auto (form owner)</option>
                            @foreach ($users as $user)
                                <option value="{{ $user->id }}" @selected((int) ($behaviorSettings['owner_id'] ?? 0) === $user->id)>{{ $user->name }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-slate-700">Currency</label>
                        <input type="text" name="behavior_settings[currency]" value="{{ old('behavior_settings.currency', $behaviorSettings['currency'] ?? 'USD') }}" maxlength="3" class="mt-1 block w-full rounded-md border-slate-300 text-sm uppercase shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-slate-700">Funnel</label>
                        <select name="behavior_settings[pipeline_id]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" data-pipeline-select>
                            <option value="">Auto (default funnel)</option>
                            @foreach ($pipelines as $pipeline)
                                <option value="{{ $pipeline->id }}" @selected($selectedPipelineId === $pipeline->id)>{{ $pipeline->name }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-slate-700">Stage</label>
                        <select name="behavior_settings[stage_id]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" data-stage-select>
                            <option value="">Auto (first stage)</option>
                            @foreach (($pipelines->firstWhere('id', $selectedPipelineId)?->stages ?? collect()) as $stage)
                                <option value="{{ $stage->id }}" @selected($selectedStageId === $stage->id)>{{ $stage->name }}</option>
                            @endforeach
                        </select>
                    </div>
                </div>
            </div>

            <div data-behavior-settings="create_company" class="rounded-md border border-slate-200 bg-slate-50 p-3">
                <h4 class="text-xs font-semibold uppercase tracking-wide text-slate-500">Company settings</h4>
                <div class="mt-3 grid grid-cols-1 gap-3 md:grid-cols-2">
                    <div>
                        <label class="block text-sm font-medium text-slate-700">Owner</label>
                        <select name="behavior_settings[owner_id]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            <option value="">Auto (form owner)</option>
                            @foreach ($users as $user)
                                <option value="{{ $user->id }}" @selected((int) ($behaviorSettings['owner_id'] ?? 0) === $user->id)>{{ $user->name }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-slate-700">Default status</label>
                        <select name="behavior_settings[status]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            @foreach (['lead' => 'Lead', 'client' => 'Client', 'partner' => 'Partner', 'inactive' => 'Inactive'] as $value => $label)
                                <option value="{{ $value }}" @selected(($behaviorSettings['status'] ?? 'lead') === $value)>{{ $label }}</option>
                            @endforeach
                        </select>
                    </div>
                </div>
            </div>

            <div data-behavior-settings="create_contact" class="rounded-md border border-slate-200 bg-slate-50 p-3">
                <h4 class="text-xs font-semibold uppercase tracking-wide text-slate-500">Contact settings</h4>
                <div class="mt-3 grid grid-cols-1 gap-3 md:grid-cols-2">
                    <div>
                        <label class="block text-sm font-medium text-slate-700">Owner</label>
                        <select name="behavior_settings[owner_id]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            <option value="">Auto (form owner)</option>
                            @foreach ($users as $user)
                                <option value="{{ $user->id }}" @selected((int) ($behaviorSettings['owner_id'] ?? 0) === $user->id)>{{ $user->name }}</option>
                            @endforeach
                        </select>
                    </div>
                </div>
            </div>

            @foreach ($behaviorFieldMapOptions as $behaviorKey => $crmFieldOptions)
                @php
                    $mapRows = $fieldMapRowsByBehavior[$behaviorKey] ?? [['crm_field' => '', 'form_field' => '']];
                @endphp
                <div data-behavior-settings="{{ $behaviorKey }}" data-field-map-editor data-field-map-behavior="{{ $behaviorKey }}" class="rounded-md border border-slate-200 bg-white p-3">
                    <div class="flex flex-wrap items-center justify-between gap-3">
                        <div>
                            <h4 class="text-xs font-semibold uppercase tracking-wide text-slate-500">Field mapping to CRM</h4>
                            <p class="mt-1 text-xs text-slate-500">Connect a form field code with a CRM field.</p>
                        </div>
                        <button type="button" class="inline-flex items-center rounded-md border border-slate-300 px-2.5 py-1 text-xs font-medium text-slate-700 hover:bg-slate-50" data-add-field-map>
                            Add mapping
                        </button>
                    </div>

                    <div class="mt-3 space-y-2" data-field-map-list>
                        @foreach ($mapRows as $mapIndex => $mapRow)
                            <div class="grid grid-cols-1 gap-2 md:grid-cols-[minmax(0,1fr)_minmax(0,1fr)_auto]" data-field-map-row>
                                <div>
                                    <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">CRM field</label>
                                    <select
                                        name="behavior_settings[field_map][{{ $behaviorKey }}][{{ $mapIndex }}][crm_field]"
                                        class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                                    >
                                        <option value="">Not selected</option>
                                        @foreach ($crmFieldOptions as $crmField => $crmLabel)
                                            <option value="{{ $crmField }}" @selected(($mapRow['crm_field'] ?? '') === $crmField)>{{ $crmLabel }}</option>
                                        @endforeach
                                    </select>
                                </div>

                                <div>
                                    <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Form field code</label>
                                    <input
                                        type="text"
                                        name="behavior_settings[field_map][{{ $behaviorKey }}][{{ $mapIndex }}][form_field]"
                                        value="{{ $mapRow['form_field'] ?? '' }}"
                                        list="form-field-name-suggestions"
                                        placeholder="lead_name"
                                        class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                                    >
                                </div>

                                <div class="flex items-end">
                                    <button type="button" class="inline-flex h-9 w-9 items-center justify-center rounded-md border border-rose-300 text-rose-700 hover:bg-rose-50" data-remove-field-map title="Delete mapping">
                                        <x-menu-icon name="fa-solid fa-trash-can" class="h-3.5 w-3.5" />
                                    </button>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
            @endforeach
        </div>

        <x-input-error class="mt-2" :messages="$errors->get('behavior_settings.pipeline_id')" />
        <x-input-error class="mt-2" :messages="$errors->get('behavior_settings.stage_id')" />
        <x-input-error class="mt-2" :messages="$errors->get('behavior_settings.field_map')" />
        <x-input-error class="mt-2" :messages="$errors->get('behavior_settings.field_map.*.*.crm_field')" />
        <x-input-error class="mt-2" :messages="$errors->get('behavior_settings.field_map.*.*.form_field')" />
    </section>

    <section class="rounded-lg border border-slate-200 bg-white p-4 shadow-sm" data-fields-scope>
        <div class="flex items-center justify-between gap-3">
            <h3 class="text-sm font-semibold text-slate-900">Form fields</h3>
            <button type="button" class="inline-flex items-center rounded-md border border-slate-300 px-3 py-1.5 text-xs font-medium text-slate-700 hover:bg-slate-50" data-add-field>
                Add field
            </button>
        </div>

        <p class="mt-1 text-xs text-slate-500">Map fields to CRM attributes using "Target". Unmapped fields are still saved in submission history.</p>

        <div class="mt-3 space-y-3" data-fields-list>
            @foreach ($fieldRows as $index => $field)
                @php
                    $fieldOptions = $field['options'] ?? [];
                    if (is_array($fieldOptions)) {
                        $fieldOptions = implode(PHP_EOL, $fieldOptions);
                    }
                @endphp
                <div class="rounded-md border border-slate-200 p-3" data-field-row>
                    <div class="grid grid-cols-1 gap-3 md:grid-cols-6">
                        <div>
                            <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Label</label>
                            <input type="text" name="fields[{{ $index }}][label]" value="{{ $field['label'] ?? '' }}" required class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div>
                            <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Name</label>
                            <input type="text" name="fields[{{ $index }}][name]" value="{{ $field['name'] ?? '' }}" required class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" placeholder="lead_name">
                        </div>
                        <div>
                            <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Type</label>
                            <select name="fields[{{ $index }}][type]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                @foreach ($fieldTypeOptions as $value => $label)
                                    <option value="{{ $value }}" @selected(($field['type'] ?? 'text') === $value)>{{ $label }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div>
                            <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Target</label>
                            <select name="fields[{{ $index }}][target]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                                @foreach ($targetOptions as $value => $label)
                                    <option value="{{ $value }}" @selected(($field['target'] ?? '') === $value)>{{ $label }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div>
                            <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Placeholder</label>
                            <input type="text" name="fields[{{ $index }}][placeholder]" value="{{ $field['placeholder'] ?? '' }}" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                        </div>
                        <div class="flex items-end justify-between gap-2">
                            <label class="inline-flex items-center gap-2 text-sm text-slate-700">
                                <input type="hidden" name="fields[{{ $index }}][required]" value="0">
                                <input type="checkbox" name="fields[{{ $index }}][required]" value="1" @checked((bool) ($field['required'] ?? false)) class="rounded border-slate-300 text-indigo-600 shadow-sm focus:ring-indigo-500">
                                Required
                            </label>
                            <button type="button" class="inline-flex h-8 w-8 items-center justify-center rounded-md border border-rose-300 text-rose-700 hover:bg-rose-50" data-remove-field title="Delete field">
                                <x-menu-icon name="fa-solid fa-trash-can" class="h-3.5 w-3.5" />
                            </button>
                        </div>
                    </div>

                    <div class="mt-3">
                        <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">List options (one value per line)</label>
                        <textarea name="fields[{{ $index }}][options]" rows="2" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" placeholder="Option 1&#10;Option 2">{{ $fieldOptions }}</textarea>
                    </div>
                </div>
            @endforeach
        </div>

        <x-input-error class="mt-2" :messages="$errors->get('fields')" />
        <x-input-error class="mt-2" :messages="$errors->get('fields.*.label')" />
        <x-input-error class="mt-2" :messages="$errors->get('fields.*.name')" />
        <x-input-error class="mt-2" :messages="$errors->get('fields.*.type')" />
    </section>

    <div class="flex items-center gap-3">
        <x-primary-button>Save form</x-primary-button>
        <a href="{{ route('forms.index') }}" class="text-sm text-slate-600 hover:text-slate-800">Cancel</a>
    </div>
</form>

<template id="form-field-template">
    <div class="rounded-md border border-slate-200 p-3" data-field-row>
        <div class="grid grid-cols-1 gap-3 md:grid-cols-6">
            <div>
                <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Label</label>
                <input type="text" name="fields[__INDEX__][label]" required class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            </div>
            <div>
                <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Name</label>
                <input type="text" name="fields[__INDEX__][name]" required class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" placeholder="new_field">
            </div>
            <div>
                <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Type</label>
                <select name="fields[__INDEX__][type]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    @foreach ($fieldTypeOptions as $value => $label)
                        <option value="{{ $value }}">{{ $label }}</option>
                    @endforeach
                </select>
            </div>
            <div>
                <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Target</label>
                <select name="fields[__INDEX__][target]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    @foreach ($targetOptions as $value => $label)
                        <option value="{{ $value }}">{{ $label }}</option>
                    @endforeach
                </select>
            </div>
            <div>
                <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Placeholder</label>
                <input type="text" name="fields[__INDEX__][placeholder]" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
            </div>
            <div class="flex items-end justify-between gap-2">
                <label class="inline-flex items-center gap-2 text-sm text-slate-700">
                    <input type="hidden" name="fields[__INDEX__][required]" value="0">
                    <input type="checkbox" name="fields[__INDEX__][required]" value="1" class="rounded border-slate-300 text-indigo-600 shadow-sm focus:ring-indigo-500">
                    Required
                </label>
                <button type="button" class="inline-flex h-8 w-8 items-center justify-center rounded-md border border-rose-300 text-rose-700 hover:bg-rose-50" data-remove-field title="Delete field">
                    <x-menu-icon name="fa-solid fa-trash-can" class="h-3.5 w-3.5" />
                </button>
            </div>
        </div>

        <div class="mt-3">
            <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">List options (one value per line)</label>
            <textarea name="fields[__INDEX__][options]" rows="2" class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500" placeholder="Option 1&#10;Option 2"></textarea>
        </div>
    </div>
</template>

@foreach ($behaviorFieldMapOptions as $behaviorKey => $crmFieldOptions)
    <template id="form-field-map-template-{{ $behaviorKey }}">
        <div class="grid grid-cols-1 gap-2 md:grid-cols-[minmax(0,1fr)_minmax(0,1fr)_auto]" data-field-map-row>
            <div>
                <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">CRM field</label>
                <select
                    name="behavior_settings[field_map][{{ $behaviorKey }}][__INDEX__][crm_field]"
                    class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                >
                    <option value="">Not selected</option>
                    @foreach ($crmFieldOptions as $crmField => $crmLabel)
                        <option value="{{ $crmField }}">{{ $crmLabel }}</option>
                    @endforeach
                </select>
            </div>

            <div>
                <label class="block text-xs font-semibold uppercase tracking-wide text-slate-500">Form field code</label>
                <input
                    type="text"
                    name="behavior_settings[field_map][{{ $behaviorKey }}][__INDEX__][form_field]"
                    list="form-field-name-suggestions"
                    placeholder="lead_name"
                    class="mt-1 block w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                >
            </div>

            <div class="flex items-end">
                <button type="button" class="inline-flex h-9 w-9 items-center justify-center rounded-md border border-rose-300 text-rose-700 hover:bg-rose-50" data-remove-field-map title="Delete mapping">
                    <x-menu-icon name="fa-solid fa-trash-can" class="h-3.5 w-3.5" />
                </button>
            </div>
        </div>
    </template>
@endforeach

<script>
    document.addEventListener('DOMContentLoaded', () => {
        const fieldsScope = document.querySelector('[data-fields-scope]');
        if (fieldsScope) {
            const list = fieldsScope.querySelector('[data-fields-list]');
            const addButton = fieldsScope.querySelector('[data-add-field]');
            const template = document.getElementById('form-field-template');
            let fieldIndex = list ? list.querySelectorAll('[data-field-row]').length : 0;

            const bindRemove = (row) => {
                const removeButton = row.querySelector('[data-remove-field]');
                if (!removeButton) {
                    return;
                }

                removeButton.addEventListener('click', () => {
                    const rows = list.querySelectorAll('[data-field-row]');
                    if (rows.length <= 1) {
                        return;
                    }
                    row.remove();
                });
            };

            list?.querySelectorAll('[data-field-row]').forEach((row) => bindRemove(row));

            addButton?.addEventListener('click', () => {
                if (!list || !template) {
                    return;
                }

                const html = template.innerHTML.replaceAll('__INDEX__', String(fieldIndex));
                fieldIndex += 1;

                const wrapper = document.createElement('div');
                wrapper.innerHTML = html.trim();
                const row = wrapper.firstElementChild;
                if (!row) {
                    return;
                }

                list.appendChild(row);
                bindRemove(row);
            });
        }

        document.querySelectorAll('[data-field-map-editor]').forEach((scope) => {
            const behaviorKey = scope.getAttribute('data-field-map-behavior') || '';
            const list = scope.querySelector('[data-field-map-list]');
            const addButton = scope.querySelector('[data-add-field-map]');
            const template = document.getElementById(`form-field-map-template-${behaviorKey}`);
            let mapIndex = list ? list.querySelectorAll('[data-field-map-row]').length : 0;

            const bindRemoveMapRow = (row) => {
                const removeButton = row.querySelector('[data-remove-field-map]');
                if (!removeButton || !list) {
                    return;
                }

                removeButton.addEventListener('click', () => {
                    const rows = list.querySelectorAll('[data-field-map-row]');
                    if (rows.length <= 1) {
                        const crmFieldSelect = row.querySelector('select');
                        const formFieldInput = row.querySelector('input');
                        if (crmFieldSelect) {
                            crmFieldSelect.value = '';
                        }
                        if (formFieldInput) {
                            formFieldInput.value = '';
                        }
                        return;
                    }

                    row.remove();
                });
            };

            list?.querySelectorAll('[data-field-map-row]').forEach((row) => bindRemoveMapRow(row));

            addButton?.addEventListener('click', () => {
                if (!list || !template) {
                    return;
                }

                const html = template.innerHTML.replaceAll('__INDEX__', String(mapIndex));
                mapIndex += 1;

                const wrapper = document.createElement('div');
                wrapper.innerHTML = html.trim();
                const row = wrapper.firstElementChild;
                if (!row) {
                    return;
                }

                list.appendChild(row);
                bindRemoveMapRow(row);
            });
        });

        const behaviorScope = document.querySelector('[data-behavior-scope]');
        if (behaviorScope) {
            const behaviorSelect = behaviorScope.querySelector('[data-behavior-select]');
            const settingsBlocks = behaviorScope.querySelectorAll('[data-behavior-settings]');
            const pipelineSelect = behaviorScope.querySelector('[data-pipeline-select]');
            const stageSelect = behaviorScope.querySelector('[data-stage-select]');
            const submitResultActionSelect = behaviorScope.querySelector('[data-submit-result-action-select]');
            const redirectUrlBlock = behaviorScope.querySelector('[data-redirect-url-block]');
            const stageMap = @json($pipelineStageMap);
            const selectedStageId = @json($selectedStageId);

            const updateBehaviorBlocks = () => {
                if (!behaviorSelect) {
                    return;
                }

                const behavior = behaviorSelect.value;
                settingsBlocks.forEach((block) => {
                    block.classList.toggle('hidden', block.dataset.behaviorSettings !== behavior);
                });
            };

            const updateStageOptions = (preferSelected = false) => {
                if (!pipelineSelect || !stageSelect) {
                    return;
                }

                const pipelineId = pipelineSelect.value;
                const stages = stageMap[pipelineId] || [];
                const preferred = preferSelected ? String(selectedStageId || '') : '';

                stageSelect.innerHTML = '';

                const autoOption = document.createElement('option');
                autoOption.value = '';
                autoOption.textContent = 'Auto (first stage)';
                stageSelect.appendChild(autoOption);

                stages.forEach((stage) => {
                    const option = document.createElement('option');
                    option.value = String(stage.id || '');
                    option.textContent = String(stage.name || '');
                    if (preferred !== '' && option.value === preferred) {
                        option.selected = true;
                    }
                    stageSelect.appendChild(option);
                });
            };

            const updateSubmitResultSettings = () => {
                if (!submitResultActionSelect || !redirectUrlBlock) {
                    return;
                }

                const showRedirectUrl = submitResultActionSelect.value === 'redirect';
                redirectUrlBlock.classList.toggle('hidden', !showRedirectUrl);

                const redirectInput = redirectUrlBlock.querySelector('input[name="behavior_settings[redirect_url]"]');
                if (redirectInput) {
                    redirectInput.required = showRedirectUrl;
                }
            };

            behaviorSelect?.addEventListener('change', updateBehaviorBlocks);
            pipelineSelect?.addEventListener('change', () => updateStageOptions(false));
            submitResultActionSelect?.addEventListener('change', updateSubmitResultSettings);

            updateBehaviorBlocks();
            updateStageOptions(true);
            updateSubmitResultSettings();
        }
    });
</script>
