<?php

namespace App\Support;

use App\Models\OrganizationCompany;
use App\Models\User;
use Illuminate\Contracts\Cache\Repository as CacheRepository;
use Illuminate\Support\Facades\Cache;

class BrandingManager
{
    private const CACHE_VERSION_KEY = 'branding-manager.version';

    private const CACHE_TTL_SECONDS = 900;

    /**
     * @return array{name:string,logo_url:string}
     */
    public function sidebarBrand(User $user): array
    {
        $cacheKey = sprintf(
            'branding-manager.user.%d.company.%d.v%d',
            $user->id,
            (int) ($user->organization_company_id ?? 0),
            $this->cacheVersion()
        );

        /** @var array{name:string,logo_url:string} $brand */
        $brand = $this->hotCache()->remember(
            $cacheKey,
            now()->addSeconds(self::CACHE_TTL_SECONDS),
            function () use ($user): array {
                return $this->resolveBrandPayload($user);
            }
        );

        return $brand;
    }

    public function flush(): void
    {
        $cache = $this->hotCache();
        if ($cache->has(self::CACHE_VERSION_KEY)) {
            $cache->increment(self::CACHE_VERSION_KEY);

            return;
        }

        $cache->forever(self::CACHE_VERSION_KEY, 2);
    }

    private function cacheVersion(): int
    {
        return (int) $this->hotCache()->rememberForever(
            self::CACHE_VERSION_KEY,
            static fn (): int => 1
        );
    }

    /**
     * @return array{name:string,logo_url:string}
     */
    private function resolveBrandPayload(User $user): array
    {
        $brandName = (string) config('app.name', 'CRM25');
        $brandLogoUrl = '/logo_crm25.png';

        $preferredBrandCompany = null;
        if ((int) ($user->organization_company_id ?? 0) > 0) {
            $preferredBrandCompany = OrganizationCompany::query()
                ->whereKey((int) $user->organization_company_id)
                ->first(['id', 'name', 'logo_path']);
        }

        $brandCompany = $preferredBrandCompany;
        $hasBranding = $brandCompany
            && trim((string) $brandCompany->name) !== ''
            && trim((string) $brandCompany->logo_path) !== '';

        if (! $hasBranding) {
            $brandCompany = OrganizationCompany::query()
                ->whereNotNull('name')
                ->where('name', '!=', '')
                ->whereNotNull('logo_path')
                ->where('logo_path', '!=', '')
                ->orderBy('sort_order')
                ->orderBy('id')
                ->first(['id', 'name', 'logo_path']);

            $hasBranding = $brandCompany
                && trim((string) $brandCompany->name) !== ''
                && trim((string) $brandCompany->logo_path) !== '';
        }

        if ($hasBranding) {
            $brandName = trim((string) $brandCompany->name);
            $brandLogoUrl = $this->publicStorageUrl((string) $brandCompany->logo_path);
        }

        return [
            'name' => $brandName,
            'logo_url' => $brandLogoUrl,
        ];
    }

    private function publicStorageUrl(string $path): string
    {
        $normalizedPath = trim($path, '/');
        if ($normalizedPath === '') {
            return '/logo_crm25.png';
        }

        return '/storage/'.$normalizedPath;
    }

    private function hotCache(): CacheRepository
    {
        /** @var CacheRepository $store */
        $store = Cache::store((string) config('cache.hot_store', 'hot'));

        return $store;
    }
}
