<?php

namespace App\Http\Controllers;

use App\Models\Deal;
use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Illuminate\View\View;

class ReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('can:viewReports');
    }

    public function index(): View
    {
        $now = now();

        $crm = [
            'deals_total' => Deal::count(),
            'deals_open' => Deal::where('status', 'open')->count(),
            'deals_won' => Deal::where('status', 'won')->count(),
            'deals_lost' => Deal::where('status', 'lost')->count(),
            'pipeline_value' => Deal::where('status', 'open')->sum('amount'),
            'won_month_value' => Deal::query()
                ->where('status', 'won')
                ->whereBetween('closed_at', [$now->copy()->startOfMonth(), $now->copy()->endOfMonth()])
                ->sum('amount'),
        ];

        $projects = [
            'total' => Project::count(),
            'active' => Project::where('status', 'active')->count(),
            'on_hold' => Project::where('status', 'on_hold')->count(),
            'completed' => Project::where('status', 'completed')->count(),
            'avg_progress' => (int) round((float) Project::avg('progress')),
            'budget' => Project::sum('budget'),
            'spent' => Project::sum('spent'),
        ];

        $tasks = [
            'total' => Task::count(),
            'open' => Task::whereIn('status', ['todo', 'in_progress', 'review'])->count(),
            'done' => Task::where('status', 'done')->count(),
            'overdue' => Task::query()
                ->whereIn('status', ['todo', 'in_progress', 'review'])
                ->whereNotNull('due_at')
                ->where('due_at', '<', $now)
                ->count(),
        ];

        $tasks['completion_rate'] = $tasks['total'] > 0
            ? (int) round(($tasks['done'] / $tasks['total']) * 100)
            : 0;

        $teamPerformance = User::query()
            ->withCount([
                'assignedTasks as open_tasks_count' => fn ($query) => $query->whereIn('status', ['todo', 'in_progress', 'review']),
                'assignedTasks as done_tasks_count' => fn ($query) => $query->where('status', 'done'),
                'deals as open_deals_count' => fn ($query) => $query->where('status', 'open'),
                'managedProjects as active_projects_count' => fn ($query) => $query->where('status', 'active'),
            ])
            ->orderByDesc('open_deals_count')
            ->orderByDesc('active_projects_count')
            ->orderBy('name')
            ->limit(12)
            ->get();

        $trend = collect(range(5, 0))->map(function (int $offset) {
            $start = now()->copy()->subMonths($offset)->startOfMonth();
            $end = $start->copy()->endOfMonth();

            return [
                'label' => $start->format('M Y'),
                'new_deals' => Deal::whereBetween('created_at', [$start, $end])->count(),
                'won_deals' => Deal::where('status', 'won')->whereBetween('closed_at', [$start, $end])->count(),
                'new_projects' => Project::whereBetween('created_at', [$start, $end])->count(),
                'completed_projects' => Project::where('status', 'completed')->whereBetween('completed_at', [$start, $end])->count(),
            ];
        });

        return view('reports.index', compact('crm', 'projects', 'tasks', 'teamPerformance', 'trend'));
    }
}
