<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\UpdateCheckResource;
use App\Http\Resources\UpdateInstallationResource;
use App\Http\Resources\UpdateSettingResource;
use App\Support\AccessControl;
use App\Support\UpdateInstaller;
use App\Support\UpdateCenterManager;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class UpdateCenterController extends Controller
{
    public function show(Request $request, UpdateCenterManager $updateCenterManager)
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'updates', 'read'), 403);
        if (! $updateCenterManager->storageIsReady()) {
            return $this->storageUnavailableResponse();
        }

        return UpdateSettingResource::make($updateCenterManager->settings());
    }

    public function update(Request $request, UpdateCenterManager $updateCenterManager)
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'updates', 'update'), 403);
        if (! $updateCenterManager->storageIsReady()) {
            return $this->storageUnavailableResponse();
        }

        $validated = $request->validate([
            'feed_url' => ['required', 'string', 'max:255'],
            'current_version' => ['nullable', 'string', 'max:64'],
            'channel' => ['required', Rule::in(['stable', 'beta', 'nightly'])],
            'is_active' => ['nullable', 'boolean'],
            'auto_check_enabled' => ['nullable', 'boolean'],
            'check_interval_minutes' => ['required', 'integer', 'min:5', 'max:10080'],
            'request_timeout_seconds' => ['required', 'integer', 'min:2', 'max:60'],
            'verify_tls' => ['nullable', 'boolean'],
            'meta' => ['nullable', 'array'],
        ]);

        $settings = $updateCenterManager->settings();
        $currentVersion = trim((string) ($validated['current_version'] ?? ''));

        $settings->fill([
            'feed_url' => $updateCenterManager->normalizeFeedUrl((string) $validated['feed_url']),
            'current_version' => $currentVersion !== '' ? $currentVersion : $updateCenterManager->defaultCurrentVersion(),
            'channel' => (string) $validated['channel'],
            'is_active' => $request->boolean('is_active', true),
            'auto_check_enabled' => $request->boolean('auto_check_enabled', true),
            'check_interval_minutes' => (int) $validated['check_interval_minutes'],
            'request_timeout_seconds' => (int) $validated['request_timeout_seconds'],
            'verify_tls' => $request->boolean('verify_tls', true),
            'meta' => is_array($validated['meta'] ?? null) ? $validated['meta'] : null,
        ]);
        $settings->save();

        return UpdateSettingResource::make($settings);
    }

    public function checks(Request $request, UpdateCenterManager $updateCenterManager)
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'updates', 'read'), 403);
        if (! $updateCenterManager->storageIsReady()) {
            return $this->storageUnavailableResponse();
        }

        $settings = $updateCenterManager->settings();
        $checks = $settings->checks()
            ->limit(100)
            ->get();

        return UpdateCheckResource::collection($checks);
    }

    public function installations(Request $request, UpdateCenterManager $updateCenterManager)
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'updates', 'read'), 403);
        if (! $updateCenterManager->storageIsReady()) {
            return $this->storageUnavailableResponse();
        }

        $settings = $updateCenterManager->settings();
        $installations = $settings->installations()
            ->with('actor:id,name,email')
            ->limit(100)
            ->get();

        return UpdateInstallationResource::collection($installations);
    }

    public function runCheck(Request $request, UpdateCenterManager $updateCenterManager)
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'updates', 'create'), 403);
        if (! $updateCenterManager->storageIsReady()) {
            return $this->storageUnavailableResponse();
        }

        $check = $updateCenterManager->checkNow();

        return (new UpdateCheckResource($check))
            ->response()
            ->setStatusCode(201);
    }

    public function install(Request $request, UpdateCenterManager $updateCenterManager, UpdateInstaller $updateInstaller)
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'updates', 'create'), 403);
        if (! $updateCenterManager->storageIsReady()) {
            return $this->storageUnavailableResponse();
        }

        $installation = $updateInstaller->installLatest($user, $updateCenterManager->settings());

        return (new UpdateInstallationResource($installation))
            ->response()
            ->setStatusCode(201);
    }

    private function storageUnavailableResponse(): JsonResponse
    {
        return response()->json([
            'message' => 'Update tables are not migrated. Run "php artisan migrate".',
            'code' => 'updates_storage_missing',
        ], 503);
    }
}
