@php
    $baseQuery = request()->except(['view', 'page']);
    $projectStatusLabels = [
        'planned' => __('Planned'),
        'active' => __('Active'),
        'on_hold' => __('On hold'),
        'completed' => __('Completed'),
        'cancelled' => __('Cancelled'),
    ];
@endphp

<x-app-layout>
    <x-slot name="header">
        <div class="flex items-center justify-between gap-4">
            <h2 class="font-semibold text-xl text-gray-800 leading-tight">Projects</h2>
            <a href="{{ route('projects.create') }}" class="inline-flex items-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white hover:bg-indigo-500">New project</a>
        </div>
    </x-slot>

    <div class="pb-12">
        <div class="w-full px-6 space-y-4">
            <section class="bg-white border border-gray-200 rounded-xl p-4">
                <div class="mb-4 flex flex-wrap gap-2">
                    @foreach ($viewOptions as $mode => $label)
                        <a
                            href="{{ route('projects.index', array_merge($baseQuery, ['view' => $mode])) }}"
                            class="inline-flex items-center rounded-md px-3 py-1.5 text-sm font-medium {{ $viewMode === $mode ? 'bg-indigo-600 text-white' : 'bg-gray-100 text-gray-700 hover:bg-gray-200' }}"
                        >
                            {{ $label }}
                        </a>
                    @endforeach
                </div>

                <form method="GET" action="{{ route('projects.index') }}" class="grid grid-cols-1 gap-3 md:grid-cols-4">
                    <input type="hidden" name="view" value="{{ $viewMode }}">

                    <div>
                        <label for="q" class="block text-xs font-medium text-gray-500 uppercase">Search</label>
                        <input id="q" type="text" name="q" value="{{ $search }}" placeholder="Name, code, company, deal" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    </div>

                    <div>
                        <label for="status" class="block text-xs font-medium text-gray-500 uppercase">Status</label>
                        <select id="status" name="status" class="mt-1 w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            <option value="">All</option>
                            @foreach (['planned' => 'Planned', 'active' => 'In work', 'on_hold' => 'Suspended', 'completed' => 'Completed', 'cancelled' => 'Canceled'] as $value => $label)
                                <option value="{{ $value }}" @selected($status === $value)>{{ __($label) }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div>
                        <label for="manager_id" class="block text-xs font-medium text-gray-500 uppercase">Supervisor</label>
                        <x-user-search-select
                            id="manager_id"
                            name="manager_id"
                            :users="$managers"
                            :selected="$managerId"
                            placeholder="All users"
                            empty-label="All"
                        />
                    </div>

                    <div class="flex items-end gap-2">
                        <button type="submit" class="inline-flex items-center rounded-md bg-gray-800 px-4 py-2 text-sm font-medium text-white hover:bg-gray-700">Apply</button>
                        <a href="{{ route('projects.index', ['view' => $viewMode]) }}" class="inline-flex items-center rounded-md border border-gray-300 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">Reset</a>
                    </div>
                </form>
            </section>

            @if ($viewMode === 'kanban')
                <section
                    id="project-board"
                    data-csrf="{{ csrf_token() }}"
                    class="space-y-3"
                >
                    <div>
                        <h3 class="text-sm font-semibold uppercase tracking-wide text-gray-700">Project Kanban</h3>
                        <p class="mt-1 text-xs text-gray-500">Move projects between statuses. Changes are saved without page reload.</p>
                    </div>

                    <div class="relative" data-project-board-scroll-wrap>
                        <div class="overflow-x-auto pb-2" data-project-board-scroll>
                            <div class="flex min-w-max items-start gap-4" data-project-board-grid>
                                @foreach ($boardColumns as $column)
                                    <div
                                        class="w-[340px] shrink-0 self-start rounded-xl border border-gray-200 border-t-4 bg-white p-3"
                                        data-status-column="{{ $column['status'] }}"
                                        style="{{ $column['surface_style'] }}"
                                    >
                                        <div class="mb-3 flex items-start justify-between gap-3">
                                            <div class="min-w-0">
                                                <p class="truncate text-sm font-semibold text-gray-900">{{ $column['label'] }}</p>
                                                <p class="text-xs text-gray-500"><span data-status-count>{{ $column['count'] }}</span> projects</p>
                                            </div>
                                            <span class="inline-flex rounded-full border px-2 py-1 text-[11px] font-semibold {{ $column['header_class'] }}">
                                                $<span data-status-budget>{{ number_format((float) $column['budget'], 0, '.', ' ') }}</span>
                                            </span>
                                        </div>

                                        <div class="max-h-[calc(100vh-18rem)] space-y-2 overflow-y-auto pr-1" data-status-list>
                                            @forelse ($column['projects'] as $project)
                                                @php
                                                    $canUpdateProject = auth()->user()?->can('update', $project) ?? false;
                                                @endphp
                                                <article
                                                    class="h-auto rounded-xl border border-gray-200 bg-white p-4 shadow-sm transition hover:border-indigo-200 hover:shadow {{ $canUpdateProject ? 'cursor-move' : 'cursor-default' }}"
                                                    draggable="{{ $canUpdateProject ? 'true' : 'false' }}"
                                                    data-project-card
                                                    data-project-id="{{ $project->id }}"
                                                    data-status="{{ $project->status }}"
                                                    data-budget="{{ (float) $project->budget }}"
                                                    data-can-update="{{ $canUpdateProject ? '1' : '0' }}"
                                                    data-update-url="{{ route('projects.update-status', $project) }}"
                                                >
                                                    <div class="flex items-start justify-between gap-3">
                                                        <div class="min-w-0">
                                                            <a href="{{ route('projects.show', $project) }}" class="block truncate text-sm font-semibold text-gray-900 hover:text-indigo-600">
                                                                {{ $project->name }}
                                                            </a>
                                                            <p class="mt-1 truncate text-xs text-gray-500">{{ $project->code }} · {{ $project->company?->name ?? __('Without company') }}</p>
                                                        </div>
                                                        <span class="inline-flex rounded-full px-2 py-1 text-[11px] font-medium {{ $project->priority === 'critical' ? 'bg-rose-100 text-rose-700' : ($project->priority === 'high' ? 'bg-amber-100 text-amber-700' : ($project->priority === 'medium' ? 'bg-sky-100 text-sky-700' : 'bg-slate-100 text-slate-700')) }}">
                                                            {{ __(ucfirst($project->priority)) }}
                                                        </span>
                                                    </div>

                                                    @if ($project->deal)
                                                        <p class="mt-3 text-xs text-indigo-600">{{ __('Deal') }}: {{ $project->deal->title }}</p>
                                                    @endif

                                                    <div class="mt-3 grid grid-cols-2 gap-3 text-xs text-gray-600">
                                                        <div>
                                                            <p class="uppercase text-[10px] tracking-wide text-gray-400">{{ __('Owner') }}</p>
                                                            <p class="mt-1 truncate">{{ $project->owner?->name ?? '—' }}</p>
                                                        </div>
                                                        <div>
                                                            <p class="uppercase text-[10px] tracking-wide text-gray-400">{{ __('Supervisor') }}</p>
                                                            <p class="mt-1 truncate">{{ $project->manager?->name ?? '—' }}</p>
                                                        </div>
                                                        <div>
                                                            <p class="uppercase text-[10px] tracking-wide text-gray-400">{{ __('Dates') }}</p>
                                                            <p class="mt-1">{{ $project->starts_at?->format('d.m.Y') ?? '—' }} → {{ $project->due_at?->format('d.m.Y') ?? '—' }}</p>
                                                        </div>
                                                        <div>
                                                            <p class="uppercase text-[10px] tracking-wide text-gray-400">{{ __('Tasks') }}</p>
                                                            <p class="mt-1">{{ $project->open_tasks_count }}/{{ $project->tasks_count }} {{ __('open') }}</p>
                                                        </div>
                                                    </div>

                                                    <div class="mt-3">
                                                        <div class="flex items-center justify-between text-xs text-gray-500">
                                                            <span>{{ __('Progress') }}</span>
                                                            <span>{{ $project->progress }}%</span>
                                                        </div>
                                                        <div class="mt-1 h-2 rounded-full bg-gray-100">
                                                            <div class="h-2 rounded-full {{ $project->progress >= 80 ? 'bg-emerald-500' : ($project->progress >= 40 ? 'bg-amber-500' : 'bg-indigo-500') }}" style="width: {{ $project->progress }}%;"></div>
                                                        </div>
                                                    </div>

                                                    <div class="mt-3 flex items-center justify-between text-xs text-gray-500">
                                                        <span>{{ __('Budget') }}: ${{ number_format((float) $project->budget, 0, '.', ' ') }}</span>
                                                        @if ($canUpdateProject)
                                                            <span class="inline-flex items-center gap-1 text-[11px] text-gray-400">
                                                                <x-menu-icon name="fa-solid fa-grip-vertical" class="h-3 w-3" />
                                                                {{ __('Move') }}
                                                            </span>
                                                        @endif
                                                    </div>
                                                </article>
                                            @empty
                                                <p class="text-xs text-gray-500" data-empty>No projects</p>
                                            @endforelse
                                        </div>
                                    </div>
                                @endforeach
                            </div>
                        </div>
                        <div
                            class="pointer-events-none absolute inset-y-0 right-0 w-16 bg-gradient-to-l from-gray-100 via-gray-100/90 to-transparent opacity-0 transition-opacity duration-200"
                            data-project-board-fade-right
                        ></div>
                    </div>
                </section>
            @else
                <section class="bg-white border border-gray-200 rounded-xl overflow-hidden">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Project</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Deadlines</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Owner/RP</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Progress</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Finance</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Tasks</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-100 bg-white">
                            @forelse ($projects as $project)
                                <tr class="hover:bg-gray-50">
                                    <td class="px-4 py-3 align-top">
                                        <a href="{{ route('projects.show', $project) }}" class="font-medium text-gray-900 hover:text-indigo-600">{{ $project->name }}</a>
                                        <div class="text-xs text-gray-500">{{ $project->code }} · {{ $project->company?->name ?? 'Without company' }}</div>
                                        @if ($project->deal)
                                            <div class="text-xs text-indigo-600">Deal: {{ $project->deal->title }}</div>
                                        @endif
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">
                                        {{ $project->starts_at?->format('d.m.Y') ?? '—' }} → {{ $project->due_at?->format('d.m.Y') ?? '—' }}
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">
                                        <div>{{ $project->owner?->name ?? '—' }}</div>
                                        <div class="text-xs text-gray-500">{{ $project->manager?->name ?? '—' }}</div>
                                    </td>
                                    <td class="px-4 py-3 align-top">
                                        <div class="text-sm text-gray-800 font-medium">{{ $project->progress }}%</div>
                                        <div class="mt-1 h-2 rounded-full bg-gray-100">
                                            <div class="h-2 rounded-full {{ $project->progress >= 80 ? 'bg-emerald-500' : ($project->progress >= 40 ? 'bg-amber-500' : 'bg-indigo-500') }}" style="width: {{ $project->progress }}%;"></div>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">
                                        <div>Budget: ${{ number_format((float) $project->budget, 0, '.', ' ') }}</div>
                                        <div class="text-xs text-gray-500">Fact: ${{ number_format((float) $project->spent, 0, '.', ' ') }}</div>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-700 align-top">
                                        <div>Total: {{ $project->tasks_count }}</div>
                                        <div class="text-xs text-gray-500">Open: {{ $project->open_tasks_count }} · Done: {{ $project->done_tasks_count }}</div>
                                    </td>
                                    <td class="px-4 py-3 align-top">
                                        <span class="inline-flex rounded-full px-2 py-1 text-xs font-medium
                                            {{ $project->status === 'completed' ? 'bg-emerald-100 text-emerald-800' : '' }}
                                            {{ $project->status === 'active' ? 'bg-indigo-100 text-indigo-800' : '' }}
                                            {{ $project->status === 'on_hold' ? 'bg-amber-100 text-amber-800' : '' }}
                                            {{ $project->status === 'planned' ? 'bg-gray-100 text-gray-700' : '' }}
                                            {{ $project->status === 'cancelled' ? 'bg-red-100 text-red-700' : '' }}
                                        ">{{ $projectStatusLabels[$project->status] ?? $project->status }}</span>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="7" class="px-4 py-8 text-center text-sm text-gray-500">No projects found.</td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </section>

                <div>{{ $projects->links() }}</div>
            @endif
        </div>
    </div>

    @if ($viewMode === 'kanban')
        <script>
            document.addEventListener('DOMContentLoaded', () => {
                const board = document.getElementById('project-board');

                if (!board) {
                    return;
                }

                const csrf = board.dataset.csrf;
                const boardScroll = board.querySelector('[data-project-board-scroll]');
                const fadeRight = board.querySelector('[data-project-board-fade-right]');
                let draggingCard = null;
                let previousColumn = null;

                const money = (value) => new Intl.NumberFormat(document.documentElement.lang || 'ru-RU', {
                    maximumFractionDigits: 0,
                }).format(Number(value || 0));

                const getColumns = () => Array.from(board.querySelectorAll('[data-status-column]'));

                const updateScrollFade = () => {
                    if (!boardScroll || !fadeRight) {
                        return;
                    }

                    const maxScrollLeft = boardScroll.scrollWidth - boardScroll.clientWidth;
                    fadeRight.classList.toggle('opacity-0', maxScrollLeft <= 2 || boardScroll.scrollLeft >= maxScrollLeft - 2);
                };

                const updateColumnStats = (column) => {
                    const list = column.querySelector('[data-status-list]');
                    const cards = list ? Array.from(list.querySelectorAll('[data-project-card]')) : [];
                    const countNode = column.querySelector('[data-status-count]');
                    const budgetNode = column.querySelector('[data-status-budget]');
                    const budgetTotal = cards.reduce((sum, card) => sum + Number(card.dataset.budget || 0), 0);
                    const emptyNode = list ? list.querySelector('[data-empty]') : null;

                    if (countNode) {
                        countNode.textContent = String(cards.length);
                    }

                    if (budgetNode) {
                        budgetNode.textContent = money(budgetTotal);
                    }

                    if (emptyNode) {
                        emptyNode.classList.toggle('hidden', cards.length > 0);
                    }

                    if (cards.length === 0 && list && !emptyNode) {
                        const placeholder = document.createElement('p');
                        placeholder.dataset.empty = '1';
                        placeholder.className = 'text-xs text-gray-500';
                        placeholder.textContent = 'No projects';
                        list.appendChild(placeholder);
                    }
                };

                const moveCardToColumn = (card, column) => {
                    const list = column.querySelector('[data-status-list]');
                    if (!list) {
                        return;
                    }

                    const emptyNode = list.querySelector('[data-empty]');
                    if (emptyNode) {
                        emptyNode.remove();
                    }

                    list.appendChild(card);
                    card.dataset.status = column.dataset.statusColumn || '';
                };

                const revertCardMove = (card) => {
                    if (!previousColumn) {
                        return;
                    }

                    moveCardToColumn(card, previousColumn);
                    updateColumnStats(previousColumn);
                };

                const persistProjectStatus = async (card, status) => {
                    const response = await fetch(card.dataset.updateUrl || '', {
                        method: 'PATCH',
                        headers: {
                            'Content-Type': 'application/json',
                            'Accept': 'application/json',
                            'X-CSRF-TOKEN': csrf,
                            'X-Requested-With': 'XMLHttpRequest',
                        },
                        body: JSON.stringify({ status }),
                    });

                    if (!response.ok) {
                        throw new Error('Project status update failed.');
                    }

                    return response.json();
                };

                board.addEventListener('dragstart', (event) => {
                    const card = event.target.closest('[data-project-card]');
                    if (!card || card.dataset.canUpdate !== '1') {
                        event.preventDefault();
                        return;
                    }

                    draggingCard = card;
                    previousColumn = card.closest('[data-status-column]');
                    card.classList.add('opacity-60');
                    event.dataTransfer.effectAllowed = 'move';
                    event.dataTransfer.setData('text/plain', card.dataset.projectId || '');
                });

                board.addEventListener('dragend', () => {
                    if (draggingCard) {
                        draggingCard.classList.remove('opacity-60');
                    }

                    draggingCard = null;
                    previousColumn = null;
                });

                getColumns().forEach((column) => {
                    const list = column.querySelector('[data-status-list]');
                    if (!list) {
                        return;
                    }

                    list.addEventListener('dragover', (event) => {
                        if (!draggingCard) {
                            return;
                        }

                        event.preventDefault();
                        event.dataTransfer.dropEffect = 'move';
                    });

                    list.addEventListener('drop', async (event) => {
                        if (!draggingCard) {
                            return;
                        }

                        event.preventDefault();

                        const card = draggingCard;
                        const sourceColumn = previousColumn;
                        const targetStatus = column.dataset.statusColumn || '';
                        const previousStatus = card.dataset.status || '';
                        if (targetStatus === '' || targetStatus === previousStatus) {
                            return;
                        }

                        moveCardToColumn(card, column);

                        if (sourceColumn) {
                            updateColumnStats(sourceColumn);
                        }
                        updateColumnStats(column);

                        try {
                            await persistProjectStatus(card, targetStatus);
                        } catch (error) {
                            if (sourceColumn) {
                                moveCardToColumn(card, sourceColumn);
                            }

                            if (sourceColumn) {
                                updateColumnStats(sourceColumn);
                            }
                            updateColumnStats(column);
                        }
                    });

                    updateColumnStats(column);
                });

                if (boardScroll) {
                    boardScroll.addEventListener('scroll', updateScrollFade, { passive: true });
                    updateScrollFade();
                }
            });
        </script>
    @endif
</x-app-layout>
