<?php

namespace App\Support\Tasks;

use App\Models\Task;
use Illuminate\Support\Collection;

class TaskBoardBuilder
{
    /**
     * @param  Collection<int, Task>  $tasks
     * @param  array<int, array<string, mixed>>  $kanbanStages
     * @return Collection<int, array<string, mixed>>
     */
    public function buildKanbanColumns(Collection $tasks, array $kanbanStages): Collection
    {
        /** @var Collection<string, Collection<int, Task>> $tasksByStatus */
        $tasksByStatus = $tasks->groupBy(fn (Task $task): string => (string) $task->status);

        return collect($kanbanStages)->map(function (array $stage) use ($tasksByStatus): array {
            $status = (string) $stage['status'];
            /** @var Collection<int, Task> $columnTasks */
            $columnTasks = $tasksByStatus->get($status, collect())->values();

            return [
                'status' => $status,
                'label' => (string) ($stage['label'] ?? $status),
                'color' => (string) ($stage['color'] ?? '#94A3B8'),
                'tasks' => $columnTasks,
                'count' => $columnTasks->count(),
                'hours' => (float) $columnTasks->sum('tracked_hours'),
            ];
        })->values();
    }

    /**
     * @param  Collection<int, Task>  $tasks
     * @return array{rows: Collection<int, array<string, mixed>>, period: array<string, mixed>|null}
     */
    public function buildGanttRows(Collection $tasks): array
    {
        if ($tasks->isEmpty()) {
            return ['rows' => collect(), 'period' => null];
        }

        $normalized = $tasks->map(function (Task $task): array {
            $start = $task->starts_at ?? $task->created_at;
            $end = $task->due_at ?? $start?->copy()->addDay();

            if ($start && $end && $end->lt($start)) {
                $end = $start->copy()->addDay();
            }

            return [
                'task' => $task,
                'start' => $start,
                'end' => $end,
            ];
        })->filter(fn (array $row) => $row['start'] && $row['end'])->values();

        if ($normalized->isEmpty()) {
            return ['rows' => collect(), 'period' => null];
        }

        $periodStart = $normalized->min('start')->copy()->startOfDay();
        $periodEnd = $normalized->max('end')->copy()->endOfDay();
        $totalSeconds = max(86400, $periodStart->diffInSeconds($periodEnd));

        $rows = $normalized->map(function (array $row) use ($periodStart, $totalSeconds): array {
            /** @var Task $task */
            $task = $row['task'];
            $start = $row['start'];
            $end = $row['end'];

            $left = ($periodStart->diffInSeconds($start, false) / $totalSeconds) * 100;
            $width = max(1.5, ($start->diffInSeconds($end) / $totalSeconds) * 100);

            return [
                'task' => $task,
                'start' => $start,
                'end' => $end,
                'left' => max(0, min(99, $left)),
                'width' => max(1.5, min(100, $width)),
                'is_overdue' => $task->status !== 'done' && $task->due_at && $task->due_at->lt(now()),
            ];
        })->values();

        $todayPosition = ($periodStart->diffInSeconds(now(), false) / $totalSeconds) * 100;

        return [
            'rows' => $rows,
            'period' => [
                'start' => $periodStart,
                'end' => $periodEnd,
                'today_position' => max(0, min(100, $todayPosition)),
            ],
        ];
    }
}
