<?php

namespace App\Http\Controllers;

use App\Http\Resources\UpdateInstallationResource;
use App\Models\UpdateInstallation;
use App\Support\UpdateInstaller;
use App\Support\UpdateCenterManager;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Validation\Rule;

class ProfileUpdateCenterController extends Controller
{
    public function update(Request $request, UpdateCenterManager $updateCenterManager): RedirectResponse
    {
        if (! $updateCenterManager->storageIsReady()) {
            return Redirect::route('profile.edit', ['section' => 'updates'])
                ->with('status', 'updates-storage-unavailable')
                ->with('error', __('Update tables are not migrated. Run "php artisan migrate".'));
        }

        $validated = $request->validate([
            'feed_url' => ['required', 'string', 'max:255'],
            'current_version' => ['nullable', 'string', 'max:64'],
            'channel' => ['required', Rule::in(['stable', 'beta', 'nightly'])],
            'is_active' => ['nullable', 'boolean'],
            'auto_check_enabled' => ['nullable', 'boolean'],
            'check_interval_minutes' => ['required', 'integer', 'min:5', 'max:10080'],
            'request_timeout_seconds' => ['required', 'integer', 'min:2', 'max:60'],
            'verify_tls' => ['nullable', 'boolean'],
            'meta' => ['nullable', 'json'],
        ]);

        $settings = $updateCenterManager->settings();
        $currentVersion = trim((string) Arr::get($validated, 'current_version', ''));

        $settings->fill([
            'feed_url' => $updateCenterManager->normalizeFeedUrl((string) $validated['feed_url']),
            'current_version' => $currentVersion !== '' ? $currentVersion : $updateCenterManager->defaultCurrentVersion(),
            'channel' => (string) $validated['channel'],
            'is_active' => $request->boolean('is_active', true),
            'auto_check_enabled' => $request->boolean('auto_check_enabled', true),
            'check_interval_minutes' => (int) $validated['check_interval_minutes'],
            'request_timeout_seconds' => (int) $validated['request_timeout_seconds'],
            'verify_tls' => $request->boolean('verify_tls', true),
            'meta' => $this->decodeJsonArray($validated['meta'] ?? null),
        ]);
        $settings->save();

        return Redirect::route('profile.edit', ['section' => 'updates'])
            ->with('status', 'updates-settings-updated')
            ->with('success', __('Update settings have been saved.'));
    }

    public function check(Request $request, UpdateCenterManager $updateCenterManager): RedirectResponse
    {
        if (! $updateCenterManager->storageIsReady()) {
            return Redirect::route('profile.edit', ['section' => 'updates'])
                ->with('status', 'updates-storage-unavailable')
                ->with('error', __('Update tables are not migrated. Run "php artisan migrate".'));
        }

        $check = $updateCenterManager->checkNow();

        if ($check->status === 'success' && $check->is_update_available) {
            return Redirect::route('profile.edit', ['section' => 'updates'])
                ->with('status', 'updates-check-completed')
                ->with('success', __('Update available: :version', ['version' => (string) ($check->remote_version ?? 'n/a')]));
        }

        if ($check->status === 'success') {
            return Redirect::route('profile.edit', ['section' => 'updates'])
                ->with('status', 'updates-check-completed')
                ->with('success', __('No new updates were found.'));
        }

        return Redirect::route('profile.edit', ['section' => 'updates'])
            ->with('status', 'updates-check-completed')
            ->with('error', __('Update check failed: :error', ['error' => (string) ($check->error_message ?? 'Unknown error.')]));
    }

    public function install(Request $request, UpdateCenterManager $updateCenterManager, UpdateInstaller $updateInstaller): RedirectResponse|JsonResponse
    {
        if (! $updateCenterManager->storageIsReady()) {
            if ($request->expectsJson()) {
                return response()->json([
                    'message' => __('Update tables are not migrated. Run "php artisan migrate".'),
                    'code' => 'updates_storage_missing',
                ], 503);
            }

            return Redirect::route('profile.edit', ['section' => 'updates'])
                ->with('status', 'updates-storage-unavailable')
                ->with('error', __('Update tables are not migrated. Run "php artisan migrate".'));
        }

        if ($request->expectsJson()) {
            $settings = $updateCenterManager->settings();
            $existingInstallation = $updateInstaller->activeInstallation($settings);
            if ($existingInstallation) {
                return UpdateInstallationResource::make($existingInstallation->loadMissing('actor:id,name,email'))
                    ->response()
                    ->setStatusCode(202);
            }

            $installation = $updateInstaller->queueLatest($request->user(), $settings);
            $installation = $updateInstaller->startBackgroundProcessing($installation);

            return UpdateInstallationResource::make($installation->fresh())
                ->response()
                ->setStatusCode(202);
        }

        $installation = $updateInstaller->installLatest($request->user(), $updateCenterManager->settings());

        if ($installation->status === 'installed') {
            return Redirect::route('profile.edit', ['section' => 'updates'])
                ->with('status', 'updates-install-completed')
                ->with('success', __('Update installed: :version', ['version' => (string) ($installation->target_version ?: 'n/a')]));
        }

        if ($installation->status === 'no_update') {
            return Redirect::route('profile.edit', ['section' => 'updates'])
                ->with('status', 'updates-install-completed')
                ->with('success', __('No new updates were found.'));
        }

        return Redirect::route('profile.edit', ['section' => 'updates'])
            ->with('status', 'updates-install-completed')
            ->with('error', __('Update installation failed: :error', ['error' => (string) ($installation->message ?: __('Unknown error.'))]));
    }

    public function showInstallation(Request $request, UpdateCenterManager $updateCenterManager, UpdateInstallation $installation): JsonResponse
    {
        if (! $updateCenterManager->storageIsReady()) {
            return response()->json([
                'message' => __('Update tables are not migrated. Run "php artisan migrate".'),
                'code' => 'updates_storage_missing',
            ], 503);
        }

        abort_unless($installation->update_setting_id === $updateCenterManager->settings()->id, 404);

        return UpdateInstallationResource::make($installation->loadMissing('actor:id,name,email'))->response();
    }

    /**
     * @return array<string, mixed>|null
     */
    private function decodeJsonArray(mixed $value): ?array
    {
        $raw = trim((string) $value);
        if ($raw === '') {
            return null;
        }

        /** @var mixed $decoded */
        $decoded = json_decode($raw, true);

        return is_array($decoded) ? $decoded : null;
    }
}
