<?php

namespace App\Http\Controllers;

use App\Support\TwoFactorAuthenticator;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class ProfileTwoFactorController extends Controller
{
    public function setup(Request $request, TwoFactorAuthenticator $twoFactorAuthenticator): RedirectResponse
    {
        $user = $request->user();

        if ($user->hasTwoFactorEnabled()) {
            return back()->with('success', __('Two-factor authentication is enabled.'));
        }

        $request->session()->put('two_factor.pending', [
            'user_id' => $user->id,
            'secret' => $twoFactorAuthenticator->generateSecret(),
            'recovery_codes' => $twoFactorAuthenticator->generateRecoveryCodes(),
        ]);

        return back()->with('success', __('Two-factor setup started. Scan the key and confirm with a code.'));
    }

    public function confirm(Request $request, TwoFactorAuthenticator $twoFactorAuthenticator): RedirectResponse
    {
        $validated = $request->validate([
            'code' => ['required', 'string', 'max:20'],
        ]);

        $user = $request->user();
        $pending = $request->session()->get('two_factor.pending');

        $secret = is_array($pending) ? (string) ($pending['secret'] ?? '') : '';
        $recoveryCodes = is_array($pending) && is_array($pending['recovery_codes'] ?? null)
            ? array_values($pending['recovery_codes'])
            : [];

        if (! is_array($pending) || (int) ($pending['user_id'] ?? 0) !== $user->id || $secret === '' || $recoveryCodes === []) {
            throw ValidationException::withMessages([
                'code' => __('Two-factor setup session expired. Start setup again.'),
            ]);
        }

        if (! $twoFactorAuthenticator->verifyCode($secret, (string) $validated['code'])) {
            throw ValidationException::withMessages([
                'code' => __('Invalid two-factor code.'),
            ]);
        }

        $user->forceFill([
            'two_factor_secret' => $secret,
            'two_factor_recovery_codes' => $recoveryCodes,
            'two_factor_confirmed_at' => now(),
        ])->save();

        $request->session()->forget('two_factor.pending');

        return back()->with('success', __('Two-factor authentication enabled.'));
    }

    public function regenerateRecoveryCodes(Request $request, TwoFactorAuthenticator $twoFactorAuthenticator): RedirectResponse
    {
        $user = $request->user();

        if (! $user->hasTwoFactorEnabled()) {
            return back()->with('error', __('Two-factor authentication is currently disabled.'));
        }

        $user->forceFill([
            'two_factor_recovery_codes' => $twoFactorAuthenticator->generateRecoveryCodes(),
        ])->save();

        return back()->with('success', __('Recovery codes regenerated.'));
    }

    public function disable(Request $request): RedirectResponse
    {
        $request->user()->forceFill([
            'two_factor_secret' => null,
            'two_factor_recovery_codes' => null,
            'two_factor_confirmed_at' => null,
        ])->save();

        $request->session()->forget('two_factor.pending');

        return back()->with('success', __('Two-factor authentication disabled.'));
    }
}
