<?php

namespace App\Http\Controllers;

use App\Models\Activity;
use App\Models\Company;
use App\Models\Contact;
use App\Models\Deal;
use App\Models\Pipeline;
use App\Models\Project;
use App\Models\Task;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class DashboardController extends Controller
{
    public function index(Request $request): View|JsonResponse
    {
        $now = now();

        $stats = [
            'companies' => Company::count(),
            'contacts' => Contact::count(),
            'open_deals' => Deal::where('status', 'open')->count(),
            'pipeline_value' => Deal::where('status', 'open')->sum('amount'),
            'won_this_month' => Deal::query()
                ->where('status', 'won')
                ->whereBetween('closed_at', [$now->copy()->startOfMonth(), $now->copy()->endOfMonth()])
                ->sum('amount'),
            'active_projects' => Project::where('status', 'active')->count(),
            'project_avg_progress' => (int) round((float) Project::avg('progress')),
            'overdue_tasks' => Task::query()
                ->whereIn('status', ['todo', 'in_progress', 'review'])
                ->whereNotNull('due_at')
                ->where('due_at', '<', $now)
                ->count(),
        ];

        $pipeline = Pipeline::query()->where('is_default', true)->first()
            ?? Pipeline::query()->oldest('sort_order')->first();

        $stages = collect();

        if ($pipeline) {
            $stages = $pipeline->stages()->withCount([
                'deals as open_deals_count' => fn ($query) => $query->where('status', 'open'),
            ])->get();
        }

        $topDeals = Deal::query()
            ->with(['company', 'stage'])
            ->where('status', 'open')
            ->orderByDesc('amount')
            ->limit(6)
            ->get();

        $priorityProjects = Project::query()
            ->with(['manager', 'company'])
            ->whereIn('status', ['active', 'on_hold'])
            ->orderByDesc('priority')
            ->orderByRaw('due_at is null')
            ->orderBy('due_at')
            ->limit(8)
            ->get();

        $upcomingTasks = Task::query()
            ->with(['assignee', 'deal', 'project'])
            ->whereIn('status', ['todo', 'in_progress', 'review'])
            ->orderByRaw('due_at is null')
            ->orderBy('due_at')
            ->limit(8)
            ->get();

        $recentActivities = Activity::query()
            ->with(['user', 'deal', 'company', 'contact'])
            ->latest('occurred_at')
            ->limit(10)
            ->get();

        $dashboardPayload = [
            'meta' => [
                'title' => __('CRM panel'),
                'subtitle' => __('Live business snapshot with the most important entities and activity.'),
                'generated_at' => $now->format('d.m.Y H:i'),
            ],
            'stats' => [
                [
                    'key' => 'companies',
                    'label' => __('Companies'),
                    'value' => (int) ($stats['companies'] ?? 0),
                    'icon' => 'fa-solid fa-building',
                    'tone' => 'default',
                ],
                [
                    'key' => 'contacts',
                    'label' => __('Contacts'),
                    'value' => (int) ($stats['contacts'] ?? 0),
                    'icon' => 'fa-solid fa-address-book',
                    'tone' => 'default',
                ],
                [
                    'key' => 'open_deals',
                    'label' => __('Open trades'),
                    'value' => (int) ($stats['open_deals'] ?? 0),
                    'icon' => 'fa-solid fa-handshake',
                    'tone' => 'info',
                ],
                [
                    'key' => 'pipeline_value',
                    'label' => __('Amount in work'),
                    'value' => (float) ($stats['pipeline_value'] ?? 0),
                    'display' => '$'.number_format((float) ($stats['pipeline_value'] ?? 0), 0, '.', ' '),
                    'icon' => 'fa-solid fa-chart-line',
                    'tone' => 'info',
                ],
                [
                    'key' => 'won_this_month',
                    'label' => __('Won this month'),
                    'value' => (float) ($stats['won_this_month'] ?? 0),
                    'display' => '$'.number_format((float) ($stats['won_this_month'] ?? 0), 0, '.', ' '),
                    'icon' => 'fa-solid fa-trophy',
                    'tone' => 'success',
                ],
                [
                    'key' => 'active_projects',
                    'label' => __('Active projects'),
                    'value' => (int) ($stats['active_projects'] ?? 0),
                    'icon' => 'fa-solid fa-diagram-project',
                    'tone' => 'default',
                ],
                [
                    'key' => 'project_avg_progress',
                    'label' => __('Average project progress'),
                    'value' => (int) ($stats['project_avg_progress'] ?? 0),
                    'display' => number_format((int) ($stats['project_avg_progress'] ?? 0)).'%',
                    'icon' => 'fa-solid fa-gauge-high',
                    'tone' => 'accent',
                ],
                [
                    'key' => 'overdue_tasks',
                    'label' => __('Overdue tasks'),
                    'value' => (int) ($stats['overdue_tasks'] ?? 0),
                    'icon' => 'fa-solid fa-triangle-exclamation',
                    'tone' => ((int) ($stats['overdue_tasks'] ?? 0)) > 0 ? 'danger' : 'default',
                ],
            ],
            'funnel' => [
                'name' => $pipeline?->name,
                'stages' => $stages->map(static fn ($stage): array => [
                    'id' => (int) $stage->id,
                    'name' => (string) $stage->name,
                    'probability' => (int) $stage->probability,
                    'open_deals_count' => (int) $stage->open_deals_count,
                ])->values()->all(),
            ],
            'top_deals' => $topDeals->map(static fn (Deal $deal): array => [
                'id' => (int) $deal->id,
                'title' => (string) $deal->title,
                'amount' => (float) $deal->amount,
                'amount_display' => '$'.number_format((float) $deal->amount, 0, '.', ' '),
                'company_name' => (string) ($deal->company?->name ?? __('Without company')),
                'stage_name' => (string) ($deal->stage?->name ?? '—'),
                'url' => route('deals.show', $deal),
            ])->values()->all(),
            'priority_projects' => $priorityProjects->map(static fn (Project $project): array => [
                'id' => (int) $project->id,
                'name' => (string) $project->name,
                'company_name' => (string) ($project->company?->name ?? __('Without company')),
                'manager_name' => (string) ($project->manager?->name ?? __('Without RP')),
                'progress' => (int) ($project->progress ?? 0),
                'due_display' => $project->due_at?->format('d.m') ?? __('No deadline'),
                'url' => route('projects.show', $project),
            ])->values()->all(),
            'upcoming_tasks' => $upcomingTasks->map(static fn (Task $task): array => [
                'id' => (int) $task->id,
                'title' => (string) $task->title,
                'assignee_name' => (string) ($task->assignee?->name ?? __('Not assigned')),
                'link_name' => (string) ($task->project?->name ?? $task->deal?->title ?? __('No connection')),
                'due_display' => $task->due_at?->format('d.m H:i') ?? __('No deadline'),
                'url' => route('tasks.show', $task),
            ])->values()->all(),
            'recent_activities' => $recentActivities->map(static fn (Activity $activity): array => [
                'id' => (int) $activity->id,
                'subject' => (string) $activity->subject,
                'type' => (string) strtoupper((string) $activity->type),
                'time_display' => $activity->occurred_at?->format('d.m H:i') ?? '',
                'user_name' => (string) ($activity->user?->name ?? __('System')),
                'url' => route('activities.show', $activity),
            ])->values()->all(),
            'links' => [
                'deals_index' => route('deals.index'),
                'deals_create' => route('deals.create'),
                'projects_index' => route('projects.index'),
                'tasks_index' => route('tasks.index'),
                'activities_index' => route('activities.index'),
            ],
        ];

        if ($request->expectsJson() || $request->wantsJson() || $request->isXmlHttpRequest()) {
            return response()->json($dashboardPayload);
        }

        return view('dashboard', [
            'dashboardPayload' => $dashboardPayload,
        ]);
    }
}
