<?php

namespace App\Support;

class DiskFileIconResolver
{
    /**
     * @var array<string, array{icon:string,tone:string,label:string}>
     */
    private const EXTENSION_MAP = [
        'pdf' => ['icon' => 'fa-solid fa-file-pdf', 'tone' => 'red', 'label' => 'PDF'],
        'doc' => ['icon' => 'fa-solid fa-file-word', 'tone' => 'blue', 'label' => 'Word'],
        'docx' => ['icon' => 'fa-solid fa-file-word', 'tone' => 'blue', 'label' => 'Word'],
        'odt' => ['icon' => 'fa-solid fa-file-word', 'tone' => 'blue', 'label' => 'Word'],
        'xls' => ['icon' => 'fa-solid fa-file-excel', 'tone' => 'green', 'label' => 'Excel'],
        'xlsx' => ['icon' => 'fa-solid fa-file-excel', 'tone' => 'green', 'label' => 'Excel'],
        'csv' => ['icon' => 'fa-solid fa-file-csv', 'tone' => 'green', 'label' => 'CSV'],
        'ods' => ['icon' => 'fa-solid fa-file-excel', 'tone' => 'green', 'label' => 'Sheet'],
        'ppt' => ['icon' => 'fa-solid fa-file-powerpoint', 'tone' => 'orange', 'label' => 'PowerPoint'],
        'pptx' => ['icon' => 'fa-solid fa-file-powerpoint', 'tone' => 'orange', 'label' => 'PowerPoint'],
        'odp' => ['icon' => 'fa-solid fa-file-powerpoint', 'tone' => 'orange', 'label' => 'Presentation'],
        'zip' => ['icon' => 'fa-solid fa-file-zipper', 'tone' => 'amber', 'label' => 'Archive'],
        'rar' => ['icon' => 'fa-solid fa-file-zipper', 'tone' => 'amber', 'label' => 'Archive'],
        '7z' => ['icon' => 'fa-solid fa-file-zipper', 'tone' => 'amber', 'label' => 'Archive'],
        'tar' => ['icon' => 'fa-solid fa-file-zipper', 'tone' => 'amber', 'label' => 'Archive'],
        'gz' => ['icon' => 'fa-solid fa-file-zipper', 'tone' => 'amber', 'label' => 'Archive'],
        'jpeg' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'jpg' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'png' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'gif' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'webp' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'svg' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'bmp' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'heic' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'mp4' => ['icon' => 'fa-solid fa-file-video', 'tone' => 'purple', 'label' => 'Video'],
        'mov' => ['icon' => 'fa-solid fa-file-video', 'tone' => 'purple', 'label' => 'Video'],
        'avi' => ['icon' => 'fa-solid fa-file-video', 'tone' => 'purple', 'label' => 'Video'],
        'mkv' => ['icon' => 'fa-solid fa-file-video', 'tone' => 'purple', 'label' => 'Video'],
        'webm' => ['icon' => 'fa-solid fa-file-video', 'tone' => 'purple', 'label' => 'Video'],
        'mp3' => ['icon' => 'fa-solid fa-file-audio', 'tone' => 'violet', 'label' => 'Audio'],
        'wav' => ['icon' => 'fa-solid fa-file-audio', 'tone' => 'violet', 'label' => 'Audio'],
        'ogg' => ['icon' => 'fa-solid fa-file-audio', 'tone' => 'violet', 'label' => 'Audio'],
        'flac' => ['icon' => 'fa-solid fa-file-audio', 'tone' => 'violet', 'label' => 'Audio'],
        'm4a' => ['icon' => 'fa-solid fa-file-audio', 'tone' => 'violet', 'label' => 'Audio'],
        'aac' => ['icon' => 'fa-solid fa-file-audio', 'tone' => 'violet', 'label' => 'Audio'],
        'txt' => ['icon' => 'fa-solid fa-file-lines', 'tone' => 'slate', 'label' => 'Text'],
        'log' => ['icon' => 'fa-solid fa-file-lines', 'tone' => 'slate', 'label' => 'Text'],
        'md' => ['icon' => 'fa-solid fa-file-lines', 'tone' => 'slate', 'label' => 'Text'],
        'rtf' => ['icon' => 'fa-solid fa-file-lines', 'tone' => 'slate', 'label' => 'Text'],
        'json' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'JSON'],
        'xml' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'yml' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'yaml' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'sql' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'SQL'],
        'php' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'js' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'ts' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'vue' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'html' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'css' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'scss' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'py' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'java' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'go' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
        'sh' => ['icon' => 'fa-solid fa-file-code', 'tone' => 'cyan', 'label' => 'Code'],
    ];

    /**
     * @var array<string, array{icon:string,tone:string,label:string}>
     */
    private const MIME_PREFIX_MAP = [
        'image/' => ['icon' => 'fa-solid fa-file-image', 'tone' => 'pink', 'label' => 'Image'],
        'video/' => ['icon' => 'fa-solid fa-file-video', 'tone' => 'purple', 'label' => 'Video'],
        'audio/' => ['icon' => 'fa-solid fa-file-audio', 'tone' => 'violet', 'label' => 'Audio'],
        'text/' => ['icon' => 'fa-solid fa-file-lines', 'tone' => 'slate', 'label' => 'Text'],
    ];

    /**
     * @return array{icon:string,label:string,tone:string,badge_class:string,icon_wrapper_class:string}
     */
    public static function resolve(?string $extension, ?string $mimeType = null): array
    {
        $normalizedExtension = strtolower(trim((string) $extension));
        $normalizedMimeType = strtolower(trim((string) $mimeType));

        $base = self::EXTENSION_MAP[$normalizedExtension]
            ?? self::resolveByMimeType($normalizedMimeType)
            ?? ['icon' => 'fa-solid fa-file', 'tone' => 'slate', 'label' => 'File'];

        return [
            'icon' => $base['icon'],
            'label' => $base['label'],
            'tone' => $base['tone'],
            'badge_class' => self::badgeClass($base['tone']),
            'icon_wrapper_class' => self::iconWrapperClass($base['tone']),
        ];
    }

    /**
     * @return array{icon:string,tone:string,label:string}|null
     */
    private static function resolveByMimeType(string $mimeType): ?array
    {
        if ($mimeType === '') {
            return null;
        }

        if ($mimeType === 'application/pdf') {
            return ['icon' => 'fa-solid fa-file-pdf', 'tone' => 'red', 'label' => 'PDF'];
        }

        if (str_contains($mimeType, 'wordprocessingml') || $mimeType === 'application/msword') {
            return ['icon' => 'fa-solid fa-file-word', 'tone' => 'blue', 'label' => 'Word'];
        }

        if (str_contains($mimeType, 'spreadsheetml') || str_contains($mimeType, 'excel')) {
            return ['icon' => 'fa-solid fa-file-excel', 'tone' => 'green', 'label' => 'Excel'];
        }

        if (str_contains($mimeType, 'presentationml') || str_contains($mimeType, 'powerpoint')) {
            return ['icon' => 'fa-solid fa-file-powerpoint', 'tone' => 'orange', 'label' => 'Presentation'];
        }

        if (str_contains($mimeType, 'zip') || str_contains($mimeType, 'compressed')) {
            return ['icon' => 'fa-solid fa-file-zipper', 'tone' => 'amber', 'label' => 'Archive'];
        }

        foreach (self::MIME_PREFIX_MAP as $prefix => $payload) {
            if (str_starts_with($mimeType, $prefix)) {
                return $payload;
            }
        }

        return null;
    }

    private static function badgeClass(string $tone): string
    {
        return match ($tone) {
            'red' => 'border-red-200 bg-red-50 text-red-700',
            'blue' => 'border-blue-200 bg-blue-50 text-blue-700',
            'green' => 'border-emerald-200 bg-emerald-50 text-emerald-700',
            'orange' => 'border-orange-200 bg-orange-50 text-orange-700',
            'amber' => 'border-amber-200 bg-amber-50 text-amber-700',
            'purple' => 'border-purple-200 bg-purple-50 text-purple-700',
            'violet' => 'border-violet-200 bg-violet-50 text-violet-700',
            'pink' => 'border-pink-200 bg-pink-50 text-pink-700',
            'cyan' => 'border-cyan-200 bg-cyan-50 text-cyan-700',
            default => 'border-slate-200 bg-slate-50 text-slate-700',
        };
    }

    private static function iconWrapperClass(string $tone): string
    {
        return match ($tone) {
            'red' => 'border-red-200 bg-red-50 text-red-600',
            'blue' => 'border-blue-200 bg-blue-50 text-blue-600',
            'green' => 'border-emerald-200 bg-emerald-50 text-emerald-600',
            'orange' => 'border-orange-200 bg-orange-50 text-orange-600',
            'amber' => 'border-amber-200 bg-amber-50 text-amber-600',
            'purple' => 'border-purple-200 bg-purple-50 text-purple-600',
            'violet' => 'border-violet-200 bg-violet-50 text-violet-600',
            'pink' => 'border-pink-200 bg-pink-50 text-pink-600',
            'cyan' => 'border-cyan-200 bg-cyan-50 text-cyan-600',
            default => 'border-slate-200 bg-slate-50 text-slate-600',
        };
    }
}
