<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\DiskResource;
use App\Models\Disk;
use App\Support\AccessControl;
use App\Support\CrmModuleManager;
use App\Support\DiskFileManager;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\StreamedResponse;

class DiskController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Disk::class, 'disk');
    }

    public function index(Request $request)
    {
        $search = trim((string) $request->input('q', ''));
        $folder = trim((string) $request->input('folder', ''));
        $ownerId = $request->integer('owner_id');

        $query = Disk::query()
            ->with('owner')
            ->when($search !== '', function ($builder) use ($search): void {
                $builder->where(function ($sub) use ($search): void {
                    $sub->where('name', 'like', "%{$search}%")
                        ->orWhere('original_name', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%")
                        ->orWhere('folder', 'like', "%{$search}%");
                });
            })
            ->when($folder !== '', fn ($builder) => $builder->where('folder', $folder))
            ->when($ownerId > 0, fn ($builder) => $builder->where('owner_id', $ownerId));

        if (! AccessControl::isElevated($request->user())) {
            $query->where('owner_id', $request->user()->id);
        }

        $disks = $query
            ->latest('id')
            ->paginate(20)
            ->withQueryString();

        return DiskResource::collection($disks);
    }

    public function store(Request $request, DiskFileManager $diskFileManager, CrmModuleManager $moduleManager)
    {
        $validated = $request->validate([
            'file' => ['required', 'file', 'max:51200'],
            'name' => ['nullable', 'string', 'max:255'],
            'folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
            'description' => ['nullable', 'string', 'max:5000'],
            'is_public' => ['nullable', 'boolean'],
        ]);
        $metadataPayload = [
            'name' => $validated['name'] ?? '',
            'folder' => $validated['folder'] ?? '',
            'description' => $validated['description'] ?? null,
            'is_public' => $request->boolean('is_public'),
        ];
        $metadataPayload = $moduleManager->applyPayloadHooks('disks.store', $metadataPayload, [
            'hook' => 'disks.store',
            'user_id' => $request->user()->id,
            'source' => 'api',
        ], array_keys($metadataPayload));

        $disk = $diskFileManager->createFromUpload($validated['file'], $request->user(), [
            'name' => $metadataPayload['name'] ?? '',
            'folder' => $metadataPayload['folder'] ?? '',
            'description' => $metadataPayload['description'] ?? null,
            'is_public' => (bool) ($metadataPayload['is_public'] ?? false),
        ]);

        return DiskResource::make($disk->load('owner'))
            ->response()
            ->setStatusCode(201);
    }

    public function show(Disk $disk): DiskResource
    {
        return DiskResource::make($disk->load('owner'));
    }

    public function update(Request $request, Disk $disk, DiskFileManager $diskFileManager, CrmModuleManager $moduleManager): DiskResource
    {
        $validated = $request->validate([
            'file' => ['nullable', 'file', 'max:51200'],
            'name' => ['nullable', 'string', 'max:255'],
            'folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
            'description' => ['nullable', 'string', 'max:5000'],
            'is_public' => ['nullable', 'boolean'],
        ]);

        $attributes = [];
        if (array_key_exists('name', $validated)) {
            $attributes['name'] = $validated['name'];
        }
        if (array_key_exists('folder', $validated)) {
            $attributes['folder'] = $validated['folder'];
        }
        if (array_key_exists('description', $validated)) {
            $attributes['description'] = $validated['description'];
        }
        if ($request->has('is_public')) {
            $attributes['is_public'] = $request->boolean('is_public');
        }
        if ($attributes !== []) {
            $attributes = $moduleManager->applyPayloadHooks('disks.update', $attributes, [
                'hook' => 'disks.update',
                'user_id' => $request->user()->id,
                'disk_id' => $disk->id,
                'source' => 'api',
            ], array_keys($attributes));
        }

        $disk = $diskFileManager->updateFromPayload(
            $disk,
            $attributes,
            $validated['file'] ?? null
        );

        return DiskResource::make($disk);
    }

    public function destroy(Disk $disk, DiskFileManager $diskFileManager)
    {
        $diskFileManager->delete($disk);

        return response()->noContent();
    }

    public function download(Disk $disk, DiskFileManager $diskFileManager): StreamedResponse
    {
        $this->authorize('view', $disk);

        return $diskFileManager->downloadResponse($disk);
    }
}
