<?php

namespace App\Policies;

use App\Models\MessengerConversation;
use App\Models\MessengerSetting;
use App\Models\User;
use App\Support\AccessControl;
use App\Support\MessengerAccess;

class MessengerConversationPolicy
{
    public function viewAny(User $user): bool
    {
        return AccessControl::allows($user, 'messengers', 'read');
    }

    public function view(User $user, MessengerConversation $conversation): bool
    {
        $decision = AccessControl::overridePermissionState($user, 'messengers', 'read');
        if ($decision !== null) {
            return $decision;
        }

        if (! AccessControl::roleAllows($user, 'messengers', 'read')) {
            return false;
        }

        if (AccessControl::isElevated($user) || $conversation->user_id === $user->id) {
            return true;
        }

        if ($conversation->user_id === null) {
            $settings = MessengerSetting::query()->first();

            return MessengerAccess::canAcceptIncoming($user, $settings);
        }

        return false;
    }

    public function create(User $user): bool
    {
        return AccessControl::allows($user, 'messengers', 'create');
    }

    public function update(User $user, MessengerConversation $conversation): bool
    {
        $decision = AccessControl::overridePermissionState($user, 'messengers', 'update');
        if ($decision !== null) {
            return $decision;
        }

        if (! AccessControl::roleAllows($user, 'messengers', 'update')) {
            return false;
        }

        if (AccessControl::isElevated($user) || $conversation->user_id === $user->id) {
            return true;
        }

        if ($conversation->user_id === null) {
            $settings = MessengerSetting::query()->first();

            return MessengerAccess::canAcceptIncoming($user, $settings);
        }

        return false;
    }

    public function delete(User $user, MessengerConversation $conversation): bool
    {
        $decision = AccessControl::overridePermissionState($user, 'messengers', 'delete');
        if ($decision !== null) {
            return $decision;
        }

        if (! AccessControl::roleAllows($user, 'messengers', 'delete')) {
            return false;
        }

        return AccessControl::isElevated($user) || $conversation->user_id === $user->id;
    }
}
