<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Project extends Model
{
    /** @use HasFactory<\Database\Factories\ProjectFactory> */
    use HasFactory;

    /**
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'code',
        'description',
        'company_id',
        'deal_id',
        'owner_id',
        'manager_id',
        'status',
        'priority',
        'health',
        'progress',
        'budget',
        'spent',
        'starts_at',
        'due_at',
        'completed_at',
        'visibility',
        'notes',
    ];

    /**
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'progress' => 'integer',
            'budget' => 'decimal:2',
            'spent' => 'decimal:2',
            'starts_at' => 'date',
            'due_at' => 'date',
            'completed_at' => 'datetime',
        ];
    }

    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    public function deal(): BelongsTo
    {
        return $this->belongsTo(Deal::class);
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function manager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'manager_id');
    }

    public function stages(): HasMany
    {
        return $this->hasMany(ProjectStage::class)->orderBy('sort_order');
    }

    public function tasks(): HasMany
    {
        return $this->hasMany(Task::class)->orderBy('sort_order')->orderBy('due_at');
    }

    public function members(): BelongsToMany
    {
        return $this->belongsToMany(User::class)
            ->withPivot(['role', 'joined_at'])
            ->withTimestamps();
    }

    public function recalculateProgress(): int
    {
        $total = $this->tasks()->count();

        if ($total === 0) {
            $progress = 0;
        } else {
            $done = $this->tasks()->where('status', 'done')->count();
            $progress = (int) round(($done / $total) * 100);
        }

        $this->updateQuietly([
            'progress' => $progress,
            'completed_at' => $progress === 100 ? ($this->completed_at ?? now()) : null,
        ]);

        return $progress;
    }
}
