<?php

namespace App\Http\Controllers;

use App\Models\Warehouse;
use App\Models\WarehouseAddress;
use App\Support\CrmModuleManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class WarehouseAddressController extends Controller
{
    public function store(Request $request, Warehouse $warehouse, CrmModuleManager $moduleManager): RedirectResponse
    {
        $this->authorize('update', $warehouse);

        $payload = $this->validatedData($request, $warehouse);
        $payload['warehouse_id'] = $warehouse->id;
        $payload['created_by'] = $request->user()->id;
        $payload = $moduleManager->applyPayloadHooks('warehouses.addresses.store', $payload, [
            'hook' => 'warehouses.addresses.store',
            'warehouse_id' => $warehouse->id,
            'user_id' => $request->user()->id,
        ], array_keys($payload));

        WarehouseAddress::query()->create($payload);

        return redirect()
            ->route('warehouses.show', $warehouse)
            ->with('success', __('Storage address has been added.'));
    }

    public function update(Request $request, Warehouse $warehouse, WarehouseAddress $warehouseAddress, CrmModuleManager $moduleManager): RedirectResponse
    {
        $this->authorize('update', $warehouse);
        $this->ensureAddressBelongsToWarehouse($warehouse, $warehouseAddress);

        $payload = $this->validatedData($request, $warehouse, $warehouseAddress);
        $payload = $moduleManager->applyPayloadHooks('warehouses.addresses.update', $payload, [
            'hook' => 'warehouses.addresses.update',
            'warehouse_id' => $warehouse->id,
            'address_id' => $warehouseAddress->id,
            'user_id' => $request->user()->id,
        ], array_keys($payload));

        $warehouseAddress->update($payload);

        return redirect()
            ->route('warehouses.show', $warehouse)
            ->with('success', __('Storage address has been updated.'));
    }

    public function destroy(Warehouse $warehouse, WarehouseAddress $warehouseAddress): RedirectResponse
    {
        $this->authorize('update', $warehouse);
        $this->ensureAddressBelongsToWarehouse($warehouse, $warehouseAddress);

        $warehouseAddress->delete();

        return redirect()
            ->route('warehouses.show', $warehouse)
            ->with('success', __('Storage address has been deleted.'));
    }

    public function generate(Request $request, Warehouse $warehouse): RedirectResponse
    {
        $this->authorize('update', $warehouse);

        $validated = $request->validate([
            'prefix' => ['nullable', 'string', 'max:20'],
            'zone' => ['nullable', 'string', 'max:80'],
            'color' => ['nullable', 'regex:/^#[0-9A-Fa-f]{6}$/'],
            'clear_existing' => ['nullable', 'boolean'],
        ]);

        $prefix = strtoupper(trim((string) ($validated['prefix'] ?? 'A')));
        if ($prefix === '') {
            $prefix = 'A';
        }

        $zone = $this->nullableString($validated['zone'] ?? null);
        $color = (string) ($validated['color'] ?? '#0EA5E9');
        $clearExisting = (bool) ($validated['clear_existing'] ?? false);
        $rows = max(1, (int) $warehouse->map_rows);
        $columns = max(1, (int) $warehouse->map_columns);
        $now = now();

        if ($clearExisting) {
            $warehouse->addresses()->delete();
        }

        $existing = $warehouse->addresses()->get(['id', 'code', 'x', 'y']);
        $usedCoordinates = [];
        $usedCodes = [];

        foreach ($existing as $address) {
            $usedCoordinates[$address->x.'-'.$address->y] = true;
            $usedCodes[(string) $address->code] = true;
        }

        $insertRows = [];
        for ($y = 1; $y <= $rows; $y++) {
            for ($x = 1; $x <= $columns; $x++) {
                $coordinateKey = $x.'-'.$y;
                if (isset($usedCoordinates[$coordinateKey])) {
                    continue;
                }

                $baseCode = sprintf('%s-%02d-%02d', $prefix, $y, $x);
                $code = $baseCode;
                $suffix = 1;
                while (isset($usedCodes[$code])) {
                    $code = $baseCode.'-'.$suffix;
                    $suffix++;
                }

                $usedCoordinates[$coordinateKey] = true;
                $usedCodes[$code] = true;

                $insertRows[] = [
                    'warehouse_id' => $warehouse->id,
                    'code' => $code,
                    'zone' => $zone,
                    'aisle' => (string) $y,
                    'rack' => (string) $x,
                    'shelf' => null,
                    'cell' => null,
                    'x' => $x,
                    'y' => $y,
                    'color' => $color,
                    'capacity' => 0,
                    'current_load' => 0,
                    'status' => 'free',
                    'note' => null,
                    'created_by' => $request->user()->id,
                    'created_at' => $now,
                    'updated_at' => $now,
                ];
            }
        }

        if ($insertRows !== []) {
            WarehouseAddress::query()->insert($insertRows);
        }

        return redirect()
            ->route('warehouses.show', $warehouse)
            ->with('success', __('Warehouse map has been generated. Added: :count addresses.', [
                'count' => count($insertRows),
            ]));
    }

    private function ensureAddressBelongsToWarehouse(Warehouse $warehouse, WarehouseAddress $warehouseAddress): void
    {
        abort_unless($warehouseAddress->warehouse_id === $warehouse->id, 404);
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedData(Request $request, Warehouse $warehouse, ?WarehouseAddress $warehouseAddress = null): array
    {
        $validated = $request->validate([
            'code' => [
                'required',
                'string',
                'max:120',
                Rule::unique('warehouse_addresses', 'code')
                    ->where(fn ($query) => $query->where('warehouse_id', $warehouse->id))
                    ->ignore($warehouseAddress?->id),
            ],
            'zone' => ['nullable', 'string', 'max:80'],
            'aisle' => ['nullable', 'string', 'max:80'],
            'rack' => ['nullable', 'string', 'max:80'],
            'shelf' => ['nullable', 'string', 'max:80'],
            'cell' => ['nullable', 'string', 'max:80'],
            'x' => ['required', 'integer', 'min:1', 'max:200'],
            'y' => ['required', 'integer', 'min:1', 'max:200'],
            'color' => ['nullable', 'regex:/^#[0-9A-Fa-f]{6}$/'],
            'capacity' => ['nullable', 'numeric', 'min:0'],
            'current_load' => ['nullable', 'numeric', 'min:0'],
            'status' => ['required', Rule::in(array_keys($this->statusOptions()))],
            'note' => ['nullable', 'string'],
        ]);

        $maxRows = max(1, (int) $warehouse->map_rows);
        $maxColumns = max(1, (int) $warehouse->map_columns);
        if ((int) $validated['x'] > $maxColumns) {
            throw ValidationException::withMessages([
                'x' => __('X coordinate is outside the configured warehouse map.'),
            ]);
        }

        if ((int) $validated['y'] > $maxRows) {
            throw ValidationException::withMessages([
                'y' => __('Y coordinate is outside the configured warehouse map.'),
            ]);
        }

        $occupiedCoordinate = WarehouseAddress::query()
            ->where('warehouse_id', $warehouse->id)
            ->where('x', (int) $validated['x'])
            ->where('y', (int) $validated['y'])
            ->when($warehouseAddress !== null, fn ($query) => $query->where('id', '!=', $warehouseAddress->id))
            ->exists();
        if ($occupiedCoordinate) {
            throw ValidationException::withMessages([
                'x' => __('Selected map cell is already used by another storage address.'),
            ]);
        }

        $validated['zone'] = $this->nullableString($validated['zone'] ?? null);
        $validated['aisle'] = $this->nullableString($validated['aisle'] ?? null);
        $validated['rack'] = $this->nullableString($validated['rack'] ?? null);
        $validated['shelf'] = $this->nullableString($validated['shelf'] ?? null);
        $validated['cell'] = $this->nullableString($validated['cell'] ?? null);
        $validated['color'] = (string) ($validated['color'] ?? '#0EA5E9');
        $validated['capacity'] = (float) ($validated['capacity'] ?? 0);
        $validated['current_load'] = (float) ($validated['current_load'] ?? 0);
        $validated['note'] = $this->nullableString($validated['note'] ?? null);
        if ($validated['capacity'] > 0 && $validated['current_load'] > $validated['capacity']) {
            throw ValidationException::withMessages([
                'current_load' => __('Current load cannot be greater than capacity.'),
            ]);
        }

        return $validated;
    }

    private function nullableString(mixed $value): ?string
    {
        if (! is_string($value)) {
            return null;
        }

        $value = trim($value);

        return $value === '' ? null : $value;
    }

    /**
     * @return array<string, string>
     */
    private function statusOptions(): array
    {
        return [
            'free' => __('Free'),
            'reserved' => __('Reserved'),
            'blocked' => __('Blocked'),
        ];
    }
}
