<?php

namespace App\Http\Controllers;

use App\Models\AccessGroup;
use App\Models\OrganizationCompany;
use App\Models\User;
use App\Support\AccessControl;
use App\Support\MailboxProvisioner;
use App\Support\MenuManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;

class ProfileAccessController extends Controller
{
    public function __construct(
        private readonly MenuManager $menuManager
    ) {
    }

    public function impersonate(Request $request, User $member): RedirectResponse
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        if ($actor->id === $member->id) {
            return Redirect::route('profile.edit', ['section' => 'users']);
        }

        Auth::login($member);
        $request->session()->regenerate();
        $request->session()->put('impersonator_id', $actor->id);
        $request->session()->put('impersonated_at', now()->toIso8601String());

        return Redirect::route('dashboard')->with('success', 'You are authorized as '.$member->name.'.');
    }

    public function leaveImpersonation(Request $request): RedirectResponse
    {
        $impersonatorId = (int) $request->session()->get('impersonator_id', 0);
        if ($impersonatorId <= 0) {
            return Redirect::route('dashboard');
        }

        /** @var User|null $impersonator */
        $impersonator = User::query()->find($impersonatorId);
        if (! $impersonator) {
            Auth::logout();

            $request->session()->invalidate();
            $request->session()->regenerateToken();

            return Redirect::route('login');
        }

        Auth::login($impersonator);
        $request->session()->regenerate();
        $request->session()->forget(['impersonator_id', 'impersonated_at']);

        return Redirect::route('profile.edit', ['section' => 'users'])->with('success', 'You are back in administrator account.');
    }

    public function update(Request $request): RedirectResponse
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        $validated = $request->validate([
            'user_id' => ['required', 'integer', 'exists:users,id'],
            'role' => ['required', 'string', Rule::in(array_keys(AccessControl::roles()))],
            'access_group_id' => ['nullable', 'integer', 'exists:access_groups,id'],
        ]);

        /** @var User $target */
        $target = User::query()->findOrFail((int) $validated['user_id']);
        $newRole = (string) $validated['role'];

        if ($target->id === $actor->id && $newRole !== AccessControl::ROLE_ADMIN) {
            $adminCount = User::query()->where('role', AccessControl::ROLE_ADMIN)->count();
            if ($adminCount <= 1) {
                return back()->withErrors([
                    'role' => 'You cannot remove the administrator role from a single system administrator.',
                ]);
            }
        }

        $permissions = $this->permissionsFromRequest($request);

        $target->fill([
            'role' => $newRole,
            'permissions' => $permissions,
            'access_group_id' => $validated['access_group_id'] ?? null,
        ]);
        $target->save();
        $this->menuManager->forgetUserCaches($target->id);

        return Redirect::route('profile.edit', ['section' => 'access'])->with('status', 'access-updated');
    }

    public function storeGroup(Request $request): RedirectResponse
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        $validated = $request->validate([
            'group_name' => ['required', 'string', 'max:120', 'unique:access_groups,name'],
        ]);

        AccessGroup::query()->create([
            'name' => trim((string) $validated['group_name']),
            'permissions' => $this->permissionsFromRequest($request),
        ]);

        return Redirect::route('profile.edit', ['section' => 'access'])->with('status', 'access-group-created');
    }

    public function updateGroup(Request $request, AccessGroup $group): RedirectResponse
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        $validated = $request->validate([
            'group_name' => ['required', 'string', 'max:120', Rule::unique('access_groups', 'name')->ignore($group->id)],
        ]);

        $group->fill([
            'name' => trim((string) $validated['group_name']),
            'permissions' => $this->permissionsFromRequest($request),
        ])->save();

        User::query()
            ->where('access_group_id', $group->id)
            ->pluck('id')
            ->each(fn ($userId) => $this->menuManager->forgetUserCaches((int) $userId));

        return Redirect::route('profile.edit', ['section' => 'access'])->with('status', 'access-group-updated');
    }

    public function destroyGroup(Request $request, AccessGroup $group): RedirectResponse
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        $affectedUserIds = User::query()
            ->where('access_group_id', $group->id)
            ->pluck('id')
            ->map(fn ($id): int => (int) $id)
            ->all();

        $group->delete();

        foreach ($affectedUserIds as $userId) {
            $this->menuManager->forgetUserCaches($userId);
        }

        return Redirect::route('profile.edit', ['section' => 'access'])->with('status', 'access-group-deleted');
    }

    public function bulkAssignGroup(Request $request): RedirectResponse
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        $validated = $request->validate([
            'access_group_id' => ['nullable', 'integer', 'exists:access_groups,id'],
            'selected_user_ids' => ['required', 'array', 'min:1'],
            'selected_user_ids.*' => ['required', 'integer', 'exists:users,id'],
        ]);

        $userIds = collect($validated['selected_user_ids'])
            ->map(fn ($id): int => (int) $id)
            ->unique()
            ->values()
            ->all();

        User::query()
            ->whereIn('id', $userIds)
            ->update([
                'access_group_id' => $validated['access_group_id'] ?? null,
            ]);

        foreach ($userIds as $userId) {
            $this->menuManager->forgetUserCaches($userId);
        }

        return Redirect::route('profile.edit', ['section' => 'access'])->with('status', 'access-group-assigned');
    }

    public function storeUser(Request $request, MailboxProvisioner $mailboxProvisioner): RedirectResponse
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:users,email'],
            'password' => ['required', 'confirmed', Password::defaults()],
            'phone' => ['nullable', 'string', 'max:50'],
            'job_title' => ['nullable', 'string', 'max:255'],
            'manager_id' => ['nullable', 'integer', 'exists:users,id'],
            'organization_company_id' => ['nullable', 'integer', 'exists:organization_companies,id'],
            'birth_date' => ['nullable', 'date', 'before_or_equal:today'],
            'role' => ['required', 'string', Rule::in(array_keys(AccessControl::roles()))],
            'access_group_id' => ['nullable', 'integer', 'exists:access_groups,id'],
            'locale' => ['nullable', Rule::in(['ru', 'en'])],
        ]);

        $permissions = $this->permissionsFromRequest($request);

        $hasChildCompanies = OrganizationCompany::query()
            ->whereNotNull('parent_id')
            ->exists();

        $organizationCompanyId = (int) ($validated['organization_company_id'] ?? 0);
        if (! $hasChildCompanies || $organizationCompanyId <= 0) {
            $organizationCompanyId = null;
        } elseif (! OrganizationCompany::query()->whereKey($organizationCompanyId)->whereNotNull('parent_id')->exists()) {
            $organizationCompanyId = null;
        }

        $user = User::query()->create([
            'name' => trim((string) $validated['name']),
            'email' => strtolower((string) $validated['email']),
            'password' => (string) $validated['password'],
            'phone' => $this->nullableText($validated['phone'] ?? null),
            'job_title' => $this->nullableText($validated['job_title'] ?? null),
            'manager_id' => $validated['manager_id'] ?? null,
            'organization_company_id' => $organizationCompanyId,
            'birth_date' => $validated['birth_date'] ?? null,
            'role' => (string) $validated['role'],
            'access_group_id' => $validated['access_group_id'] ?? null,
            'permissions' => $permissions,
            'locale' => $validated['locale'] ?? config('app.locale', 'en'),
        ]);

        if ($mailboxProvisioner->shouldAutoProvisionOnUserCreate()) {
            try {
                $mailboxProvisioner->ensurePrimaryMailbox($user, $actor);
            } catch (\Throwable $exception) {
                Log::warning('Auto mailbox provisioning on admin user create failed.', [
                    'user_id' => $user->id,
                    'actor_id' => $actor->id,
                    'message' => $exception->getMessage(),
                ]);
            }
        }

        $this->menuManager->forgetUserCaches($user->id);

        return Redirect::route('profile.edit', ['section' => 'users'])->with('status', 'user-created');
    }

    public function downloadUsersTemplate(Request $request): Response
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        $headers = [
            'name',
            'email',
            'password',
            'phone',
            'job_title',
            'manager_email',
            'company_path',
            'birth_date',
            'role',
            'access_group',
            'locale',
        ];

        foreach (array_keys(AccessControl::entities()) as $entity) {
            foreach (array_keys(AccessControl::actions()) as $action) {
                $headers[] = $entity.'.'.$action;
            }
        }

        $sampleRow = [
            'Alex Carter',
            'alex@example.test',
            'SecurePass123',
            '+1 555 100',
            'Sales manager',
            'supervisor@example.test',
            'Holding / Branch',
            '1991-05-12',
            'user',
            'Sales team',
            'en',
        ];

        foreach (array_keys(AccessControl::entities()) as $entity) {
            foreach (array_keys(AccessControl::actions()) as $action) {
                $sampleRow[] = $entity === 'tasks' ? '1' : '0';
            }
        }

        $handle = fopen('php://temp', 'r+');
        fputcsv($handle, $headers);
        fputcsv($handle, $sampleRow);
        rewind($handle);
        $csv = stream_get_contents($handle) ?: '';
        fclose($handle);

        return response($csv, 200, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="users-import-template.csv"',
        ]);
    }

    public function importUsers(Request $request, MailboxProvisioner $mailboxProvisioner): RedirectResponse
    {
        $actor = $request->user();
        abort_unless(AccessControl::canManageAccess($actor), 403);

        $validated = $request->validate([
            'import_file' => ['required', 'file', 'mimes:csv,txt', 'max:5120'],
        ]);

        /** @var \Illuminate\Http\UploadedFile $file */
        $file = $validated['import_file'];
        $handle = fopen($file->getRealPath(), 'r');
        if (! $handle) {
            return Redirect::back()->withErrors(['import' => __('Unable to read CSV file.')]);
        }

        $header = fgetcsv($handle);
        if (! is_array($header)) {
            fclose($handle);
            return Redirect::back()->withErrors(['import' => __('CSV is empty.')]);
        }

        $header = array_map(static fn ($item) => trim((string) $item), $header);
        $headerIndex = [];
        foreach ($header as $index => $label) {
            if ($label !== '') {
                $headerIndex[$label] = $index;
            }
        }

        foreach (['name', 'email', 'password', 'role'] as $required) {
            if (! array_key_exists($required, $headerIndex)) {
                fclose($handle);
                return Redirect::back()->withErrors(['import' => __('Missing required column: :column', ['column' => $required])]);
            }
        }

        $permissionColumns = [];
        foreach (array_keys(AccessControl::entities()) as $entity) {
            foreach (array_keys(AccessControl::actions()) as $action) {
                $column = $entity.'.'.$action;
                if (array_key_exists($column, $headerIndex)) {
                    $permissionColumns[$column] = [$entity, $action];
                }
            }
        }
        $hasPermissionColumns = $permissionColumns !== [];

        $companyMap = $this->buildCompanyMap();
        $childCompanyIds = $companyMap['child_ids'];
        $companyPathMap = $companyMap['path'];
        $hasChildCompanies = $childCompanyIds !== [];

        $groupMap = $this->buildAccessGroupMap();
        $managerMap = $this->buildManagerMap();

        $rows = [];
        $errors = [];
        $rowNumber = 1;
        $seenEmails = [];

        while (($data = fgetcsv($handle)) !== false) {
            $rowNumber++;
            if ($this->rowIsEmpty($data)) {
                continue;
            }

            $row = $this->rowToAssoc($data, $headerIndex);

            $validator = Validator::make($row, [
                'name' => ['required', 'string', 'max:255'],
                'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:users,email'],
                'password' => ['required', 'string', 'min:8'],
                'phone' => ['nullable', 'string', 'max:50'],
                'job_title' => ['nullable', 'string', 'max:255'],
                'birth_date' => ['nullable', 'date', 'before_or_equal:today'],
                'role' => ['required', 'string', Rule::in(array_keys(AccessControl::roles()))],
                'locale' => ['nullable', Rule::in(['ru', 'en'])],
            ]);

            if ($validator->fails()) {
                $errors[] = "Row {$rowNumber}: ".implode('; ', $validator->errors()->all());
                continue;
            }

            $email = strtolower((string) $row['email']);
            if (in_array($email, $seenEmails, true)) {
                $errors[] = "Row {$rowNumber}: duplicate email {$email} in import file.";
                continue;
            }
            $seenEmails[] = $email;

            $managerId = $this->resolveManagerId($row['manager_email'] ?? null, $managerMap, $errors, $rowNumber);
            $accessGroupId = $this->resolveAccessGroupId($row['access_group'] ?? null, $groupMap, $errors, $rowNumber);
            $organizationCompanyId = $this->resolveOrganizationCompanyId(
                $row,
                $hasChildCompanies,
                $companyPathMap,
                $childCompanyIds,
                $errors,
                $rowNumber
            );

            $permissions = [];
            if ($hasPermissionColumns) {
                $matrixInput = [];
                foreach ($permissionColumns as $column => [$entity, $action]) {
                    $matrixInput[$entity][$action] = $this->truthy($row[$column] ?? null);
                }
                $permissions = AccessControl::flattenPermissionMatrix($matrixInput);
            }

            $rows[] = [
                'name' => trim((string) $row['name']),
                'email' => $email,
                'password' => (string) $row['password'],
                'phone' => $this->nullableText($row['phone'] ?? null),
                'job_title' => $this->nullableText($row['job_title'] ?? null),
                'manager_id' => $managerId,
                'organization_company_id' => $organizationCompanyId,
                'birth_date' => ($row['birth_date'] ?? '') ?: null,
                'role' => (string) $row['role'],
                'access_group_id' => $accessGroupId,
                'permissions' => $permissions,
                'locale' => ($row['locale'] ?? '') ?: config('app.locale', 'en'),
            ];
        }

        fclose($handle);

        if ($errors !== []) {
            return Redirect::back()
                ->withErrors(['import' => __('Import errors detected.')])
                ->with('import_errors', $errors);
        }

        foreach ($rows as $payload) {
            $user = User::query()->create($payload);
            if ($mailboxProvisioner->shouldAutoProvisionOnUserCreate()) {
                try {
                    $mailboxProvisioner->ensurePrimaryMailbox($user, $actor);
                } catch (\Throwable $exception) {
                    Log::warning('Auto mailbox provisioning on users import failed.', [
                        'user_id' => $user->id,
                        'actor_id' => $actor->id,
                        'message' => $exception->getMessage(),
                    ]);
                }
            }
            $this->menuManager->forgetUserCaches($user->id);
        }

        return Redirect::route('profile.edit', ['section' => 'users'])->with('status', 'users-imported');
    }

    /**
     * @return array<string, bool>
     */
    private function permissionsFromRequest(Request $request): array
    {
        $rawPermissions = $request->input('permissions', []);

        return AccessControl::flattenPermissionMatrix(is_array($rawPermissions) ? $rawPermissions : []);
    }

    private function nullableText(mixed $value): ?string
    {
        if (! is_string($value)) {
            return null;
        }

        $trimmed = trim($value);

        return $trimmed === '' ? null : $trimmed;
    }

    private function rowIsEmpty(array $row): bool
    {
        foreach ($row as $value) {
            if (trim((string) $value) !== '') {
                return false;
            }
        }

        return true;
    }

    /**
     * @param  array<string, int>  $headerIndex
     * @return array<string, string>
     */
    private function rowToAssoc(array $row, array $headerIndex): array
    {
        $assoc = [];
        foreach ($headerIndex as $label => $index) {
            $assoc[$label] = isset($row[$index]) ? trim((string) $row[$index]) : '';
        }

        return $assoc;
    }

    private function truthy(mixed $value): bool
    {
        $raw = strtolower(trim((string) $value));

        return in_array($raw, ['1', 'true', 'yes', 'y', 'on'], true);
    }

    /**
     * @return array<string, int>
     */
    private function buildAccessGroupMap(): array
    {
        return AccessGroup::query()
            ->orderBy('name')
            ->get(['id', 'name'])
            ->mapWithKeys(fn ($group) => [strtolower((string) $group->name) => (int) $group->id])
            ->all();
    }

    /**
     * @return array<string, int>
     */
    private function buildManagerMap(): array
    {
        return User::query()
            ->orderBy('email')
            ->get(['id', 'email'])
            ->mapWithKeys(fn ($user) => [strtolower((string) $user->email) => (int) $user->id])
            ->all();
    }

    /**
     * @return array{path: array<string, int>, child_ids: array<int, int>}
     */
    private function buildCompanyMap(): array
    {
        $companies = OrganizationCompany::query()
            ->with('parent:id,name')
            ->get(['id', 'name', 'parent_id']);

        $pathMap = [];
        $childIds = [];

        foreach ($companies as $company) {
            if ($company->parent_id) {
                $parentName = $company->parent?->name ?? '';
                $path = trim($parentName.' / '.$company->name);
                if ($path !== '') {
                    $pathMap[strtolower($path)] = $company->id;
                }
                $childIds[$company->id] = $company->id;
            }
        }

        return [
            'path' => $pathMap,
            'child_ids' => $childIds,
        ];
    }

    /**
     * @param  array<string, int>  $managerMap
     * @param  array<int, string>  $errors
     */
    private function resolveManagerId(?string $email, array $managerMap, array &$errors, int $rowNumber): ?int
    {
        $email = strtolower(trim((string) $email));
        if ($email === '') {
            return null;
        }

        if (! array_key_exists($email, $managerMap)) {
            $errors[] = "Row {$rowNumber}: manager_email {$email} not found.";
            return null;
        }

        return $managerMap[$email];
    }

    /**
     * @param  array<string, int>  $groupMap
     * @param  array<int, string>  $errors
     */
    private function resolveAccessGroupId(?string $name, array $groupMap, array &$errors, int $rowNumber): ?int
    {
        $name = strtolower(trim((string) $name));
        if ($name === '') {
            return null;
        }

        if (! array_key_exists($name, $groupMap)) {
            $errors[] = "Row {$rowNumber}: access_group {$name} not found.";
            return null;
        }

        return $groupMap[$name];
    }

    /**
     * @param  array<string, string>  $row
     * @param  array<string, int>  $companyPathMap
     * @param  array<int, int>  $childCompanyIds
     * @param  array<int, string>  $errors
     */
    private function resolveOrganizationCompanyId(array $row, bool $hasChildCompanies, array $companyPathMap, array $childCompanyIds, array &$errors, int $rowNumber): ?int
    {
        if (! $hasChildCompanies || $childCompanyIds === []) {
            return null;
        }

        $path = strtolower(trim((string) ($row['company_path'] ?? '')));
        if ($path === '') {
            return null;
        }

        $resolved = $companyPathMap[$path] ?? null;
        if (! $resolved) {
            $errors[] = "Row {$rowNumber}: company_path {$path} not found.";
            return null;
        }

        return array_key_exists($resolved, $childCompanyIds) ? $resolved : null;
    }
}
