<?php

namespace App\Http\Controllers;

use App\Models\CrmModule;
use App\Support\CrmModuleManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use RuntimeException;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class ModuleController extends Controller
{
    public function scaffold(Request $request, CrmModuleManager $moduleManager): BinaryFileResponse
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:120'],
            'slug' => ['required', 'string', 'max:120'],
            'version' => ['nullable', 'string', 'max:60'],
            'description' => ['nullable', 'string', 'max:1000'],
        ]);

        $archivePath = $moduleManager->createScaffoldArchive($validated);
        $downloadName = trim((string) $validated['slug']) !== ''
            ? trim((string) $validated['slug']).'-module-scaffold.zip'
            : 'crm-module-scaffold.zip';

        return response()
            ->download($archivePath, $downloadName, ['Content-Type' => 'application/zip'])
            ->deleteFileAfterSend(true);
    }

    public function store(Request $request, CrmModuleManager $moduleManager): RedirectResponse
    {
        $validated = $request->validate([
            'archive' => ['required', 'file', 'mimes:zip', 'max:20480'],
        ]);

        try {
            $module = $moduleManager->installFromArchive($validated['archive'], $request->user());
        } catch (RuntimeException $exception) {
            return $this->redirectToModuleSection()
                ->withInput()
                ->withErrors(['archive' => $exception->getMessage()]);
        }

        return $this->redirectToModuleSection()
            ->with('success', sprintf('Module "%s" has been installed.', $module->name));
    }

    public function update(Request $request, CrmModule $module, CrmModuleManager $moduleManager): RedirectResponse
    {
        $validated = $request->validate([
            'is_enabled' => ['required', 'boolean'],
        ]);

        $moduleManager->setEnabled($module, (bool) $validated['is_enabled']);

        return $this->redirectToModuleSection()
            ->with('success', $module->is_enabled ? 'Module enabled.' : 'Module disabled.');
    }

    public function destroy(CrmModule $module, CrmModuleManager $moduleManager): RedirectResponse
    {
        $name = $module->name;
        $moduleManager->delete($module);

        return $this->redirectToModuleSection()
            ->with('success', sprintf('Module "%s" has been deleted.', $name));
    }

    private function redirectToModuleSection(): RedirectResponse
    {
        return redirect()->route('profile.edit', ['section' => 'modules']);
    }
}
