<?php

namespace App\Http\Controllers;

use App\Models\DealStage;
use App\Models\Pipeline;
use App\Models\User;
use App\Models\WebForm;
use App\Support\AccessControl;
use App\Support\CrmModuleManager;
use App\Support\FormStyleManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;

class FormController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(WebForm::class, 'form');
    }

    public function index(Request $request): View
    {
        /** @var User $user */
        $user = $request->user();

        $formsQuery = WebForm::query()
            ->with('owner:id,name,email')
            ->withCount('submissions')
            ->latest('id');

        if (! AccessControl::isElevated($user)) {
            $formsQuery->where('owner_id', $user->id);
        }

        $forms = $formsQuery->paginate(20)->withQueryString();

        return view('forms.index', [
            'forms' => $forms,
        ]);
    }

    public function create(): View
    {
        return view('forms.create', [
            ...$this->formViewData(),
            'form' => new WebForm([
                'is_active' => true,
                'submit_label' => 'Send',
                'show_reset_button' => false,
                'reset_label' => 'Clear',
                'success_message' => 'Thank you! The form has been submitted.',
                'behavior' => 'create_task',
                'behavior_settings' => [
                    'submit_result_action' => 'reload',
                ],
                'fields' => $this->defaultFields(),
                'style_settings' => FormStyleManager::defaults(),
            ]),
        ]);
    }

    public function store(Request $request, CrmModuleManager $moduleManager): RedirectResponse
    {
        $payload = $this->validatedPayload($request);
        $payload['owner_id'] = $request->user()->id;
        $payload = $moduleManager->applyPayloadHooks('forms.store', $payload, [
            'hook' => 'forms.store',
            'user_id' => $request->user()->id,
        ], array_keys($payload));

        $form = WebForm::query()->create($payload);

        return redirect()
            ->route('forms.edit', $form)
            ->with('success', 'Form created.');
    }

    public function edit(WebForm $form): View
    {
        $form->loadCount('submissions');
        $recentSubmissions = $form->submissions()
            ->latest('submitted_at')
            ->limit(10)
            ->get();

        return view('forms.edit', [
            ...$this->formViewData(),
            'form' => $form,
            'recentSubmissions' => $recentSubmissions,
        ]);
    }

    public function update(Request $request, WebForm $form, CrmModuleManager $moduleManager): RedirectResponse
    {
        $payload = $this->validatedPayload($request, $form);
        $payload = $moduleManager->applyPayloadHooks('forms.update', $payload, [
            'hook' => 'forms.update',
            'user_id' => $request->user()->id,
            'form_id' => $form->id,
        ], array_keys($payload));

        $form->update($payload);

        return redirect()
            ->route('forms.edit', $form)
            ->with('success', 'Form updated.');
    }

    public function destroy(WebForm $form): RedirectResponse
    {
        $form->delete();

        return redirect()
            ->route('forms.index')
            ->with('success', 'Form deleted.');
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedPayload(Request $request, ?WebForm $form = null): array
    {
        $validator = \validator($request->all(), [
            'name' => ['required', 'string', 'max:255'],
            'slug' => [
                'nullable',
                'string',
                'max:120',
                'regex:/^[a-z0-9]+(?:-[a-z0-9]+)*$/',
                Rule::unique('web_forms', 'slug')->ignore($form?->id),
            ],
            'description' => ['nullable', 'string'],
            'submit_label' => ['required', 'string', 'max:80'],
            'show_reset_button' => ['nullable', 'boolean'],
            'reset_label' => ['nullable', 'string', 'max:80'],
            'success_message' => ['required', 'string', 'max:255'],
            'is_active' => ['nullable', 'boolean'],
            'behavior' => ['required', Rule::in(array_keys($this->behaviorOptions()))],
            'fields' => ['required', 'array', 'min:1'],
            'fields.*.label' => ['required', 'string', 'max:80'],
            'fields.*.name' => ['required', 'string', 'max:80', 'regex:/^[a-z][a-z0-9_]*$/', 'distinct'],
            'fields.*.type' => ['required', Rule::in(array_keys($this->fieldTypeOptions()))],
            'fields.*.required' => ['nullable', 'boolean'],
            'fields.*.placeholder' => ['nullable', 'string', 'max:255'],
            'fields.*.target' => ['nullable', 'string', 'max:80'],
            'fields.*.options' => ['nullable', 'string'],
        ]);

        $validated = $validator->validate();
        $behavior = (string) $validated['behavior'];

        $normalizedFields = collect($validated['fields'])
            ->map(fn (array $field): array => $this->normalizeField($field))
            ->values()
            ->all();
        $fieldNames = collect($normalizedFields)
            ->pluck('name')
            ->filter(fn (mixed $value): bool => is_string($value) && $value !== '')
            ->values()
            ->all();
        $behaviorSettings = $this->validatedBehaviorSettings($request, $behavior, $fieldNames);
        $styleSettings = FormStyleManager::normalize($request->input('style_settings', []));

        $slug = trim((string) ($validated['slug'] ?? ''));
        if ($slug === '') {
            $slug = $this->generateUniqueSlug((string) $validated['name'], $form?->id);
        }

        return [
            'name' => (string) $validated['name'],
            'slug' => $slug,
            'description' => Arr::get($validated, 'description'),
            'submit_label' => (string) $validated['submit_label'],
            'show_reset_button' => $request->boolean('show_reset_button'),
            'reset_label' => trim((string) ($validated['reset_label'] ?? '')) !== ''
                ? (string) $validated['reset_label']
                : 'Clear',
            'success_message' => (string) $validated['success_message'],
            'is_active' => $request->boolean('is_active', true),
            'behavior' => $behavior,
            'behavior_settings' => $behaviorSettings,
            'fields' => $normalizedFields,
            'style_settings' => $styleSettings,
        ];
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedBehaviorSettings(Request $request, string $behavior, array $availableFieldNames): array
    {
        $input = $request->input('behavior_settings', []);
        if (! is_array($input)) {
            $input = [];
        }

        $settings = $this->validatedSubmitResultSettings($input);

        if ($behavior === 'create_task') {
            $settingsValidator = \validator(['behavior_settings' => $input], [
                'behavior_settings.assignee_id' => ['nullable', 'integer', 'exists:users,id'],
                'behavior_settings.priority' => ['nullable', Rule::in(['low', 'medium', 'high', 'urgent'])],
            ]);

            $validated = $settingsValidator->validate();
            $settings = array_merge($settings, (array) ($validated['behavior_settings'] ?? []));
            $settings['priority'] = (string) ($settings['priority'] ?? 'medium');
        }

        if ($behavior === 'create_company') {
            $settingsValidator = \validator(['behavior_settings' => $input], [
                'behavior_settings.owner_id' => ['nullable', 'integer', 'exists:users,id'],
                'behavior_settings.status' => ['nullable', Rule::in(['lead', 'client', 'partner', 'inactive'])],
            ]);

            $validated = $settingsValidator->validate();
            $settings = array_merge($settings, (array) ($validated['behavior_settings'] ?? []));
            $settings['status'] = (string) ($settings['status'] ?? 'lead');
        }

        if ($behavior === 'create_contact') {
            $settingsValidator = \validator(['behavior_settings' => $input], [
                'behavior_settings.owner_id' => ['nullable', 'integer', 'exists:users,id'],
            ]);

            $validated = $settingsValidator->validate();
            $settings = array_merge($settings, (array) ($validated['behavior_settings'] ?? []));
        }

        if ($behavior === 'create_deal') {
            $settingsValidator = \validator(['behavior_settings' => $input], [
                'behavior_settings.owner_id' => ['nullable', 'integer', 'exists:users,id'],
                'behavior_settings.pipeline_id' => ['nullable', 'integer', 'exists:pipelines,id'],
                'behavior_settings.stage_id' => ['nullable', 'integer', 'exists:deal_stages,id'],
                'behavior_settings.currency' => ['nullable', 'string', 'size:3'],
            ]);

            $validated = $settingsValidator->validate();
            $settings = array_merge($settings, (array) ($validated['behavior_settings'] ?? []));
            $settings['currency'] = strtoupper((string) ($settings['currency'] ?? 'USD'));

            $pipelineId = (int) ($settings['pipeline_id'] ?? 0);
            $stageId = (int) ($settings['stage_id'] ?? 0);
            if ($pipelineId > 0 && $stageId > 0) {
                $stage = DealStage::query()->find($stageId);
                if (! $stage || (int) $stage->pipeline_id !== $pipelineId) {
                    throw ValidationException::withMessages([
                        'behavior_settings.stage_id' => 'The selected stage does not belong to the selected funnel.',
                    ]);
                }
            }
        }

        $settings['field_map'] = $this->normalizeBehaviorFieldMap(
            $this->extractFieldMapInput($input, $behavior),
            $behavior,
            $availableFieldNames
        );

        return $settings;
    }

    /**
     * @param  array<string, mixed>  $input
     * @return array<string, mixed>
     */
    private function validatedSubmitResultSettings(array $input): array
    {
        $validator = \validator(['behavior_settings' => $input], [
            'behavior_settings.submit_result_action' => ['nullable', Rule::in(['reload', 'hide_form', 'redirect'])],
            'behavior_settings.redirect_url' => ['nullable', 'string', 'max:2000'],
        ]);

        $validated = $validator->validate();
        $settings = (array) ($validated['behavior_settings'] ?? []);

        $submitResultAction = (string) ($settings['submit_result_action'] ?? 'reload');
        if (! in_array($submitResultAction, ['reload', 'hide_form', 'redirect'], true)) {
            $submitResultAction = 'reload';
        }

        $normalized = [
            'submit_result_action' => $submitResultAction,
        ];

        $redirectUrl = trim((string) ($settings['redirect_url'] ?? ''));
        if ($submitResultAction === 'redirect') {
            if ($redirectUrl === '') {
                throw ValidationException::withMessages([
                    'behavior_settings.redirect_url' => __('Redirect URL is required for redirect mode.'),
                ]);
            }

            if (! $this->isAllowedRedirectUrl($redirectUrl)) {
                throw ValidationException::withMessages([
                    'behavior_settings.redirect_url' => __('Redirect URL must start with "/" or "http(s)://".'),
                ]);
            }

            $normalized['redirect_url'] = $redirectUrl;
        }

        return $normalized;
    }

    private function isAllowedRedirectUrl(string $url): bool
    {
        if (str_starts_with($url, '/')) {
            return true;
        }

        if (! preg_match('#^https?://#i', $url)) {
            return false;
        }

        return filter_var($url, FILTER_VALIDATE_URL) !== false;
    }

    /**
     * @param  array<string, mixed>  $settingsInput
     * @return array<int, array<string, mixed>>
     */
    private function extractFieldMapInput(array $settingsInput, string $behavior): array
    {
        $raw = $settingsInput['field_map'] ?? [];
        if (! is_array($raw)) {
            return [];
        }

        if (array_is_list($raw)) {
            return $raw;
        }

        $behaviorRows = $raw[$behavior] ?? [];

        return is_array($behaviorRows) ? $behaviorRows : [];
    }

    /**
     * @param  array<int, array<string, mixed>>  $rows
     * @param  list<string>  $availableFieldNames
     * @return list<array{crm_field:string, form_field:string}>
     */
    private function normalizeBehaviorFieldMap(array $rows, string $behavior, array $availableFieldNames): array
    {
        $allowedTargets = array_keys($this->behaviorFieldMapOptions()[$behavior] ?? []);
        if ($allowedTargets === []) {
            return [];
        }

        $validator = \validator(['rows' => $rows], [
            'rows' => ['nullable', 'array', 'max:40'],
            'rows.*.crm_field' => ['nullable', 'string', 'max:80', Rule::in($allowedTargets)],
            'rows.*.form_field' => ['nullable', 'string', 'max:80', 'regex:/^[a-z][a-z0-9_]*$/'],
        ]);
        $validated = $validator->validate();

        $availableLookup = array_flip($availableFieldNames);
        $normalized = collect($validated['rows'] ?? [])
            ->filter(fn (mixed $row): bool => is_array($row))
            ->map(function (array $row): array {
                return [
                    'crm_field' => trim((string) ($row['crm_field'] ?? '')),
                    'form_field' => Str::snake(trim((string) ($row['form_field'] ?? ''))),
                ];
            })
            ->filter(fn (array $row): bool => $row['crm_field'] !== '' && $row['form_field'] !== '')
            ->unique('crm_field')
            ->values();

        $unknownFormField = $normalized
            ->pluck('form_field')
            ->first(fn (string $fieldName): bool => ! isset($availableLookup[$fieldName]));
        if (is_string($unknownFormField) && $unknownFormField !== '') {
            throw ValidationException::withMessages([
                'behavior_settings.field_map' => 'The mapped form field "'.$unknownFormField.'" does not exist in this form.',
            ]);
        }

        return $normalized->all();
    }

    /**
     * @param  array<string, mixed>  $field
     * @return array<string, mixed>
     */
    private function normalizeField(array $field): array
    {
        $name = Str::snake((string) ($field['name'] ?? ''));
        $type = (string) ($field['type'] ?? 'text');

        $options = [];
        if ($type === 'select') {
            $raw = (string) ($field['options'] ?? '');
            $options = collect(preg_split('/\r\n|\r|\n/', $raw) ?: [])
                ->map(fn (string $option): string => trim($option))
                ->filter()
                ->values()
                ->all();
        }

        return [
            'label' => trim((string) ($field['label'] ?? '')),
            'name' => $name,
            'type' => $type,
            'required' => (bool) ($field['required'] ?? false),
            'placeholder' => trim((string) ($field['placeholder'] ?? '')),
            'target' => trim((string) ($field['target'] ?? '')),
            'options' => $options,
        ];
    }

    private function generateUniqueSlug(string $name, ?int $ignoreId = null): string
    {
        $base = Str::slug($name);
        if ($base === '') {
            $base = 'form';
        }

        $slug = $base;
        $index = 2;

        while (WebForm::query()
            ->when($ignoreId, fn ($query) => $query->where('id', '!=', $ignoreId))
            ->where('slug', $slug)
            ->exists()) {
            $slug = $base.'-'.$index;
            $index++;
        }

        return $slug;
    }

    /**
     * @return array<string, string>
     */
    private function behaviorOptions(): array
    {
        return [
            'create_task' => 'Create task',
            'create_deal' => 'Create lead/deal',
            'create_company' => 'Create company',
            'create_contact' => 'Create contact',
        ];
    }

    /**
     * @return array<string, string>
     */
    private function fieldTypeOptions(): array
    {
        return [
            'text' => 'Text',
            'email' => 'Email',
            'phone' => 'Telephone',
            'textarea' => 'Multiline text',
            'number' => 'Number',
            'date' => 'Date',
            'datetime' => 'Date and time',
            'select' => 'List',
            'checkbox' => 'Checkbox',
        ];
    }

    /**
     * @return list<array<string, mixed>>
     */
    private function defaultFields(): array
    {
        return [
            [
                'label' => 'Name',
                'name' => 'name',
                'type' => 'text',
                'required' => true,
                'placeholder' => '',
                'target' => 'name',
                'options' => [],
            ],
        ];
    }

    /**
     * @return array<string, mixed>
     */
    private function formViewData(): array
    {
        $users = User::query()->orderBy('name')->get(['id', 'name']);
        $pipelines = Pipeline::query()
            ->with(['stages' => fn ($query) => $query->orderBy('sort_order')])
            ->orderBy('sort_order')
            ->get(['id', 'name', 'is_default']);

        $targetOptions = [
            '' => 'Only store value',
            'title' => 'Title',
            'description' => 'Description',
            'name' => 'Name',
            'first_name' => 'First name',
            'last_name' => 'Last name',
            'email' => 'Email',
            'phone' => 'Phone',
            'company_name' => 'Company name',
            'company_id' => 'Company ID',
            'contact_id' => 'Contact ID',
            'deal_id' => 'Deal ID',
            'industry' => 'Industry',
            'website' => 'Website',
            'address' => 'Address',
            'source' => 'Source',
            'notes' => 'Notes',
            'amount' => 'Amount',
            'currency' => 'Currency',
            'due_at' => 'Deadline (due_at)',
        ];

        return [
            'behaviorOptions' => $this->behaviorOptions(),
            'behaviorFieldMapOptions' => $this->behaviorFieldMapOptions(),
            'fieldTypeOptions' => $this->fieldTypeOptions(),
            'targetOptions' => $targetOptions,
            'users' => $users,
            'pipelines' => $pipelines,
        ];
    }

    /**
     * @return array<string, array<string, string>>
     */
    private function behaviorFieldMapOptions(): array
    {
        return [
            'create_task' => [
                'title' => 'Task: title',
                'description' => 'Task: description',
                'assignee_id' => 'Task: assignee ID',
                'priority' => 'Task: priority',
                'status' => 'Task: status',
                'deal_id' => 'Task: deal ID',
                'company_id' => 'Task: company ID',
                'contact_id' => 'Task: contact ID',
                'estimated_hours' => 'Task: estimated hours',
                'starts_at' => 'Task: start date/time',
                'due_at' => 'Task: due date/time',
            ],
            'create_deal' => [
                'title' => 'Deal: title',
                'description' => 'Deal: description',
                'owner_id' => 'Deal: owner ID',
                'pipeline_id' => 'Deal: funnel ID',
                'stage_id' => 'Deal: stage ID',
                'company_id' => 'Deal: company ID',
                'contact_id' => 'Deal: contact ID',
                'amount' => 'Deal: amount',
                'currency' => 'Deal: currency',
                'priority' => 'Deal: priority',
                'status' => 'Deal: status',
                'source' => 'Deal: source',
                'expected_close_at' => 'Deal: expected close date',
            ],
            'create_company' => [
                'name' => 'Company: name',
                'owner_id' => 'Company: owner ID',
                'status' => 'Company: status',
                'industry' => 'Company: industry',
                'website' => 'Company: website',
                'phone' => 'Company: phone',
                'email' => 'Company: email',
                'address' => 'Company: address',
                'source' => 'Company: source',
                'notes' => 'Company: notes',
            ],
            'create_contact' => [
                'first_name' => 'Contact: first name',
                'last_name' => 'Contact: last name',
                'title' => 'Contact: job title',
                'email' => 'Contact: email',
                'phone' => 'Contact: phone',
                'company_id' => 'Contact: company ID',
                'owner_id' => 'Contact: owner ID',
                'source' => 'Contact: source',
                'notes' => 'Contact: notes',
            ],
        ];
    }
}
