<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\MailServiceSettingResource;
use App\Models\MailServiceSetting;
use App\Support\AccessControl;
use App\Support\MailboxProvisioner;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class MailSettingsController extends Controller
{
    public function show(Request $request, MailboxProvisioner $mailboxProvisioner): MailServiceSettingResource
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'mail', 'read'), 403);

        $settings = $mailboxProvisioner->settingsOrDefault();
        if (! $settings->webhook_secret) {
            $settings->webhook_secret = Str::random(40);
        }

        return MailServiceSettingResource::make($settings);
    }

    public function update(Request $request): MailServiceSettingResource
    {
        $user = $request->user();
        abort_unless(AccessControl::allows($user, 'mail', 'update'), 403);

        $validated = $request->validate([
            'provider' => ['required', 'string', 'max:60'],
            'is_active' => ['nullable', 'boolean'],
            'domain' => ['required', 'string', 'max:120'],
            'api_base_url' => ['nullable', 'string', 'max:255'],
            'api_key' => ['nullable', 'string', 'max:255'],
            'api_secret' => ['nullable', 'string', 'max:255'],
            'account_id' => ['nullable', 'string', 'max:255'],
            'webhook_secret' => ['nullable', 'string', 'max:255'],
            'auto_provision_on_registration' => ['nullable', 'boolean'],
            'auto_provision_on_user_create' => ['nullable', 'boolean'],
            'default_status' => ['required', Rule::in(['active', 'suspended', 'disabled'])],
            'default_quota_mb' => ['required', 'integer', 'min:1', 'max:102400'],
        ]);

        $payload = [
            'provider' => trim((string) $validated['provider']),
            'is_active' => $request->boolean('is_active'),
            'domain' => $this->normalizeDomain((string) $validated['domain']),
            'api_base_url' => $this->nullableText($validated['api_base_url'] ?? null),
            'api_key' => $this->nullableText($validated['api_key'] ?? null),
            'api_secret' => $this->nullableText($validated['api_secret'] ?? null),
            'account_id' => $this->nullableText($validated['account_id'] ?? null),
            'webhook_secret' => $this->nullableText($validated['webhook_secret'] ?? null),
            'auto_provision_on_registration' => $request->boolean('auto_provision_on_registration', true),
            'auto_provision_on_user_create' => $request->boolean('auto_provision_on_user_create', true),
            'default_status' => (string) $validated['default_status'],
            'default_quota_mb' => (int) $validated['default_quota_mb'],
        ];

        if (($payload['webhook_secret'] ?? '') === '') {
            $payload['webhook_secret'] = MailServiceSetting::query()->value('webhook_secret') ?: Str::random(40);
        }

        $settings = MailServiceSetting::query()->first() ?? new MailServiceSetting();
        $settings->fill($payload)->save();

        return MailServiceSettingResource::make($settings);
    }

    private function normalizeDomain(string $value): string
    {
        $value = strtolower(trim($value));
        $value = ltrim($value, '@');
        if (str_contains($value, '://')) {
            $host = parse_url($value, PHP_URL_HOST);
            $value = is_string($host) ? $host : '';
        }

        $value = preg_replace('/[^a-z0-9.-]+/', '', $value) ?: '';
        $value = trim($value, '.-');

        if ($value === '') {
            $value = (string) parse_url((string) config('app.url'), PHP_URL_HOST);
            $value = trim(strtolower($value));
        }

        return $value !== '' ? $value : 'crm.local';
    }

    private function nullableText(mixed $value): ?string
    {
        $value = trim((string) $value);

        return $value === '' ? null : $value;
    }
}

