<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\ContactResource;
use App\Models\Contact;
use App\Support\CrmModuleManager;
use Illuminate\Http\UploadedFile;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Storage;

class ContactController extends Controller
{
    public function __construct()
    {
        $this->authorizeResource(Contact::class, 'contact');
    }

    public function index(Request $request)
    {
        $search = trim((string) $request->input('q', ''));

        $contacts = Contact::query()
            ->with(['company', 'owner'])
            ->when($search !== '', function ($query) use ($search): void {
                $query->where(function ($sub) use ($search): void {
                    $sub->where('first_name', 'like', "%{$search}%")
                        ->orWhere('last_name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%");
                });
            })
            ->latest()
            ->paginate(20)
            ->withQueryString();

        return ContactResource::collection($contacts);
    }

    public function store(Request $request, CrmModuleManager $moduleManager): ContactResource
    {
        $validated = $this->validatedData($request);
        $validated = $moduleManager->applyPayloadHooks('contacts.store', $validated, [
            'hook' => 'contacts.store',
            'user_id' => $request->user()->id,
            'source' => 'api',
        ], array_keys($validated));
        $avatar = $request->file('avatar');
        $contact = Contact::create(Arr::except($validated, ['avatar', 'remove_avatar']));
        if ($avatar instanceof UploadedFile) {
            $contact->avatar_path = $avatar->store('contact-avatars', 'public');
            $contact->save();
        }

        return ContactResource::make($contact->load(['company', 'owner']));
    }

    public function show(Contact $contact): ContactResource
    {
        return ContactResource::make($contact->load(['company', 'owner']));
    }

    public function update(Request $request, Contact $contact, CrmModuleManager $moduleManager): ContactResource
    {
        $validated = $this->validatedData($request);
        $validated = $moduleManager->applyPayloadHooks('contacts.update', $validated, [
            'hook' => 'contacts.update',
            'user_id' => $request->user()->id,
            'contact_id' => $contact->id,
            'source' => 'api',
        ], array_keys($validated));
        $avatar = $request->file('avatar');
        $removeAvatar = $request->boolean('remove_avatar');

        $contact->update(Arr::except($validated, ['avatar', 'remove_avatar']));
        $this->syncAvatar($contact, $avatar, $removeAvatar);

        return ContactResource::make($contact->load(['company', 'owner']));
    }

    public function destroy(Contact $contact)
    {
        $this->deleteAvatarIfExists($contact->avatar_path);
        $contact->delete();

        return response()->noContent();
    }

    /**
     * @return array<string, mixed>
     */
    private function validatedData(Request $request): array
    {
        return $request->validate([
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['nullable', 'string', 'max:255'],
            'title' => ['nullable', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
            'phone' => ['nullable', 'string', 'max:50'],
            'company_id' => ['nullable', 'exists:companies,id'],
            'owner_id' => ['nullable', 'exists:users,id'],
            'source' => ['nullable', 'string', 'max:255'],
            'last_contacted_at' => ['nullable', 'date'],
            'notes' => ['nullable', 'string'],
            'avatar' => ['nullable', 'image', 'max:10240'],
            'remove_avatar' => ['nullable', 'boolean'],
        ]);
    }

    private function syncAvatar(Contact $contact, ?UploadedFile $avatar, bool $removeAvatar): void
    {
        $oldPath = $contact->avatar_path;

        if ($removeAvatar) {
            $contact->avatar_path = null;
        }

        if ($avatar instanceof UploadedFile) {
            $contact->avatar_path = $avatar->store('contact-avatars', 'public');
        }

        $contact->save();

        if (
            is_string($oldPath)
            && $oldPath !== ''
            && $oldPath !== $contact->avatar_path
        ) {
            $this->deleteAvatarIfExists($oldPath);
        }
    }

    private function deleteAvatarIfExists(?string $path): void
    {
        if (! is_string($path) || trim($path) === '') {
            return;
        }

        Storage::disk('public')->delete($path);
    }
}
