<?php

namespace App\Http\Controllers;

use App\Models\Disk;
use App\Models\DiskFolder;
use App\Models\User;
use App\Support\AccessControl;
use App\Support\DiskFileManager;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class DiskExplorerController extends Controller
{
    public function createFolder(Request $request, DiskFileManager $diskFileManager): JsonResponse
    {
        $this->authorize('create', Disk::class);

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:120'],
            'parent_folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
            'owner_id' => ['nullable', 'integer', 'exists:users,id'],
        ]);

        $actor = $request->user();
        $ownerId = $this->resolveOwnerIdForFolderOperation($actor, (int) ($validated['owner_id'] ?? 0), true);
        $owner = $ownerId === (int) $actor->id
            ? $actor
            : User::query()->findOrFail($ownerId);

        $name = $this->normalizeFolderName((string) ($validated['name'] ?? ''));
        if ($name === '') {
            throw ValidationException::withMessages([
                'name' => __('Catalog name is required.'),
            ]);
        }

        $parentFolder = $diskFileManager->normalizeFolderPath((string) ($validated['parent_folder'] ?? ''));
        $path = $diskFileManager->normalizeFolderPath($parentFolder !== '' ? $parentFolder.'/'.$name : $name);
        if ($path === '') {
            throw ValidationException::withMessages([
                'name' => __('Catalog path is invalid.'),
            ]);
        }

        $folder = DiskFolder::query()->updateOrCreate(
            [
                'owner_id' => $owner->id,
                'path' => $path,
            ],
            [
                'name' => $name,
                'parent_path' => $parentFolder !== '' ? $parentFolder : null,
            ]
        );

        Storage::disk('public')->makeDirectory('disk/'.$path);

        return response()->json([
            'message' => __('Catalog created.'),
            'folder' => [
                'id' => (int) $folder->id,
                'path' => (string) $folder->path,
                'name' => (string) $folder->name,
                'parent_path' => (string) ($folder->parent_path ?? ''),
                'owner_id' => (int) $folder->owner_id,
            ],
        ]);
    }

    public function deleteFolder(Request $request, DiskFileManager $diskFileManager): JsonResponse
    {
        $actor = $request->user();
        $this->ensureAccess($actor, 'delete');

        $validated = $request->validate([
            'folder' => ['required', 'string', 'max:180', 'not_regex:/\.\./'],
            'owner_id' => ['nullable', 'integer', 'exists:users,id'],
        ]);

        $folder = $diskFileManager->normalizeFolderPath((string) $validated['folder']);
        if ($folder === '') {
            throw ValidationException::withMessages([
                'folder' => __('Folder path is invalid.'),
            ]);
        }

        $ownerId = $this->resolveOwnerIdForFolderOperation($actor, (int) ($validated['owner_id'] ?? 0), false);

        $disks = Disk::query()
            ->where('owner_id', $ownerId)
            ->where(function (Builder $query) use ($folder): void {
                $this->applyFolderPrefixScope($query, 'folder', $folder);
            })
            ->get();

        foreach ($disks as $disk) {
            $diskFileManager->delete($disk);
        }

        DiskFolder::query()
            ->where('owner_id', $ownerId)
            ->where(function (Builder $query) use ($folder): void {
                $this->applyFolderPrefixScope($query, 'path', $folder);
            })
            ->delete();

        Storage::disk('public')->deleteDirectory('disk/'.$folder);

        return response()->json([
            'message' => __('Folder deleted.'),
        ]);
    }

    public function moveFolder(Request $request, DiskFileManager $diskFileManager): JsonResponse
    {
        $actor = $request->user();
        $this->ensureAccess($actor, 'update');

        $validated = $request->validate([
            'source_folder' => ['required', 'string', 'max:180', 'not_regex:/\.\./'],
            'destination_folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
            'owner_id' => ['nullable', 'integer', 'exists:users,id'],
        ]);

        $ownerId = $this->resolveOwnerIdForFolderOperation($actor, (int) ($validated['owner_id'] ?? 0), false);
        $sourceFolder = $diskFileManager->normalizeFolderPath((string) $validated['source_folder']);
        $destinationParent = $diskFileManager->normalizeFolderPath((string) ($validated['destination_folder'] ?? ''));

        if ($sourceFolder === '') {
            throw ValidationException::withMessages([
                'source_folder' => __('Source folder is invalid.'),
            ]);
        }

        $targetFolder = $diskFileManager->normalizeFolderPath(
            $destinationParent !== ''
                ? $destinationParent.'/'.basename($sourceFolder)
                : basename($sourceFolder)
        );

        if ($targetFolder === $sourceFolder) {
            return response()->json([
                'message' => __('Folder moved.'),
                'folder' => [
                    'path' => $targetFolder,
                    'owner_id' => $ownerId,
                ],
            ]);
        }

        if (str_starts_with($targetFolder.'/', $sourceFolder.'/')) {
            throw ValidationException::withMessages([
                'destination_folder' => __('You cannot move a folder into its nested folder.'),
            ]);
        }

        $targetExists = DiskFolder::query()
            ->where('owner_id', $ownerId)
            ->where('path', $targetFolder)
            ->exists();

        if ($targetExists) {
            throw ValidationException::withMessages([
                'destination_folder' => __('Folder with this name already exists.'),
            ]);
        }

        $affectedDisks = Disk::query()
            ->where('owner_id', $ownerId)
            ->where(function (Builder $query) use ($sourceFolder): void {
                $this->applyFolderPrefixScope($query, 'folder', $sourceFolder);
            })
            ->get();

        $fileMappings = [];
        foreach ($affectedDisks as $disk) {
            $oldStoragePath = trim((string) $disk->storage_path);
            $newStoragePath = $this->replacePrefixedPath($oldStoragePath, 'disk/'.$sourceFolder, 'disk/'.$targetFolder);
            $newFolderPath = $this->replacePrefixedPath((string) ($disk->folder ?? ''), $sourceFolder, $targetFolder);

            if ($oldStoragePath !== $newStoragePath && Storage::disk('public')->exists($newStoragePath)) {
                throw ValidationException::withMessages([
                    'destination_folder' => __('Target file already exists: :path', ['path' => $newStoragePath]),
                ]);
            }

            $fileMappings[] = [
                'disk_id' => (int) $disk->id,
                'old_storage_path' => $oldStoragePath,
                'new_storage_path' => $newStoragePath,
                'new_folder' => $newFolderPath,
            ];
        }

        $moved = [];
        try {
            foreach ($fileMappings as $mapping) {
                $oldStoragePath = $mapping['old_storage_path'];
                $newStoragePath = $mapping['new_storage_path'];
                if ($oldStoragePath === $newStoragePath) {
                    continue;
                }

                if (! Storage::disk('public')->exists($oldStoragePath)) {
                    throw ValidationException::withMessages([
                        'source_folder' => __('Source file does not exist: :path', ['path' => $oldStoragePath]),
                    ]);
                }

                Storage::disk('public')->makeDirectory((string) pathinfo($newStoragePath, PATHINFO_DIRNAME));

                if (! Storage::disk('public')->move($oldStoragePath, $newStoragePath)) {
                    throw ValidationException::withMessages([
                        'destination_folder' => __('Could not move file to target folder.'),
                    ]);
                }

                $moved[] = [
                    'from' => $newStoragePath,
                    'to' => $oldStoragePath,
                ];
            }
        } catch (\Throwable $exception) {
            foreach (array_reverse($moved) as $rollback) {
                if (Storage::disk('public')->exists((string) $rollback['from'])) {
                    Storage::disk('public')->move((string) $rollback['from'], (string) $rollback['to']);
                }
            }

            throw $exception;
        }

        DB::transaction(function () use ($ownerId, $sourceFolder, $targetFolder, $fileMappings, $diskFileManager): void {
            foreach ($fileMappings as $mapping) {
                Disk::query()
                    ->whereKey((int) $mapping['disk_id'])
                    ->update([
                        'folder' => $mapping['new_folder'] !== '' ? $mapping['new_folder'] : null,
                        'storage_path' => $mapping['new_storage_path'],
                    ]);
            }

            $folders = DiskFolder::query()
                ->where('owner_id', $ownerId)
                ->where(function (Builder $query) use ($sourceFolder): void {
                    $this->applyFolderPrefixScope($query, 'path', $sourceFolder);
                })
                ->get();

            foreach ($folders as $folder) {
                $newPath = $this->replacePrefixedPath((string) $folder->path, $sourceFolder, $targetFolder);
                $newParentPath = $folder->parent_path
                    ? $this->replacePrefixedPath((string) $folder->parent_path, $sourceFolder, $targetFolder)
                    : null;

                $folder->update([
                    'path' => $newPath,
                    'name' => basename($newPath),
                    'parent_path' => $newParentPath !== '' ? $newParentPath : null,
                ]);
            }

            DiskFolder::query()->updateOrCreate(
                [
                    'owner_id' => $ownerId,
                    'path' => $targetFolder,
                ],
                [
                    'name' => basename($targetFolder),
                    'parent_path' => str_contains($targetFolder, '/')
                        ? trim((string) dirname($targetFolder), '/.')
                        : null,
                ]
            );

            $owner = User::query()->find($ownerId);
            if ($owner) {
                $diskFileManager->ensureFolderCatalog($targetFolder, $owner);
            }
        });

        return response()->json([
            'message' => __('Folder moved.'),
            'folder' => [
                'path' => $targetFolder,
                'owner_id' => $ownerId,
            ],
        ]);
    }

    public function copyFolder(Request $request, DiskFileManager $diskFileManager): JsonResponse
    {
        $actor = $request->user();
        $this->ensureAccess($actor, 'create');

        $validated = $request->validate([
            'source_folder' => ['required', 'string', 'max:180', 'not_regex:/\.\./'],
            'destination_folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
            'owner_id' => ['nullable', 'integer', 'exists:users,id'],
        ]);

        $ownerId = $this->resolveOwnerIdForFolderOperation($actor, (int) ($validated['owner_id'] ?? 0), false);
        $sourceFolder = $diskFileManager->normalizeFolderPath((string) $validated['source_folder']);
        $destinationParent = $diskFileManager->normalizeFolderPath((string) ($validated['destination_folder'] ?? ''));

        if ($sourceFolder === '') {
            throw ValidationException::withMessages([
                'source_folder' => __('Source folder is invalid.'),
            ]);
        }

        $targetCandidate = $diskFileManager->normalizeFolderPath(
            $destinationParent !== ''
                ? $destinationParent.'/'.basename($sourceFolder)
                : basename($sourceFolder)
        );

        $targetFolder = $this->generateUniqueFolderPath($ownerId, $targetCandidate);

        $sourceFolders = DiskFolder::query()
            ->where('owner_id', $ownerId)
            ->where(function (Builder $query) use ($sourceFolder): void {
                $this->applyFolderPrefixScope($query, 'path', $sourceFolder);
            })
            ->orderBy('path')
            ->get();

        $sourceDisks = Disk::query()
            ->where('owner_id', $ownerId)
            ->where(function (Builder $query) use ($sourceFolder): void {
                $this->applyFolderPrefixScope($query, 'folder', $sourceFolder);
            })
            ->get();

        if ($sourceFolders->isEmpty() && $sourceDisks->isEmpty()) {
            throw ValidationException::withMessages([
                'source_folder' => __('Source folder does not exist.'),
            ]);
        }

        $copyMappings = [];
        foreach ($sourceDisks as $disk) {
            $oldStoragePath = trim((string) $disk->storage_path);
            $candidateStoragePath = $this->replacePrefixedPath($oldStoragePath, 'disk/'.$sourceFolder, 'disk/'.$targetFolder);
            $newStoragePath = $this->buildUniqueStoragePath(
                (string) pathinfo($candidateStoragePath, PATHINFO_DIRNAME),
                (string) pathinfo($candidateStoragePath, PATHINFO_BASENAME)
            );
            $newFolderPath = $this->replacePrefixedPath((string) ($disk->folder ?? ''), $sourceFolder, $targetFolder);

            $copyMappings[] = [
                'disk' => $disk,
                'old_storage_path' => $oldStoragePath,
                'new_storage_path' => $newStoragePath,
                'new_folder' => $newFolderPath,
            ];
        }

        $copiedStoragePaths = [];
        try {
            foreach ($copyMappings as $mapping) {
                $oldStoragePath = $mapping['old_storage_path'];
                $newStoragePath = $mapping['new_storage_path'];

                if (! Storage::disk('public')->exists($oldStoragePath)) {
                    throw ValidationException::withMessages([
                        'source_folder' => __('Source file does not exist: :path', ['path' => $oldStoragePath]),
                    ]);
                }

                Storage::disk('public')->makeDirectory((string) pathinfo($newStoragePath, PATHINFO_DIRNAME));
                if (! Storage::disk('public')->copy($oldStoragePath, $newStoragePath)) {
                    throw ValidationException::withMessages([
                        'destination_folder' => __('Could not copy file to target folder.'),
                    ]);
                }

                $copiedStoragePaths[] = $newStoragePath;
            }
        } catch (\Throwable $exception) {
            foreach ($copiedStoragePaths as $storagePath) {
                Storage::disk('public')->delete($storagePath);
            }

            throw $exception;
        }

        DB::transaction(function () use ($ownerId, $sourceFolder, $targetFolder, $sourceFolders, $copyMappings, $diskFileManager): void {
            if ($sourceFolders->isNotEmpty()) {
                foreach ($sourceFolders as $source) {
                    $newPath = $this->replacePrefixedPath((string) $source->path, $sourceFolder, $targetFolder);
                    $newParentPath = $source->parent_path
                        ? $this->replacePrefixedPath((string) $source->parent_path, $sourceFolder, $targetFolder)
                        : null;

                    DiskFolder::query()->updateOrCreate(
                        [
                            'owner_id' => $ownerId,
                            'path' => $newPath,
                        ],
                        [
                            'name' => basename($newPath),
                            'parent_path' => $newParentPath !== '' ? $newParentPath : null,
                        ]
                    );
                }
            } else {
                DiskFolder::query()->updateOrCreate(
                    [
                        'owner_id' => $ownerId,
                        'path' => $targetFolder,
                    ],
                    [
                        'name' => basename($targetFolder),
                        'parent_path' => str_contains($targetFolder, '/')
                            ? trim((string) dirname($targetFolder), '/.')
                            : null,
                    ]
                );
            }

            foreach ($copyMappings as $mapping) {
                /** @var Disk $sourceDisk */
                $sourceDisk = $mapping['disk'];
                $copy = $sourceDisk->replicate(['created_at', 'updated_at']);
                $copy->folder = $mapping['new_folder'] !== '' ? $mapping['new_folder'] : null;
                $copy->storage_path = $mapping['new_storage_path'];
                $copy->save();
            }

            $owner = User::query()->find($ownerId);
            if ($owner) {
                $diskFileManager->ensureFolderCatalog($targetFolder, $owner);
            }
        });

        return response()->json([
            'message' => __('Folder copied.'),
            'folder' => [
                'path' => $targetFolder,
                'owner_id' => $ownerId,
            ],
        ]);
    }

    public function moveFile(Request $request, Disk $disk, DiskFileManager $diskFileManager): JsonResponse
    {
        $this->authorize('update', $disk);

        $validated = $request->validate([
            'destination_folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
        ]);

        $destinationFolder = $diskFileManager->normalizeFolderPath((string) ($validated['destination_folder'] ?? ''));
        $oldStoragePath = trim((string) $disk->storage_path);
        $filename = basename($oldStoragePath);
        if ($filename === '' || $filename === '.' || $filename === '..') {
            $filename = trim((string) $disk->original_name);
        }

        $directory = $destinationFolder !== '' ? 'disk/'.$destinationFolder : 'disk';
        $newStoragePath = $this->buildUniqueStoragePath($directory, $filename, $oldStoragePath);

        if ($newStoragePath !== $oldStoragePath) {
            if (! Storage::disk('public')->exists($oldStoragePath)) {
                throw ValidationException::withMessages([
                    'destination_folder' => __('Source file does not exist.'),
                ]);
            }

            Storage::disk('public')->makeDirectory((string) pathinfo($newStoragePath, PATHINFO_DIRNAME));
            if (! Storage::disk('public')->move($oldStoragePath, $newStoragePath)) {
                throw ValidationException::withMessages([
                    'destination_folder' => __('Unable to move file.'),
                ]);
            }
        }

        $disk->fill([
            'folder' => $destinationFolder !== '' ? $destinationFolder : null,
            'storage_path' => $newStoragePath,
        ])->save();

        $owner = $disk->owner ?? User::query()->find($disk->owner_id);
        if ($destinationFolder !== '' && $owner) {
            $diskFileManager->ensureFolderCatalog($destinationFolder, $owner);
        }

        return response()->json([
            'message' => __('File moved.'),
            'file' => [
                'id' => (int) $disk->id,
                'folder' => (string) ($disk->folder ?? ''),
            ],
        ]);
    }

    public function copyFile(Request $request, Disk $disk, DiskFileManager $diskFileManager): JsonResponse
    {
        $this->authorize('view', $disk);
        $this->authorize('create', Disk::class);

        $validated = $request->validate([
            'destination_folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
        ]);

        $destinationFolder = $diskFileManager->normalizeFolderPath((string) ($validated['destination_folder'] ?? ''));
        $sourceStoragePath = trim((string) $disk->storage_path);
        $filename = basename($sourceStoragePath);
        if ($filename === '' || $filename === '.' || $filename === '..') {
            $filename = trim((string) $disk->original_name);
        }

        if (! Storage::disk('public')->exists($sourceStoragePath)) {
            throw ValidationException::withMessages([
                'destination_folder' => __('Source file does not exist.'),
            ]);
        }

        $directory = $destinationFolder !== '' ? 'disk/'.$destinationFolder : 'disk';
        $targetStoragePath = $this->buildUniqueStoragePath($directory, $filename);

        Storage::disk('public')->makeDirectory((string) pathinfo($targetStoragePath, PATHINFO_DIRNAME));
        if (! Storage::disk('public')->copy($sourceStoragePath, $targetStoragePath)) {
            throw ValidationException::withMessages([
                'destination_folder' => __('Unable to copy file.'),
            ]);
        }

        $copy = Disk::query()->create([
            'name' => (string) $disk->name,
            'original_name' => (string) $disk->original_name,
            'storage_path' => $targetStoragePath,
            'folder' => $destinationFolder !== '' ? $destinationFolder : null,
            'mime_type' => $disk->mime_type,
            'extension' => $disk->extension,
            'size' => (int) $disk->size,
            'description' => $disk->description,
            'is_public' => (bool) $disk->is_public,
            'owner_id' => (int) $disk->owner_id,
        ]);

        $owner = $disk->owner ?? User::query()->find($disk->owner_id);
        if ($destinationFolder !== '' && $owner) {
            $diskFileManager->ensureFolderCatalog($destinationFolder, $owner);
        }

        return response()->json([
            'message' => __('File copied.'),
            'file' => [
                'id' => (int) $copy->id,
                'folder' => (string) ($copy->folder ?? ''),
            ],
        ]);
    }

    public function deleteFile(Disk $disk, DiskFileManager $diskFileManager): JsonResponse
    {
        $this->authorize('delete', $disk);

        $diskFileManager->delete($disk);

        return response()->json([
            'message' => __('File deleted.'),
        ]);
    }

    private function ensureAccess(User $user, string $action): void
    {
        if (! AccessControl::allows($user, 'disks', $action)) {
            abort(403);
        }
    }

    private function resolveOwnerIdForFolderOperation(User $actor, int $requestedOwnerId, bool $allowDefaultWhenElevated): int
    {
        if (! AccessControl::isElevated($actor)) {
            return (int) $actor->id;
        }

        if ($requestedOwnerId > 0) {
            return $requestedOwnerId;
        }

        if ($allowDefaultWhenElevated) {
            return (int) $actor->id;
        }

        throw ValidationException::withMessages([
            'owner_id' => __('Specify the folder owner for this operation.'),
        ]);
    }

    private function normalizeFolderName(string $name): string
    {
        $name = trim($name);
        $name = str_replace(['\\', '/'], ' ', $name);
        $name = preg_replace('/\s+/', ' ', $name) ?? '';

        return trim($name);
    }

    private function applyFolderPrefixScope(Builder $query, string $column, string $folder): void
    {
        $query->where($column, $folder)
            ->orWhere($column, 'like', $folder.'/%');
    }

    private function replacePrefixedPath(string $value, string $fromPrefix, string $toPrefix): string
    {
        $value = trim($value);
        $fromPrefix = trim($fromPrefix);
        $toPrefix = trim($toPrefix);

        if ($value === $fromPrefix) {
            return $toPrefix;
        }

        if ($fromPrefix !== '' && str_starts_with($value, $fromPrefix.'/')) {
            return $toPrefix.substr($value, strlen($fromPrefix));
        }

        return $value;
    }

    private function generateUniqueFolderPath(int $ownerId, string $candidate): string
    {
        $candidate = trim($candidate, '/');
        if ($candidate === '') {
            $candidate = 'New folder';
        }

        $base = $candidate;
        $suffix = 1;

        while (DiskFolder::query()->where('owner_id', $ownerId)->where('path', $candidate)->exists()) {
            $suffix++;
            $candidate = $base.' (copy '.$suffix.')';
        }

        return $candidate;
    }

    private function buildUniqueStoragePath(string $directory, string $filename, ?string $currentPath = null): string
    {
        $directory = trim(str_replace('\\', '/', $directory), '/');
        if ($directory === '.') {
            $directory = '';
        }
        $filename = trim($filename);
        if ($filename === '') {
            $filename = 'file.bin';
        }

        $name = (string) pathinfo($filename, PATHINFO_FILENAME);
        $extension = strtolower((string) pathinfo($filename, PATHINFO_EXTENSION));
        $candidate = $directory !== '' ? $directory.'/'.$filename : $filename;
        $suffix = 1;

        while (
            Storage::disk('public')->exists($candidate)
            && ($currentPath === null || $candidate !== $currentPath)
        ) {
            $suffix++;
            $candidateName = $name.'-'.$suffix.($extension !== '' ? '.'.$extension : '');
            $candidate = $directory !== '' ? $directory.'/'.$candidateName : $candidateName;
        }

        return $candidate;
    }
}
