@php
    use App\Support\AccessControl;

    $user = auth()->user();
    $isRu = str_starts_with((string) app()->getLocale(), 'ru');
    $t = static fn (string $ru, string $en): string => $isRu ? $ru : $en;

    $sections = [
        [
            'id' => 'workspace',
            'icon' => 'fa-solid fa-compass',
            'duration' => $t('5 мин', '5 min'),
            'entity' => null,
            'route' => 'dashboard',
            'title' => $t('Рабочее пространство и навигация', 'Workspace and navigation'),
            'summary' => $t('Как ориентироваться в CRM, работать с левым меню, правой панелью и быстрыми переходами в сайдбар.', 'How to navigate CRM, use left menu, right rail and quick sidebar transitions.'),
            'capabilities' => [
                $t('Быстрое переключение по разделам из левого меню.', 'Fast switching between sections from the left menu.'),
                $t('Открытие карточек задач/сделок/контактов в сайдбаре.', 'Opening task/deal/contact cards in sidebar mode.'),
                $t('Работа с правой панелью: уведомления и список чатов.', 'Using the right rail: notifications and chat list.'),
            ],
            'scenario' => [
                $t('Откройте пункт «Статистика» и проверьте верхние метрики.', 'Open “Statistics” and check top metrics.'),
                $t('Перейдите в «Задачи», откройте любую задачу в сайдбаре.', 'Go to “Tasks” and open any task in sidebar.'),
                $t('Вернитесь назад и закрепите удобный порядок меню.', 'Return back and keep a convenient menu order.'),
            ],
            'tips' => [
                $t('Если панель слева свернута, наведите курсор для временного раскрытия.', 'If left panel is collapsed, hover to expand it temporarily.'),
                $t('Права на разделы влияют на видимость пунктов меню.', 'Section permissions affect menu visibility.'),
            ],
            'keywords' => $t('навигация меню сайдбар', 'navigation menu sidebar'),
        ],
        [
            'id' => 'news',
            'icon' => 'fa-solid fa-newspaper',
            'duration' => $t('4 мин', '4 min'),
            'entity' => null,
            'route' => 'news.index',
            'title' => $t('Новости', 'News'),
            'summary' => $t('Лента корпоративных новостей без лишних сущностей: только актуальные публикации и быстрый просмотр.', 'Corporate news feed without extra entities: only current posts and quick reading.'),
            'capabilities' => [
                $t('Просмотр свежих новостей компании.', 'Viewing fresh company news.'),
                $t('Поддержка превью-изображения для публикаций.', 'Preview image support for posts.'),
                $t('Отметка просмотров публикации.', 'Tracking who has viewed a post.'),
            ],
            'scenario' => [
                $t('Откройте раздел «Новости».', 'Open “News”.'),
                $t('Откройте публикацию и проверьте детали.', 'Open a post and review details.'),
                $t('Вернитесь в ленту и отсортируйте по актуальности.', 'Return to the feed and review most recent updates.'),
            ],
            'tips' => [
                $t('Используйте новости для внутренних объявлений команды.', 'Use news for internal team announcements.'),
                $t('Короткие заголовки повышают читаемость ленты.', 'Short titles improve readability.'),
            ],
            'keywords' => $t('новости публикации лента', 'news feed posts'),
        ],
        [
            'id' => 'tasks',
            'icon' => 'fa-solid fa-list-check',
            'duration' => $t('12 мин', '12 min'),
            'entity' => 'tasks',
            'route' => 'tasks.index',
            'title' => $t('Задачи: список, канбан, сроки и подзадачи', 'Tasks: list, kanban, deadlines and subtasks'),
            'summary' => $t('Основной рабочий раздел сотрудника: постановка, исполнение, контроль сроков и статусов.', 'Primary employee workspace: assignment, execution, deadlines and status control.'),
            'capabilities' => [
                $t('Переключение между списком, канбаном и другими представлениями.', 'Switching between list, kanban and other views.'),
                $t('Перетаскивание задач между стадиями без перезагрузки.', 'Drag-and-drop tasks between stages without reload.'),
                $t('Работа с подзадачами, ответственным, дедлайном и приоритетом.', 'Working with subtasks, assignee, deadline and priority.'),
            ],
            'scenario' => [
                $t('Создайте задачу с дедлайном и ответственным.', 'Create a task with deadline and assignee.'),
                $t('Перетащите задачу на следующую стадию в канбане.', 'Drag the task to the next stage in kanban.'),
                $t('Добавьте подзадачу и отметьте выполнение.', 'Add a subtask and mark completion.'),
            ],
            'tips' => [
                $t('Начинайте с фильтра: название, статус, ответственный.', 'Start with base filters: title, status, assignee.'),
                $t('Старайтесь держать задачи короткими и измеримыми.', 'Keep tasks concise and measurable.'),
            ],
            'keywords' => $t('задачи канбан список подзадачи', 'tasks kanban list subtasks'),
        ],
        [
            'id' => 'deals',
            'icon' => 'fa-solid fa-briefcase',
            'duration' => $t('10 мин', '10 min'),
            'entity' => 'deals',
            'route' => 'deals.index',
            'title' => $t('Сделки и воронки', 'Deals and pipelines'),
            'summary' => $t('Ведение сделок по стадиям, работа с ответственными, суммами и связанными сущностями.', 'Tracking deals by stages with owners, amounts and linked entities.'),
            'capabilities' => [
                $t('Просмотр сделок в списке и канбане.', 'View deals in list and kanban modes.'),
                $t('Перемещение сделки между стадиями воронки.', 'Move a deal through pipeline stages.'),
                $t('Связь с компанией, контактом и задачами.', 'Link deals to companies, contacts and tasks.'),
            ],
            'scenario' => [
                $t('Откройте нужную воронку.', 'Open the required pipeline.'),
                $t('Создайте сделку и заполните сумму/приоритет.', 'Create a deal and fill amount/priority.'),
                $t('Перетащите карточку в следующую стадию.', 'Drag the card to the next stage.'),
            ],
            'tips' => [
                $t('Фиксируйте причину потери сделки сразу.', 'Record lost reason immediately.'),
                $t('Регулярно обновляйте ожидаемую дату закрытия.', 'Regularly update expected close date.'),
            ],
            'keywords' => $t('сделки воронки стадии', 'deals pipelines stages'),
        ],
        [
            'id' => 'contacts-companies',
            'icon' => 'fa-solid fa-address-book',
            'duration' => $t('8 мин', '8 min'),
            'entity' => 'contacts',
            'route' => 'contacts.index',
            'title' => $t('Контакты и компании', 'Contacts and companies'),
            'summary' => $t('Единая клиентская база: карточки, связи и быстрый переход к сделкам и задачам.', 'Unified client base: cards, links and quick navigation to deals and tasks.'),
            'capabilities' => [
                $t('Создание и редактирование карточек контактов.', 'Create and edit contact cards.'),
                $t('Связывание контактов с компаниями.', 'Link contacts with companies.'),
                $t('Добавление аватаров для компаний и контактов.', 'Set avatars for companies and contacts.'),
            ],
            'scenario' => [
                $t('Создайте контакт и привяжите его к компании.', 'Create a contact and attach it to a company.'),
                $t('Заполните телефон, email и должность.', 'Fill phone, email and job title.'),
                $t('Проверьте связанные сделки из карточки.', 'Check linked deals from the card.'),
            ],
            'tips' => [
                $t('Избегайте дублей, сначала ищите существующий контакт.', 'Avoid duplicates, search before creating.'),
                $t('Поддерживайте актуальность контактов после каждого звонка.', 'Keep contact data up to date after each call.'),
            ],
            'keywords' => $t('контакты компании карточка', 'contacts companies card'),
        ],
        [
            'id' => 'calendar',
            'icon' => 'fa-solid fa-calendar-days',
            'duration' => $t('7 мин', '7 min'),
            'entity' => 'calendar',
            'route' => 'calendar.index',
            'title' => $t('Календарь задач и событий', 'Task and event calendar'),
            'summary' => $t('Планирование загрузки: календарные представления, перенос задач и контроль дедлайнов.', 'Workload planning with calendar views, drag rescheduling and deadline control.'),
            'capabilities' => [
                $t('Режимы месяц/неделя/день/список.', 'Month/week/day/list modes.'),
                $t('Перенос задач перетаскиванием.', 'Drag-and-drop task rescheduling.'),
                $t('Просмотр задач и событий сделок в едином окне.', 'Unified view for tasks and deal events.'),
            ],
            'scenario' => [
                $t('Откройте календарь и выберите «Неделя».', 'Open calendar and choose “Week”.'),
                $t('Перетащите задачу на новую дату.', 'Drag a task to a new date.'),
                $t('Проверьте, что срок в задаче обновился.', 'Verify task due date is updated.'),
            ],
            'tips' => [
                $t('Используйте календарь как ежедневную точку старта.', 'Use calendar as a daily start point.'),
                $t('Планируйте буфер между задачами высокой важности.', 'Keep time buffer between high priority tasks.'),
            ],
            'keywords' => $t('календарь сроки события', 'calendar deadlines events'),
        ],
        [
            'id' => 'projects',
            'icon' => 'fa-solid fa-diagram-project',
            'duration' => $t('9 мин', '9 min'),
            'entity' => 'projects',
            'route' => 'projects.index',
            'title' => $t('Проекты', 'Projects'),
            'summary' => $t('Работа с проектными этапами, задачами команды и контролем прогресса.', 'Work with project stages, team tasks and progress tracking.'),
            'capabilities' => [
                $t('Карточка проекта с участниками и сроками.', 'Project card with members and deadlines.'),
                $t('Связка проектных задач со стадиями.', 'Project task-stage linkage.'),
                $t('Контроль прогресса по проекту.', 'Project progress control.'),
            ],
            'scenario' => [
                $t('Откройте проект и проверьте состав участников.', 'Open project and review team members.'),
                $t('Создайте задачу в нужной стадии проекта.', 'Create task in a required project stage.'),
                $t('Отследите изменение прогресса после выполнения.', 'Track progress change after completion.'),
            ],
            'tips' => [
                $t('Названия стадий делайте однозначными для команды.', 'Use clear stage names for the team.'),
                $t('Регулярно обновляйте ответственных за блоки работ.', 'Regularly update owners for work blocks.'),
            ],
            'keywords' => $t('проекты этапы прогресс', 'projects stages progress'),
        ],
        [
            'id' => 'activities',
            'icon' => 'fa-solid fa-bolt',
            'duration' => $t('6 мин', '6 min'),
            'entity' => 'activities',
            'route' => 'activities.index',
            'title' => $t('Активности', 'Activities'),
            'summary' => $t('Журнал действий и коммуникаций: встречи, звонки, письма и служебные события.', 'Activity journal for meetings, calls, emails and service events.'),
            'capabilities' => [
                $t('Регистрация активности с датой и типом.', 'Register activity with date and type.'),
                $t('Фильтрация по пользователю и периоду.', 'Filter by user and period.'),
                $t('Быстрый переход к связанной сущности.', 'Quick jump to linked entity.'),
            ],
            'scenario' => [
                $t('Создайте новую активность по клиенту.', 'Create a new client-related activity.'),
                $t('Заполните тип и комментарий.', 'Fill type and comment.'),
                $t('Проверьте отображение в общем журнале.', 'Verify it appears in the journal.'),
            ],
            'tips' => [
                $t('Фиксируйте действия сразу после контакта.', 'Log activity right after communication.'),
                $t('Используйте единый формат коротких комментариев.', 'Use a unified short comment format.'),
            ],
            'keywords' => $t('активности журнал события', 'activities journal events'),
        ],
        [
            'id' => 'disk',
            'icon' => 'fa-solid fa-hard-drive',
            'duration' => $t('10 мин', '10 min'),
            'entity' => 'disks',
            'route' => 'disks.index',
            'title' => $t('Диск и файлы', 'Disk and files'),
            'summary' => $t('Хранение и совместная работа с файлами: каталоги, предпросмотр и ссылки.', 'File storage and collaboration: folders, preview and share links.'),
            'capabilities' => [
                $t('Загрузка файлов через drag-and-drop.', 'Upload files via drag-and-drop area.'),
                $t('Каталоги и работа в формате проводника.', 'Folders and explorer-like navigation.'),
                $t('Предпросмотр PDF/Excel/Word внутри CRM.', 'Preview PDF/Excel/Word inside CRM.'),
            ],
            'scenario' => [
                $t('Создайте каталог под проект.', 'Create a project folder.'),
                $t('Перетащите в него файлы.', 'Drag files into it.'),
                $t('Скопируйте ссылку и отправьте в чат.', 'Copy file link and send it to chat.'),
            ],
            'tips' => [
                $t('Соблюдайте единые правила именования файлов.', 'Use consistent file naming rules.'),
                $t('Группируйте файлы по клиенту или проекту.', 'Group files by client or project.'),
            ],
            'keywords' => $t('диск файлы каталоги', 'disk files folders'),
        ],
        [
            'id' => 'forms',
            'icon' => 'fa-solid fa-file-lines',
            'duration' => $t('11 мин', '11 min'),
            'entity' => 'forms',
            'route' => 'forms.index',
            'title' => $t('Формы', 'Forms'),
            'summary' => $t('Создание веб-форм, настройка полей и поведения после отправки.', 'Create web forms, configure fields and post-submit behavior.'),
            'capabilities' => [
                $t('Конструктор полей с динамическим добавлением.', 'Field builder with dynamic adding.'),
                $t('Связь полей формы с полями CRM.', 'Field mapping between form and CRM entities.'),
                $t('Сценарии после отправки: сообщение, редирект, перезагрузка.', 'Post-submit behavior: message, redirect, reload.'),
            ],
            'scenario' => [
                $t('Создайте форму и добавьте базовые поля.', 'Create a form and add base fields.'),
                $t('Настройте создание лида или задачи после отправки.', 'Configure lead or task creation on submit.'),
                $t('Проверьте публичную ссылку формы.', 'Test public form URL.'),
            ],
            'tips' => [
                $t('Начинайте с минимального набора полей.', 'Start with a minimal set of fields.'),
                $t('Проверяйте обязательность и формат до публикации.', 'Validate required fields and formats before publishing.'),
            ],
            'keywords' => $t('формы вебформа поля', 'forms webform fields'),
        ],
        [
            'id' => 'chat-notifications',
            'icon' => 'fa-solid fa-comments',
            'duration' => $t('8 мин', '8 min'),
            'entity' => null,
            'route' => 'dashboard',
            'title' => $t('Чаты и уведомления', 'Chats and notifications'),
            'summary' => $t('Коммуникации команды в правой панели и сайдбаре: сообщения, статусы, отметка прочтения.', 'Team communications in right rail and sidebar: messages, statuses and read indicators.'),
            'capabilities' => [
                $t('Открытие чата в сайдбаре по клику на сотрудника.', 'Open chat in sidebar by clicking an employee.'),
                $t('Отправка сообщения по Enter, перенос строки по Shift+Enter.', 'Send by Enter, new line by Shift+Enter.'),
                $t('Индикация online и статуса прочтения сообщений.', 'Online and read-status indicators.'),
            ],
            'scenario' => [
                $t('Откройте чат с сотрудником справа.', 'Open chat with an employee from right rail.'),
                $t('Отправьте сообщение и проверьте галочки прочтения.', 'Send a message and verify read marks.'),
                $t('Откройте профиль сотрудника из имени в чате.', 'Open employee profile from chat name.'),
            ],
            'tips' => [
                $t('Используйте короткие сообщения с контекстом задачи/сделки.', 'Use short messages with task/deal context.'),
                $t('Вложения из диска ускоряют обмен файлами.', 'Disk attachments speed up file sharing.'),
            ],
            'keywords' => $t('чат уведомления сообщения', 'chat notifications messages'),
        ],
        [
            'id' => 'messengers',
            'icon' => 'fa-solid fa-comment-dots',
            'duration' => $t('7 мин', '7 min'),
            'entity' => 'messengers',
            'route' => 'messengers.index',
            'title' => $t('Мессенджеры', 'Messengers'),
            'summary' => $t('Работа с внешними каналами сообщений внутри CRM (если раздел доступен по правам).', 'Work with external messenger channels inside CRM (if allowed by permissions).'),
            'capabilities' => [
                $t('Список переписок с фильтрами.', 'Conversation list with filters.'),
                $t('Карточка диалога и отправка сообщений.', 'Dialog card and message sending.'),
                $t('Выбор ответственного пользователя по переписке.', 'Assign responsible user for conversation.'),
            ],
            'scenario' => [
                $t('Откройте список диалогов.', 'Open conversation list.'),
                $t('Найдите переписку по имени контакта.', 'Find conversation by contact name.'),
                $t('Отправьте ответ и проверьте статус.', 'Send reply and verify status.'),
            ],
            'tips' => [
                $t('Используйте фильтры по каналу и периоду.', 'Use channel and period filters.'),
                $t('Ведите переписку в едином формате для команды.', 'Keep a unified communication format for the team.'),
            ],
            'keywords' => $t('мессенджеры диалоги каналы', 'messengers conversations channels'),
        ],
        [
            'id' => 'telephony',
            'icon' => 'fa-solid fa-phone',
            'duration' => $t('6 мин', '6 min'),
            'entity' => 'telephony',
            'route' => 'telephony.index',
            'title' => $t('Телефония', 'Telephony'),
            'summary' => $t('Журнал звонков и рабочая аналитика по входящим/исходящим (при наличии прав).', 'Call log and activity analytics for inbound/outbound calls (with granted access).'),
            'capabilities' => [
                $t('Просмотр истории звонков.', 'View call history.'),
                $t('Фильтрация по периоду и направлению.', 'Filter by period and direction.'),
                $t('Переход к записи звонка, если она доступна.', 'Open call recording when available.'),
            ],
            'scenario' => [
                $t('Откройте журнал звонков.', 'Open call log.'),
                $t('Примените фильтр за неделю.', 'Apply last-week filter.'),
                $t('Откройте запись звонка из строки журнала.', 'Open recording from call row.'),
            ],
            'tips' => [
                $t('Заполняйте внешние ID для интеграций без дублей.', 'Keep external IDs clean to avoid integration duplicates.'),
                $t('Используйте длительность и статус для оценки качества обработки.', 'Use duration and status for quality analysis.'),
            ],
            'keywords' => $t('телефония звонки журнал', 'telephony calls log'),
        ],
        [
            'id' => 'profile',
            'icon' => 'fa-solid fa-user',
            'duration' => $t('6 мин', '6 min'),
            'entity' => null,
            'route' => 'profile.edit',
            'title' => $t('Профиль и персональные настройки', 'Profile and personal settings'),
            'summary' => $t('Настройка личных параметров: аватар, язык, оформление и стартовая страница.', 'Configure personal params: avatar, language, appearance and start page.'),
            'capabilities' => [
                $t('Смена аватара с настройкой зума.', 'Avatar change with zoom.'),
                $t('Быстрое переключение RU/EN.', 'Fast RU/EN switch.'),
                $t('Персональные настройки интерфейса и вида разделов.', 'Personal UI settings and section view preferences.'),
            ],
            'scenario' => [
                $t('Откройте «Профиль».', 'Open “Profile”.'),
                $t('Загрузите фото и настройте масштаб.', 'Upload photo and adjust zoom.'),
                $t('Переключите язык и проверьте меню.', 'Switch language and verify menu labels.'),
            ],
            'tips' => [
                $t('Выберите стартовую страницу под ваш рабочий сценарий.', 'Set start page for your daily workflow.'),
                $t('Периодически обновляйте телефон и должность в профиле.', 'Keep phone and job title in profile up to date.'),
            ],
            'keywords' => $t('профиль язык аватар', 'profile language avatar'),
        ],
    ];

    $catalog = collect($sections)->map(fn (array $section): array => [
        'id' => (string) $section['id'],
        'title' => (string) $section['title'],
        'keywords' => (string) ($section['keywords'] ?? ''),
    ])->values()->all();

    $accessibleCount = collect($sections)->filter(function (array $section) use ($user): bool {
        $entity = $section['entity'] ?? null;
        if (! is_string($entity) || $entity === '') {
            return true;
        }

        return AccessControl::allows($user, $entity, 'read');
    })->count();
@endphp

<x-app-layout>
    <x-slot name="header">
        <div class="flex flex-wrap items-start justify-between gap-4">
            <div>
                <h2 class="text-xl font-semibold leading-tight text-gray-800">
                    {{ $t('Каталог обучения сотрудников', 'Employee training catalog') }}
                </h2>
                <p class="mt-1 text-sm text-gray-500">
                    {{ $t('Фронтенд-центр обучения для пользователей без админ-прав.', 'Frontend learning center for users without admin rights.') }}
                </p>
            </div>
            <div class="rounded-lg border border-slate-200 bg-white px-3 py-2 text-sm text-slate-600">
                {{ $t('Доступно по вашим правам:', 'Available with your permissions:') }}
                <span class="font-semibold text-slate-900">{{ $accessibleCount }}/{{ count($sections) }}</span>
            </div>
        </div>
    </x-slot>

    <div
        class="w-full px-6 pb-10"
        x-data="academyCatalog({
            sections: @js($catalog),
            labels: {
                all: @js($t('Все', 'All')),
                clear: @js($t('Сбросить прогресс', 'Reset progress')),
                done: @js($t('Пройдено', 'Completed')),
                notDone: @js($t('Отметить как пройдено', 'Mark as completed'))
            }
        })"
        x-init="init()"
    >
        <div class="grid grid-cols-1 gap-6 xl:grid-cols-[320px,minmax(0,1fr)]">
            <aside class="space-y-4 xl:sticky xl:top-6 xl:h-[calc(100vh-8rem)] xl:overflow-y-auto">
                <section class="rounded-xl border border-slate-200 bg-white p-4 shadow-sm">
                    <h3 class="text-sm font-semibold text-slate-900">{{ $t('Оглавление', 'Contents') }}</h3>
                    <div class="mt-3">
                        <input
                            type="text"
                            x-model.trim="query"
                            class="w-full rounded-md border-slate-300 text-sm shadow-sm focus:border-indigo-500 focus:ring-indigo-500"
                            placeholder="{{ $t('Поиск по разделам…', 'Search sections…') }}"
                        >
                    </div>
                    <div class="mt-3 h-2 w-full overflow-hidden rounded-full bg-slate-100">
                        <div class="h-full rounded-full bg-indigo-500 transition-all duration-200" :style="`width: ${progressPercent()}%`"></div>
                    </div>
                    <p class="mt-2 text-xs text-slate-500">
                        <span x-text="progressText()"></span>
                    </p>
                    <button
                        type="button"
                        @click="resetProgress()"
                        class="mt-3 inline-flex items-center rounded-md border border-slate-300 px-2.5 py-1.5 text-xs font-medium text-slate-700 hover:bg-slate-50"
                    >
                        {{ $t('Сбросить прогресс', 'Reset progress') }}
                    </button>
                </section>

                <section class="rounded-xl border border-slate-200 bg-white p-3 shadow-sm">
                    <div class="space-y-1">
                        @foreach ($sections as $section)
                            <button
                                type="button"
                                @click="scrollToSection(@js($section['id']))"
                                x-show="isVisible(@js($section['id']))"
                                :class="isCompleted(@js($section['id'])) ? 'border-emerald-200 bg-emerald-50 text-emerald-800' : 'border-slate-200 bg-white text-slate-700 hover:bg-slate-50'"
                                class="flex w-full items-center gap-2 rounded-lg border px-2.5 py-2 text-left text-sm transition"
                            >
                                <x-menu-icon :name="$section['icon']" class="h-4 w-4 shrink-0" />
                                <span class="min-w-0 flex-1 truncate">{{ $section['title'] }}</span>
                                <span class="text-[10px] font-semibold uppercase tracking-wide" x-text="isCompleted(@js($section['id'])) ? labels.done : ''"></span>
                            </button>
                        @endforeach
                    </div>
                </section>
            </aside>

            <main class="space-y-4">
                <section class="rounded-xl border border-slate-200 bg-white p-5 shadow-sm">
                    <h3 class="text-base font-semibold text-slate-900">{{ $t('Как проходить обучение', 'How to complete training') }}</h3>
                    <div class="mt-3 grid gap-3 md:grid-cols-3">
                        <div class="rounded-lg border border-slate-200 bg-slate-50 p-3 text-sm text-slate-700">
                            <p class="font-medium text-slate-900">1. {{ $t('Выберите раздел', 'Pick a section') }}</p>
                            <p class="mt-1">{{ $t('Начните с «Задач» и «Сделок» — это базовый ежедневный поток.', 'Start with “Tasks” and “Deals” as your daily baseline workflow.') }}</p>
                        </div>
                        <div class="rounded-lg border border-slate-200 bg-slate-50 p-3 text-sm text-slate-700">
                            <p class="font-medium text-slate-900">2. {{ $t('Повторите сценарий', 'Run the scenario') }}</p>
                            <p class="mt-1">{{ $t('В каждом блоке есть пошаговый сценарий, повторите его в рабочем интерфейсе.', 'Each block has a step-by-step scenario, repeat it in the actual UI.') }}</p>
                        </div>
                        <div class="rounded-lg border border-slate-200 bg-slate-50 p-3 text-sm text-slate-700">
                            <p class="font-medium text-slate-900">3. {{ $t('Отметьте как пройдено', 'Mark as completed') }}</p>
                            <p class="mt-1">{{ $t('Прогресс сохраняется локально в браузере, без серверной логики.', 'Progress is stored locally in the browser with frontend-only logic.') }}</p>
                        </div>
                    </div>
                </section>

                @foreach ($sections as $section)
                    @php
                        $entity = $section['entity'] ?? null;
                        $hasAccess = ! is_string($entity) || $entity === '' ? true : AccessControl::allows($user, $entity, 'read');
                        $routeName = $section['route'] ?? null;
                        $routeUrl = is_string($routeName) && $routeName !== '' ? route($routeName) : null;
                    @endphp
                    <article
                        id="academy-{{ $section['id'] }}"
                        x-show="isVisible(@js($section['id']))"
                        class="rounded-xl border border-slate-200 bg-white p-5 shadow-sm"
                    >
                        <div class="flex flex-wrap items-start justify-between gap-3">
                            <div class="min-w-0 flex-1">
                                <div class="flex items-center gap-2">
                                    <x-menu-icon :name="$section['icon']" class="h-5 w-5 text-indigo-600" />
                                    <h3 class="truncate text-lg font-semibold text-slate-900">{{ $section['title'] }}</h3>
                                </div>
                                <p class="mt-2 text-sm text-slate-600">{{ $section['summary'] }}</p>
                            </div>
                            <div class="flex items-center gap-2">
                                <span class="rounded-full border border-slate-200 bg-slate-50 px-2.5 py-1 text-xs font-medium text-slate-700">{{ $section['duration'] }}</span>
                                @if ($hasAccess)
                                    <span class="rounded-full border border-emerald-200 bg-emerald-50 px-2.5 py-1 text-xs font-medium text-emerald-700">{{ $t('Доступен', 'Available') }}</span>
                                @else
                                    <span class="rounded-full border border-amber-200 bg-amber-50 px-2.5 py-1 text-xs font-medium text-amber-700">{{ $t('Нет доступа', 'No access') }}</span>
                                @endif
                            </div>
                        </div>

                        <div class="mt-4 flex flex-wrap items-center gap-2">
                            @if ($routeUrl && $hasAccess)
                                <a
                                    href="{{ $routeUrl }}"
                                    class="inline-flex items-center gap-2 rounded-md bg-indigo-600 px-3 py-1.5 text-xs font-semibold text-white hover:bg-indigo-500"
                                >
                                    <x-menu-icon name="fa-solid fa-arrow-up-right-from-square" class="h-3.5 w-3.5" />
                                    {{ $t('Открыть раздел', 'Open section') }}
                                </a>
                            @elseif ($routeUrl)
                                <span class="inline-flex items-center gap-2 rounded-md border border-slate-200 bg-slate-50 px-3 py-1.5 text-xs font-medium text-slate-500">
                                    <x-menu-icon name="fa-solid fa-lock" class="h-3.5 w-3.5" />
                                    {{ $t('Раздел скрыт по правам', 'Section hidden by permissions') }}
                                </span>
                            @endif
                            <button
                                type="button"
                                @click="toggleCompleted(@js($section['id']))"
                                :class="isCompleted(@js($section['id'])) ? 'border-emerald-300 bg-emerald-50 text-emerald-700' : 'border-slate-300 bg-white text-slate-700 hover:bg-slate-50'"
                                class="inline-flex items-center gap-2 rounded-md border px-3 py-1.5 text-xs font-medium"
                            >
                                <x-menu-icon name="fa-solid fa-circle-check" class="h-3.5 w-3.5" />
                                <span x-text="isCompleted(@js($section['id'])) ? labels.done : labels.notDone"></span>
                            </button>
                        </div>

                        <div class="mt-4 grid grid-cols-1 gap-4 lg:grid-cols-3">
                            <div class="rounded-lg border border-slate-200 bg-slate-50 p-3">
                                <h4 class="text-sm font-semibold text-slate-900">{{ $t('Возможности', 'Capabilities') }}</h4>
                                <ul class="mt-2 list-disc space-y-1 pl-4 text-sm text-slate-700">
                                    @foreach ($section['capabilities'] as $capability)
                                        <li>{{ $capability }}</li>
                                    @endforeach
                                </ul>
                            </div>
                            <div class="rounded-lg border border-slate-200 bg-slate-50 p-3">
                                <h4 class="text-sm font-semibold text-slate-900">{{ $t('Пошаговый сценарий', 'Step-by-step scenario') }}</h4>
                                <ol class="mt-2 list-decimal space-y-1 pl-4 text-sm text-slate-700">
                                    @foreach ($section['scenario'] as $step)
                                        <li>{{ $step }}</li>
                                    @endforeach
                                </ol>
                            </div>
                            <div class="rounded-lg border border-slate-200 bg-slate-50 p-3">
                                <h4 class="text-sm font-semibold text-slate-900">{{ $t('Практические советы', 'Practical tips') }}</h4>
                                <ul class="mt-2 list-disc space-y-1 pl-4 text-sm text-slate-700">
                                    @foreach ($section['tips'] as $tip)
                                        <li>{{ $tip }}</li>
                                    @endforeach
                                </ul>
                            </div>
                        </div>
                    </article>
                @endforeach
            </main>
        </div>
    </div>

    @once
        <script>
            document.addEventListener('alpine:init', () => {
                Alpine.data('academyCatalog', ({ sections = [], labels = {} } = {}) => ({
                    sections: Array.isArray(sections) ? sections : [],
                    labels: {
                        all: labels.all || 'All',
                        clear: labels.clear || 'Reset progress',
                        done: labels.done || 'Done',
                        notDone: labels.notDone || 'Mark as completed',
                    },
                    query: '',
                    completed: {},
                    storageKey: 'crm25.academy.progress.v1',

                    init() {
                        this.loadProgress();
                        this.$nextTick(() => this.scrollFromHash());
                    },

                    loadProgress() {
                        try {
                            const raw = window.localStorage.getItem(this.storageKey);
                            const parsed = raw ? JSON.parse(raw) : {};
                            this.completed = parsed && typeof parsed === 'object' ? parsed : {};
                        } catch (error) {
                            this.completed = {};
                        }
                    },

                    saveProgress() {
                        window.localStorage.setItem(this.storageKey, JSON.stringify(this.completed));
                    },

                    resetProgress() {
                        this.completed = {};
                        this.saveProgress();
                    },

                    isVisible(id) {
                        const section = this.sections.find((item) => String(item.id) === String(id));
                        if (!section) {
                            return true;
                        }

                        const needle = String(this.query || '').trim().toLowerCase();
                        if (needle === '') {
                            return true;
                        }

                        const haystack = `${section.title || ''} ${section.keywords || ''}`.toLowerCase();

                        return haystack.includes(needle);
                    },

                    toggleCompleted(id) {
                        const key = String(id);
                        this.completed[key] = !Boolean(this.completed[key]);
                        this.saveProgress();
                    },

                    isCompleted(id) {
                        return Boolean(this.completed[String(id)]);
                    },

                    completedCount() {
                        return this.sections.filter((section) => this.isCompleted(section.id)).length;
                    },

                    progressPercent() {
                        if (this.sections.length === 0) {
                            return 0;
                        }

                        return Math.round((this.completedCount() / this.sections.length) * 100);
                    },

                    progressText() {
                        return `${this.completedCount()} / ${this.sections.length} (${this.progressPercent()}%)`;
                    },

                    scrollToSection(id) {
                        const node = document.getElementById(`academy-${id}`);
                        if (!node) {
                            return;
                        }

                        node.scrollIntoView({ behavior: 'smooth', block: 'start' });
                        window.history.replaceState(null, '', `#academy-${id}`);
                    },

                    scrollFromHash() {
                        const hash = window.location.hash || '';
                        if (!hash.startsWith('#academy-')) {
                            return;
                        }

                        const node = document.querySelector(hash);
                        if (!node) {
                            return;
                        }

                        node.scrollIntoView({ behavior: 'smooth', block: 'start' });
                    },
                }));
            });
        </script>
    @endonce
</x-app-layout>
