<?php

namespace App\Http\Controllers;

use App\Models\Disk;
use App\Models\DiskFolder;
use App\Models\User;
use App\Support\AccessControl;
use App\Support\CrmModuleManager;
use App\Support\DiskFileManager;
use App\Support\SectionAccessManager;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response as HttpResponse;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;
use RuntimeException;
use Symfony\Component\HttpFoundation\StreamedResponse;

class DiskController extends Controller
{
    public function index(Request $request, SectionAccessManager $sectionAccessManager): View
    {
        $this->authorize('viewAny', Disk::class);

        $user = $request->user();
        $isElevated = AccessControl::isElevated($user);
        $search = trim((string) $request->input('q', ''));
        $folder = trim((string) $request->input('folder', ''));
        $ownerId = $isElevated ? $request->integer('owner_id') : (int) $user->id;

        $query = Disk::query()
            ->with('owner:id,name,email')
            ->when($search !== '', function ($builder) use ($search): void {
                $builder->where(function ($sub) use ($search): void {
                    $sub->where('name', 'like', "%{$search}%")
                        ->orWhere('original_name', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%")
                        ->orWhere('folder', 'like', "%{$search}%");
                });
            });

        if ($folder !== '') {
            $query->where('folder', $folder);
        }

        $query->when($ownerId > 0, fn ($builder) => $builder->where('owner_id', $ownerId));

        if (! $isElevated) {
            $query->where('owner_id', $user->id);
        }

        $disks = $query
            ->orderByDesc('updated_at')
            ->orderByDesc('id')
            ->get();

        $folderCatalogs = DiskFolder::query()
            ->when(! $isElevated, fn ($builder) => $builder->where('owner_id', $user->id))
            ->when($ownerId > 0, fn ($builder) => $builder->where('owner_id', $ownerId))
            ->orderBy('owner_id')
            ->orderBy('path')
            ->get(['id', 'name', 'path', 'parent_path', 'owner_id']);

        $fileFolderRows = Disk::query()
            ->when(! $isElevated, fn ($builder) => $builder->where('owner_id', $user->id))
            ->when($ownerId > 0, fn ($builder) => $builder->where('owner_id', $ownerId))
            ->whereNotNull('folder')
            ->where('folder', '!=', '')
            ->select(['owner_id', 'folder'])
            ->distinct()
            ->orderBy('owner_id')
            ->orderBy('folder')
            ->get()
            ->map(static fn (Disk $disk): array => [
                'owner_id' => (int) $disk->owner_id,
                'path' => trim((string) ($disk->folder ?? '')),
            ])
            ->filter(static fn (array $item): bool => $item['path'] !== '')
            ->values();

        $availableFolders = $folderCatalogs
            ->pluck('path')
            ->merge($fileFolderRows->pluck('path'))
            ->unique()
            ->sort()
            ->values();

        $folderFileCounts = Disk::query()
            ->when(! $isElevated, fn ($builder) => $builder->where('owner_id', $user->id))
            ->when($ownerId > 0, fn ($builder) => $builder->where('owner_id', $ownerId))
            ->whereNotNull('folder')
            ->where('folder', '!=', '')
            ->selectRaw('owner_id, folder, count(*) as aggregate')
            ->groupBy('owner_id', 'folder')
            ->get()
            ->mapWithKeys(static fn (Disk $disk): array => [
                ((int) $disk->owner_id).':'.trim((string) ($disk->folder ?? '')) => (int) ($disk->aggregate ?? 0),
            ]);

        $folderItems = $this->buildFolderItems($folderCatalogs, $fileFolderRows);

        $canManageSectionAccess = $sectionAccessManager->canManage($user, 'disks');
        $sectionAccessUsers = $canManageSectionAccess
            ? User::query()->orderBy('name')->get(['id', 'name', 'email', 'role', 'permissions'])
            : collect();
        $ownerOptions = $isElevated
            ? User::query()->orderBy('name')->get(['id', 'name'])
            : collect([$user->only(['id', 'name'])]);

        return view('disks.index', [
            'disks' => $disks,
            'search' => $search,
            'folder' => $folder,
            'ownerId' => $ownerId,
            'availableFolders' => $availableFolders,
            'folderItems' => $folderItems,
            'folderFileCounts' => $folderFileCounts,
            'ownerOptions' => $ownerOptions,
            'isElevated' => $isElevated,
            'canManageSectionAccess' => $canManageSectionAccess,
            'sectionAccessUsers' => $sectionAccessUsers,
        ]);
    }

    public function storeFolder(Request $request, DiskFileManager $diskFileManager, CrmModuleManager $moduleManager): RedirectResponse
    {
        $this->authorize('create', Disk::class);

        $validated = $request->validate([
            'catalog_name' => ['required', 'string', 'max:120'],
            'parent_folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
        ]);
        $validated = $moduleManager->applyPayloadHooks('disks.folders.store', $validated, [
            'hook' => 'disks.folders.store',
            'user_id' => $request->user()->id,
        ], array_keys($validated));

        $name = trim((string) $validated['catalog_name']);
        $name = str_replace(['\\', '/'], ' ', $name);
        $name = preg_replace('/\s+/', ' ', $name) ?? '';
        $name = trim($name);

        if ($name === '') {
            return redirect()
                ->route('disks.index')
                ->withErrors(['catalog_name' => __('Catalog name is required.')])
                ->withInput();
        }

        $parentFolder = $diskFileManager->normalizeFolderPath((string) ($validated['parent_folder'] ?? ''));
        $path = $parentFolder !== '' ? $parentFolder.'/'.$name : $name;
        $path = $diskFileManager->normalizeFolderPath($path);

        if ($path === '') {
            return redirect()
                ->route('disks.index')
                ->withErrors(['catalog_name' => __('Catalog path is invalid.')])
                ->withInput();
        }

        DiskFolder::query()->updateOrCreate(
            [
                'owner_id' => $request->user()->id,
                'path' => $path,
            ],
            [
                'name' => $name,
                'parent_path' => $parentFolder !== '' ? $parentFolder : null,
            ]
        );

        Storage::disk('public')->makeDirectory('disk/'.$path);

        return redirect()
            ->route('disks.index', ['folder' => $path])
            ->with('success', 'Catalog created.');
    }

    public function store(Request $request, DiskFileManager $diskFileManager, CrmModuleManager $moduleManager): RedirectResponse
    {
        $this->authorize('create', Disk::class);

        $validated = $request->validate([
            'file' => ['required', 'file', 'max:51200'],
            'name' => ['nullable', 'string', 'max:255'],
            'folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
            'description' => ['nullable', 'string', 'max:5000'],
            'is_public' => ['nullable', 'boolean'],
        ]);
        $metadataPayload = [
            'name' => $validated['name'] ?? '',
            'folder' => $validated['folder'] ?? '',
            'description' => $validated['description'] ?? null,
            'is_public' => $request->boolean('is_public'),
        ];
        $metadataPayload = $moduleManager->applyPayloadHooks('disks.store', $metadataPayload, [
            'hook' => 'disks.store',
            'user_id' => $request->user()->id,
        ], array_keys($metadataPayload));

        $diskFileManager->createFromUpload($validated['file'], $request->user(), [
            'name' => $metadataPayload['name'] ?? '',
            'folder' => $metadataPayload['folder'] ?? '',
            'description' => $metadataPayload['description'] ?? null,
            'is_public' => (bool) ($metadataPayload['is_public'] ?? false),
        ]);

        return redirect()
            ->route('disks.index')
            ->with('success', 'File uploaded to Disk.');
    }

    public function update(Request $request, Disk $disk, DiskFileManager $diskFileManager, CrmModuleManager $moduleManager): RedirectResponse
    {
        $this->authorize('update', $disk);

        $validated = $request->validate([
            'file' => ['nullable', 'file', 'max:51200'],
            'name' => ['nullable', 'string', 'max:255'],
            'folder' => ['nullable', 'string', 'max:180', 'not_regex:/\.\./'],
            'description' => ['nullable', 'string', 'max:5000'],
            'is_public' => ['nullable', 'boolean'],
        ]);

        $hookPayload = [];
        if (array_key_exists('name', $validated)) {
            $hookPayload['name'] = $validated['name'];
        }
        if (array_key_exists('folder', $validated)) {
            $hookPayload['folder'] = $validated['folder'];
        }
        if (array_key_exists('description', $validated)) {
            $hookPayload['description'] = $validated['description'];
        }
        if ($request->has('is_public')) {
            $hookPayload['is_public'] = $request->boolean('is_public');
        }
        if ($hookPayload !== []) {
            $hookPayload = $moduleManager->applyPayloadHooks('disks.update', $hookPayload, [
                'hook' => 'disks.update',
                'user_id' => $request->user()->id,
                'disk_id' => $disk->id,
            ], array_keys($hookPayload));
        }

        $diskFileManager->updateFromPayload(
            $disk,
            $hookPayload,
            $validated['file'] ?? null
        );

        return redirect()
            ->route('disks.index')
            ->with('success', 'Disk file updated.');
    }

    public function destroy(Disk $disk, DiskFileManager $diskFileManager): RedirectResponse
    {
        $this->authorize('delete', $disk);

        $diskFileManager->delete($disk);

        return redirect()
            ->route('disks.index')
            ->with('success', 'Disk file deleted.');
    }

    public function download(Disk $disk, DiskFileManager $diskFileManager): StreamedResponse
    {
        $this->authorize('view', $disk);

        return $diskFileManager->downloadResponse($disk);
    }

    public function inline(Disk $disk, DiskFileManager $diskFileManager): StreamedResponse
    {
        $this->authorize('view', $disk);

        return $diskFileManager->inlineResponse($disk);
    }

    public function wordSource(Disk $disk, DiskFileManager $diskFileManager): HttpResponse|StreamedResponse
    {
        $this->authorize('view', $disk);

        try {
            return $diskFileManager->wordPreviewSourceResponse($disk);
        } catch (RuntimeException $exception) {
            return response($exception->getMessage(), 422, [
                'Content-Type' => 'text/plain; charset=UTF-8',
            ]);
        }
    }

    public function spreadsheetSource(Disk $disk, DiskFileManager $diskFileManager): HttpResponse|StreamedResponse
    {
        $this->authorize('view', $disk);

        try {
            return $diskFileManager->spreadsheetPreviewSourceResponse($disk);
        } catch (RuntimeException $exception) {
            return response($exception->getMessage(), 422, [
                'Content-Type' => 'text/plain; charset=UTF-8',
            ]);
        }
    }

    public function preview(Disk $disk, DiskFileManager $diskFileManager): View
    {
        $this->authorize('view', $disk);

        $preview = [
            'kind' => 'unsupported',
            'title' => 'Preview unavailable',
            'html' => null,
            'message' => 'Built-in preview supports PDF, DOC, DOCX, XLSX, XLS and CSV.',
        ];

        try {
            $preview = $diskFileManager->buildPreviewPayload($disk);
        } catch (RuntimeException $exception) {
            $preview = [
                'kind' => 'unsupported',
                'title' => 'Preview unavailable',
                'html' => null,
                'message' => $exception->getMessage(),
            ];
        }

        return view('disks.preview', [
            'disk' => $disk,
            'preview' => $preview,
        ]);
    }

    /**
     * @param  Collection<int, DiskFolder>  $folderCatalogs
     * @param  Collection<int, array{owner_id:int,path:string}>  $fileFolderRows
     * @return Collection<int, array{key:string,path:string,name:string,is_catalog:bool,owner_id:int,parent_path:?string}>
     */
    private function buildFolderItems(Collection $folderCatalogs, Collection $fileFolderRows): Collection
    {
        $catalogItems = $folderCatalogs->map(static function (DiskFolder $folder): array {
            $path = (string) $folder->path;
            $ownerId = (int) $folder->owner_id;

            return [
                'key' => $ownerId.':'.$path,
                'path' => $path,
                'name' => (string) $folder->name,
                'is_catalog' => true,
                'owner_id' => $ownerId,
                'parent_path' => $folder->parent_path ? (string) $folder->parent_path : null,
            ];
        })->keyBy(static fn (array $item): string => $item['key']);

        foreach ($fileFolderRows as $row) {
            $normalizedPath = trim((string) ($row['path'] ?? ''));
            $rowOwnerId = (int) ($row['owner_id'] ?? 0);
            $key = $rowOwnerId.':'.$normalizedPath;

            if ($catalogItems->has($key)) {
                continue;
            }

            if ($normalizedPath === '') {
                continue;
            }

            $segments = explode('/', $normalizedPath);
            $catalogItems->put($key, [
                'key' => $key,
                'path' => $normalizedPath,
                'name' => (string) end($segments),
                'is_catalog' => false,
                'owner_id' => $rowOwnerId,
                'parent_path' => str_contains($normalizedPath, '/')
                    ? trim((string) dirname($normalizedPath), '/.')
                    : null,
            ]);
        }

        return $catalogItems
            ->sortBy(static fn (array $item): string => ((int) $item['owner_id']).':'.$item['path'])
            ->values();
    }
}
