<?php

declare(strict_types=1);

function update_center_bootstrap(string $rootPath): array
{
    $storagePath = $rootPath.'/storage';
    $packagePath = $storagePath.'/packages';
    $dbPath = $storagePath.'/update-center.sqlite';

    foreach ([$storagePath, $packagePath] as $path) {
        if (! is_dir($path)) {
            mkdir($path, 0775, true);
        }
    }

    $config = [
        'app_name' => 'CRM25 Update Center',
        'base_url' => update_center_detect_base_url(),
        'default_product' => 'crm25',
        'admin_user' => 'crm25',
        'admin_password' => 'change-me-now',
        'db_path' => $dbPath,
        'package_path' => $packagePath,
        'max_upload_bytes' => 1024 * 1024 * 1024,
    ];

    $configFile = $rootPath.'/config.local.php';
    if (is_file($configFile)) {
        $overrides = require $configFile;
        if (is_array($overrides)) {
            $config = array_replace($config, $overrides);
        }
    }

    $pdo = new PDO('sqlite:'.$config['db_path']);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    $pdo->exec('PRAGMA foreign_keys = ON');

    $config['pdo'] = $pdo;

    return $config;
}

function update_center_detect_base_url(): string
{
    $scheme = (! empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = (string) ($_SERVER['HTTP_HOST'] ?? 'localhost');

    return $scheme.'://'.$host;
}

function update_center_escape(?string $value): string
{
    return htmlspecialchars((string) $value, ENT_QUOTES, 'UTF-8');
}

function update_center_json(array $payload, int $status = 200): never
{
    http_response_code($status);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    exit;
}

function update_center_slug(string $value): string
{
    $slug = strtolower(trim($value));
    $slug = preg_replace('/[^a-z0-9]+/', '-', $slug) ?: 'release';
    $slug = trim($slug, '-');

    return $slug !== '' ? $slug : 'release';
}

function update_center_is_version_comparable(string $version): bool
{
    return (bool) preg_match('/^\d+(\.\d+){1,3}([\-+][A-Za-z0-9.\-]+)?$/', trim($version));
}

function update_center_is_update_available(string $currentVersion, string $remoteVersion): bool
{
    $currentVersion = trim($currentVersion);
    $remoteVersion = trim($remoteVersion);

    if ($currentVersion === '' || $remoteVersion === '') {
        return false;
    }

    if (update_center_is_version_comparable($currentVersion) && update_center_is_version_comparable($remoteVersion)) {
        return version_compare($remoteVersion, $currentVersion, '>');
    }

    return $remoteVersion !== $currentVersion;
}
