<?php

declare(strict_types=1);

final class UpdateReleaseRepository
{
    public function __construct(
        private readonly PDO $pdo,
        private readonly string $baseUrl
    ) {
        $this->ensureSchema();
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function all(): array
    {
        $statement = $this->pdo->query(
            'SELECT * FROM releases ORDER BY datetime(published_at) DESC, id DESC'
        );

        return array_map(
            fn (array $row): array => $this->mapRow($row),
            $statement->fetchAll() ?: []
        );
    }

    public function latest(string $product, string $channel): ?array
    {
        $statement = $this->pdo->prepare(
            'SELECT * FROM releases
             WHERE product = :product
               AND channel = :channel
               AND is_active = 1
             ORDER BY datetime(published_at) DESC, id DESC
             LIMIT 1'
        );
        $statement->execute([
            'product' => $product,
            'channel' => $channel,
        ]);

        $row = $statement->fetch();

        return is_array($row) ? $this->mapRow($row) : null;
    }

    public function find(int $id): ?array
    {
        $statement = $this->pdo->prepare('SELECT * FROM releases WHERE id = :id LIMIT 1');
        $statement->execute(['id' => $id]);
        $row = $statement->fetch();

        return is_array($row) ? $this->mapRow($row) : null;
    }

    /**
     * @param  array<string, mixed>  $payload
     */
    public function create(array $payload): int
    {
        $statement = $this->pdo->prepare(
            'INSERT INTO releases (
                product,
                version,
                build,
                channel,
                notes,
                file_name,
                original_file_name,
                file_path,
                file_size,
                checksum_sha256,
                is_active,
                published_at,
                created_at,
                updated_at
             ) VALUES (
                :product,
                :version,
                :build,
                :channel,
                :notes,
                :file_name,
                :original_file_name,
                :file_path,
                :file_size,
                :checksum_sha256,
                :is_active,
                :published_at,
                :created_at,
                :updated_at
             )'
        );

        $now = gmdate('c');
        $statement->execute([
            'product' => (string) $payload['product'],
            'version' => (string) $payload['version'],
            'build' => (string) ($payload['build'] ?? ''),
            'channel' => (string) $payload['channel'],
            'notes' => (string) ($payload['notes'] ?? ''),
            'file_name' => (string) $payload['file_name'],
            'original_file_name' => (string) $payload['original_file_name'],
            'file_path' => (string) $payload['file_path'],
            'file_size' => (int) $payload['file_size'],
            'checksum_sha256' => (string) $payload['checksum_sha256'],
            'is_active' => ! empty($payload['is_active']) ? 1 : 0,
            'published_at' => (string) ($payload['published_at'] ?? $now),
            'created_at' => $now,
            'updated_at' => $now,
        ]);

        return (int) $this->pdo->lastInsertId();
    }

    public function incrementDownloadCount(int $id): void
    {
        $statement = $this->pdo->prepare(
            'UPDATE releases
             SET download_count = download_count + 1, updated_at = :updated_at
             WHERE id = :id'
        );
        $statement->execute([
            'id' => $id,
            'updated_at' => gmdate('c'),
        ]);
    }

    private function ensureSchema(): void
    {
        $this->pdo->exec(
            'CREATE TABLE IF NOT EXISTS releases (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                product TEXT NOT NULL DEFAULT "crm25",
                version TEXT NOT NULL,
                build TEXT NOT NULL DEFAULT "",
                channel TEXT NOT NULL DEFAULT "stable",
                notes TEXT NOT NULL DEFAULT "",
                file_name TEXT NOT NULL,
                original_file_name TEXT NOT NULL,
                file_path TEXT NOT NULL,
                file_size INTEGER NOT NULL DEFAULT 0,
                checksum_sha256 TEXT NOT NULL,
                download_count INTEGER NOT NULL DEFAULT 0,
                is_active INTEGER NOT NULL DEFAULT 1,
                published_at TEXT NOT NULL,
                created_at TEXT NOT NULL,
                updated_at TEXT NOT NULL
            )'
        );
        $this->pdo->exec(
            'CREATE INDEX IF NOT EXISTS idx_releases_product_channel_published
             ON releases(product, channel, published_at DESC, id DESC)'
        );
    }

    /**
     * @param  array<string, mixed>  $row
     * @return array<string, mixed>
     */
    private function mapRow(array $row): array
    {
        $id = (int) $row['id'];

        return [
            'id' => $id,
            'product' => (string) $row['product'],
            'version' => (string) $row['version'],
            'build' => (string) $row['build'],
            'channel' => (string) $row['channel'],
            'notes' => (string) $row['notes'],
            'file_name' => (string) $row['file_name'],
            'original_file_name' => (string) $row['original_file_name'],
            'file_path' => (string) $row['file_path'],
            'file_size' => (int) $row['file_size'],
            'checksum_sha256' => (string) $row['checksum_sha256'],
            'download_count' => (int) $row['download_count'],
            'is_active' => (bool) $row['is_active'],
            'published_at' => (string) $row['published_at'],
            'created_at' => (string) $row['created_at'],
            'updated_at' => (string) $row['updated_at'],
            'download_url' => rtrim($this->baseUrl, '/').'/downloads/'.$id.'/'.rawurlencode((string) $row['file_name']),
        ];
    }
}
