<?php

declare(strict_types=1);

session_start();

$rootPath = dirname(__DIR__);

require $rootPath.'/src/bootstrap.php';
require $rootPath.'/src/UpdateReleaseRepository.php';

$config = update_center_bootstrap($rootPath);
$repository = new UpdateReleaseRepository($config['pdo'], $config['base_url']);

$path = parse_url((string) ($_SERVER['REQUEST_URI'] ?? '/'), PHP_URL_PATH) ?: '/';
$method = strtoupper((string) ($_SERVER['REQUEST_METHOD'] ?? 'GET'));

try {
    if ($path === '/health') {
        update_center_json([
            'status' => 'ok',
            'app' => $config['app_name'],
            'time' => gmdate('c'),
        ]);
    }

    if ($path === '/api/v1/crm25/updates' || $path === '/api/v1/updates' || $path === '/api/updates' || $path === '/updates.json') {
        $product = trim((string) ($_GET['product'] ?? $config['default_product']));
        $channel = trim((string) ($_GET['channel'] ?? 'stable'));
        $currentVersion = trim((string) ($_GET['version'] ?? ''));

        if (! in_array($channel, ['stable', 'beta', 'nightly'], true)) {
            $channel = 'stable';
        }

        $release = $repository->latest($product !== '' ? $product : $config['default_product'], $channel);
        if ($release === null) {
            update_center_json([
                'message' => 'No published updates found for the requested product/channel.',
                'product' => $product !== '' ? $product : $config['default_product'],
                'channel' => $channel,
                'is_update_available' => false,
            ], 404);
        }

        update_center_json([
            'product' => $release['product'],
            'channel' => $release['channel'],
            'version' => $release['version'],
            'latest_version' => $release['version'],
            'build' => $release['build'],
            'latest_build' => $release['build'],
            'notes' => $release['notes'],
            'published_at' => $release['published_at'],
            'download_url' => $release['download_url'],
            'checksum_sha256' => $release['checksum_sha256'],
            'file_name' => $release['original_file_name'],
            'file_size' => $release['file_size'],
            'is_update_available' => $currentVersion !== ''
                ? update_center_is_update_available($currentVersion, $release['version'])
                : true,
        ]);
    }

    if (preg_match('#^/downloads/(\d+)/#', $path, $matches) === 1) {
        $release = $repository->find((int) $matches[1]);
        if ($release === null || ! is_file($release['file_path'])) {
            http_response_code(404);
            echo 'Release package not found.';
            exit;
        }

        $repository->incrementDownloadCount((int) $release['id']);

        header('Content-Type: application/zip');
        header('Content-Length: '.filesize($release['file_path']));
        header('Content-Disposition: attachment; filename="'.basename((string) $release['original_file_name']).'"');
        readfile($release['file_path']);
        exit;
    }

    if (str_starts_with($path, '/admin')) {
        update_center_require_admin($config);

        if (empty($_SESSION['update_center_csrf'])) {
            $_SESSION['update_center_csrf'] = bin2hex(random_bytes(24));
        }

        if ($method === 'POST' && $path === '/admin/releases') {
            update_center_assert_csrf();
            $releaseId = update_center_handle_upload($repository, $config);
            header('Location: /admin?created='.$releaseId);
            exit;
        }

        $releases = $repository->all();
        echo update_center_render_admin($config, $releases);
        exit;
    }

    echo update_center_render_home($config, $repository->all());
} catch (Throwable $exception) {
    http_response_code(500);
    header('Content-Type: text/html; charset=utf-8');
    echo '<h1>Update Center Error</h1>';
    echo '<pre>'.update_center_escape($exception->getMessage()).'</pre>';
}

/**
 * @return int
 */
function update_center_handle_upload(UpdateReleaseRepository $repository, array $config): int
{
    $version = trim((string) ($_POST['version'] ?? ''));
    $build = trim((string) ($_POST['build'] ?? ''));
    $channel = trim((string) ($_POST['channel'] ?? 'stable'));
    $product = trim((string) ($_POST['product'] ?? $config['default_product']));
    $notes = trim((string) ($_POST['notes'] ?? ''));
    $isActive = isset($_POST['is_active']) && (string) $_POST['is_active'] === '1';

    if ($version === '') {
        throw new RuntimeException('Version is required.');
    }

    if (! in_array($channel, ['stable', 'beta', 'nightly'], true)) {
        throw new RuntimeException('Invalid channel.');
    }

    if (! isset($_FILES['package']) || ! is_array($_FILES['package'])) {
        throw new RuntimeException('Release package is required.');
    }

    $uploadedFile = $_FILES['package'];
    $errorCode = (int) ($uploadedFile['error'] ?? UPLOAD_ERR_NO_FILE);
    if ($errorCode !== UPLOAD_ERR_OK) {
        throw new RuntimeException('Upload failed with code '.$errorCode.'.');
    }

    $tmpName = (string) ($uploadedFile['tmp_name'] ?? '');
    $originalName = (string) ($uploadedFile['name'] ?? 'release.zip');
    $size = (int) ($uploadedFile['size'] ?? 0);

    if ($size <= 0) {
        throw new RuntimeException('Uploaded file is empty.');
    }

    if ($size > (int) $config['max_upload_bytes']) {
        throw new RuntimeException('Uploaded file exceeds the maximum allowed size.');
    }

    $extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
    if ($extension !== 'zip') {
        throw new RuntimeException('Only ZIP packages are supported.');
    }

    $safeVersion = update_center_slug($version);
    $safeOriginal = update_center_slug(pathinfo($originalName, PATHINFO_FILENAME)).'.zip';
    $storedFileName = gmdate('YmdHis').'-'.$safeVersion.'-'.$safeOriginal;
    $targetPath = rtrim((string) $config['package_path'], '/').'/'.$storedFileName;

    if (! move_uploaded_file($tmpName, $targetPath)) {
        throw new RuntimeException('Unable to move uploaded package.');
    }

    $releaseId = $repository->create([
        'product' => $product !== '' ? $product : $config['default_product'],
        'version' => $version,
        'build' => $build,
        'channel' => $channel,
        'notes' => $notes,
        'file_name' => $storedFileName,
        'original_file_name' => $originalName,
        'file_path' => $targetPath,
        'file_size' => filesize($targetPath) ?: $size,
        'checksum_sha256' => hash_file('sha256', $targetPath),
        'is_active' => $isActive,
        'published_at' => gmdate('c'),
    ]);

    return $releaseId;
}

function update_center_require_admin(array $config): void
{
    $expectedUser = (string) $config['admin_user'];
    $expectedPassword = (string) $config['admin_password'];
    $actualUser = (string) ($_SERVER['PHP_AUTH_USER'] ?? '');
    $actualPassword = (string) ($_SERVER['PHP_AUTH_PW'] ?? '');

    if ($actualUser === $expectedUser && hash_equals($expectedPassword, $actualPassword)) {
        return;
    }

    header('WWW-Authenticate: Basic realm="CRM25 Update Center"');
    http_response_code(401);
    echo 'Authentication required.';
    exit;
}

function update_center_assert_csrf(): void
{
    $sessionToken = (string) ($_SESSION['update_center_csrf'] ?? '');
    $requestToken = trim((string) ($_POST['_token'] ?? ''));

    if ($sessionToken === '' || ! hash_equals($sessionToken, $requestToken)) {
        throw new RuntimeException('Invalid CSRF token.');
    }
}

/**
 * @param  array<int, array<string, mixed>>  $releases
 */
function update_center_render_home(array $config, array $releases): string
{
    $latestStable = null;
    foreach ($releases as $release) {
        if ($release['channel'] === 'stable' && $release['is_active']) {
            $latestStable = $release;
            break;
        }
    }

    $baseUrl = rtrim((string) $config['base_url'], '/');
    $installScriptUrl = $baseUrl.'/install.sh';
    $installCommand = 'curl -fsSL '.$installScriptUrl.' | sudo bash';
    $feedExample = $baseUrl.'/api/v1/crm25/updates?product=crm25&channel=stable&version=1.007';

    ob_start();
    ?>
<!doctype html>
<html lang="ru">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title><?= update_center_escape($config['app_name']) ?></title>
    <style>
        :root {
            color-scheme: light;
            --bg: #f3f6fb;
            --panel: #ffffff;
            --line: #d6dfef;
            --ink: #11223d;
            --muted: #667792;
            --brand: #0f62fe;
            --brand-2: #1148b8;
            --soft: #e8f0ff;
        }
        * { box-sizing: border-box; }
        body {
            margin: 0;
            font-family: "Segoe UI", Arial, sans-serif;
            background:
                radial-gradient(circle at top right, rgba(15, 98, 254, 0.14), transparent 24rem),
                linear-gradient(180deg, #f7faff 0%, var(--bg) 100%);
            color: var(--ink);
        }
        .wrap { max-width: 1120px; margin: 0 auto; padding: 32px 24px 72px; }
        .hero, .panel {
            background: rgba(255,255,255,0.92);
            backdrop-filter: blur(10px);
            border: 1px solid var(--line);
            border-radius: 24px;
            box-shadow: 0 20px 60px rgba(17, 34, 61, 0.08);
        }
        .hero { padding: 32px; display: grid; gap: 22px; }
        .hero-grid { display: grid; gap: 16px; grid-template-columns: repeat(auto-fit, minmax(240px, 1fr)); }
        .badge { display: inline-flex; align-items: center; gap: 8px; width: fit-content; border-radius: 999px; padding: 8px 12px; background: var(--soft); color: var(--brand-2); font-size: 13px; font-weight: 700; letter-spacing: .04em; text-transform: uppercase; }
        h1 { margin: 0; font-size: clamp(32px, 5vw, 54px); line-height: 1.02; }
        h2 { margin: 0; font-size: clamp(24px, 3vw, 34px); }
        p { margin: 0; color: var(--muted); line-height: 1.6; }
        .cta { display: inline-flex; align-items: center; justify-content: center; border-radius: 14px; padding: 14px 18px; background: var(--brand); color: #fff; font-weight: 700; text-decoration: none; border: 0; cursor: pointer; }
        .cta-secondary { background: #fff; color: var(--ink); border: 1px solid var(--line); }
        .meta { display: grid; gap: 12px; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); }
        .meta-card, .release-card, .step-card, .doc-card {
            border: 1px solid var(--line);
            border-radius: 18px;
            background: var(--panel);
            padding: 18px;
        }
        .meta-label { color: var(--muted); font-size: 12px; text-transform: uppercase; letter-spacing: .08em; }
        .meta-value { margin-top: 8px; font-size: 20px; font-weight: 700; }
        .grid { margin-top: 24px; display: grid; gap: 18px; grid-template-columns: repeat(auto-fit, minmax(260px, 1fr)); }
        .panel { margin-top: 28px; padding: 24px; }
        .channel { display: inline-flex; border-radius: 999px; padding: 6px 10px; background: #edf4ff; color: var(--brand-2); font-size: 12px; font-weight: 700; text-transform: uppercase; }
        code { background: #eef3fb; border-radius: 8px; padding: 2px 6px; word-break: break-word; }
        .install-layout { display: grid; gap: 18px; grid-template-columns: minmax(0, 1.3fr) minmax(280px, 0.7fr); }
        .command-box {
            margin-top: 18px;
            border: 1px solid #c6d5f0;
            background: linear-gradient(180deg, #0d1a33 0%, #13284c 100%);
            color: #eff5ff;
            border-radius: 20px;
            padding: 18px;
            box-shadow: inset 0 1px 0 rgba(255,255,255,0.06);
        }
        .command-label { color: #9bb4df; font-size: 12px; text-transform: uppercase; letter-spacing: .14em; }
        .command-row { display: flex; gap: 12px; align-items: center; margin-top: 12px; flex-wrap: wrap; }
        .command-row code {
            display: block;
            flex: 1 1 420px;
            padding: 14px 16px;
            background: rgba(7, 16, 32, 0.76);
            color: #f7fbff;
            border: 1px solid rgba(255,255,255,0.12);
            border-radius: 14px;
        }
        .copy-feedback { display: inline-flex; min-height: 20px; align-items: center; color: #9bb4df; font-size: 13px; }
        .steps-grid { display: grid; gap: 14px; grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); margin-top: 20px; }
        .step-number {
            display: inline-flex;
            width: 32px;
            height: 32px;
            align-items: center;
            justify-content: center;
            border-radius: 999px;
            background: var(--soft);
            color: var(--brand-2);
            font-weight: 800;
        }
        .step-card h3 { margin: 14px 0 8px; font-size: 18px; }
        .check-list { display: grid; gap: 10px; margin-top: 16px; }
        .check-item { display: flex; gap: 10px; align-items: flex-start; color: var(--ink); }
        .check-dot {
            width: 10px;
            height: 10px;
            margin-top: 8px;
            border-radius: 999px;
            background: linear-gradient(135deg, #0f62fe 0%, #2e8fff 100%);
            flex: 0 0 auto;
        }
        .release-actions { display: flex; gap: 12px; flex-wrap: wrap; margin-top: 18px; }
        .docs-grid {
            display: grid;
            gap: 18px;
            grid-template-columns: repeat(auto-fit, minmax(260px, 1fr));
            margin-top: 18px;
        }
        .doc-card h3 { margin: 0 0 12px; font-size: 18px; }
        .doc-list { margin: 0; padding-left: 18px; color: var(--muted); line-height: 1.7; }
        .doc-list li + li { margin-top: 6px; }
        .code-stack {
            display: grid;
            gap: 10px;
            margin-top: 14px;
        }
        .code-stack code {
            display: block;
            padding: 12px 14px;
            border: 1px solid var(--line);
            background: #f5f8fe;
        }
        @media (max-width: 980px) {
            .install-layout { grid-template-columns: 1fr; }
        }
        @media (max-width: 640px) {
            .wrap { padding: 16px; }
            .hero, .panel { border-radius: 18px; }
            .command-row { align-items: stretch; }
        }
    </style>
</head>
<body>
    <main class="wrap">
        <section class="hero">
            <span class="badge">CRM25 Update Center</span>
            <div class="hero-grid">
                <div>
                    <h1>Установка CRM25 на чистый Linux-сервер одной командой</h1>
                    <p>Сервер обновлений публикует релизы CRM25 и готовый инсталлятор. На голом Debian/Ubuntu он сам ставит окружение, настраивает MySQL, Redis, Nginx, очереди, websocket и поднимает CRM.</p>
                </div>
                <div class="meta-card">
                    <div class="meta-label">Команда установки</div>
                    <div class="meta-value"><code>curl … | sudo bash</code></div>
                    <p style="margin-top:10px;">Во время установки спрашивается только домен. Если просто нажать Enter, CRM ставится на текущий IP сервера.</p>
                </div>
            </div>
            <div class="meta">
                <div class="meta-card">
                    <div class="meta-label">Источник обновлений</div>
                    <div class="meta-value"><?= update_center_escape($baseUrl) ?></div>
                </div>
                <div class="meta-card">
                    <div class="meta-label">Последняя стабильная версия</div>
                    <div class="meta-value"><?= update_center_escape($latestStable['version'] ?? 'Not published yet') ?></div>
                </div>
                <div class="meta-card">
                    <div class="meta-label">Опубликовано релизов</div>
                    <div class="meta-value"><?= count($releases) ?></div>
                </div>
            </div>
        </section>

        <section class="panel">
            <div class="install-layout">
                <div>
                    <div>
                        <h2>Быстрая установка</h2>
                        <p style="margin-top:10px;">Инсталлятор скачивает текущий релиз, устанавливает пакеты, создает MySQL-базу, защищает Redis, настраивает `.env`, запускает миграции, создает администратора, включает `queue:work`, `reverb:start` и `schedule:run`.</p>
                    </div>

                    <div class="command-box">
                        <div class="command-label">One-liner для сервера</div>
                        <div class="command-row">
                            <code id="install-command"><?= update_center_escape($installCommand) ?></code>
                            <button type="button" class="cta" data-copy-command="<?= update_center_escape($installCommand) ?>">Скопировать команду</button>
                        </div>
                        <div class="command-row" style="margin-top:10px;">
                            <span class="copy-feedback" data-copy-feedback>Команда готова для вставки в терминал root-пользователя.</span>
                        </div>
                        <div class="command-row" style="margin-top:14px;">
                            <span style="color:#c9daf8;">Доступы администратора после установки сохраняются в <code>/root/crm25-install-credentials.txt</code>.</span>
                        </div>
                        <div class="command-row" style="margin-top:8px;">
                            <span style="color:#c9daf8;">Полный лог установки сохраняется в <code>/var/log/crm25-installer.log</code>.</span>
                        </div>
                    </div>

                    <div class="steps-grid">
                        <article class="step-card">
                            <span class="step-number">1</span>
                            <h3>Запуск</h3>
                            <p>Выполните команду на чистом сервере Debian/Ubuntu под `root` или через `sudo`.</p>
                        </article>
                        <article class="step-card">
                            <span class="step-number">2</span>
                            <h3>Домен или IP</h3>
                            <p>Инсталлятор спрашивает домен. Если нажать Enter без ввода, он ставит CRM на текущий публичный IP сервера.</p>
                        </article>
                        <article class="step-card">
                            <span class="step-number">3</span>
                            <h3>Готовое окружение</h3>
                            <p>После установки будут подняты Nginx, PHP-FPM, MySQL, Redis, Supervisor, cron и realtime через Reverb.</p>
                        </article>
                    </div>
                </div>

                <div class="meta-card">
                    <div class="meta-label">Что ставится автоматически</div>
                    <div class="check-list">
                        <div class="check-item"><span class="check-dot"></span><span>PHP 8.4, PHP-FPM, Composer, Node.js, Nginx</span></div>
                        <div class="check-item"><span class="check-dot"></span><span>MySQL с отдельной базой и пользователем</span></div>
                        <div class="check-item"><span class="check-dot"></span><span>Redis с паролем и локальным bind</span></div>
                        <div class="check-item"><span class="check-dot"></span><span>Supervisor для очереди и websocket</span></div>
                        <div class="check-item"><span class="check-dot"></span><span>Cron для `schedule:run`</span></div>
                        <div class="check-item"><span class="check-dot"></span><span>Администратор CRM и стартовая воронка сделок</span></div>
                    </div>

                    <div style="margin-top:18px;">
                        <div class="meta-label">Скрипт установки</div>
                        <div style="margin-top:10px;"><code><?= update_center_escape($installScriptUrl) ?></code></div>
                    </div>

                    <div style="margin-top:18px;">
                        <div class="meta-label">Feed для CRM</div>
                        <div style="margin-top:10px;"><code><?= update_center_escape($feedExample) ?></code></div>
                    </div>

                    <div class="release-actions">
                        <?php if ($latestStable !== null): ?>
                            <a class="cta" href="<?= update_center_escape($latestStable['download_url']) ?>">Скачать текущий ZIP</a>
                        <?php endif; ?>
                        <a class="cta cta-secondary" href="<?= update_center_escape($installScriptUrl) ?>">Открыть install.sh</a>
                    </div>
                </div>
            </div>
        </section>

        <section class="panel">
            <div style="display:flex; justify-content:space-between; gap:16px; align-items:end; flex-wrap:wrap;">
                <div>
                    <h2>Документация по установке</h2>
                    <p>Страница повторяет сценарий из серверной документации: установка на чистый Linux, одна команда, один вопрос о домене и полный автоконфиг окружения.</p>
                </div>
                <div><code>Поддержка: Debian / Ubuntu + apt</code></div>
            </div>

            <div class="docs-grid">
                <article class="doc-card">
                    <h3>Поддерживаемый сценарий</h3>
                    <ul class="doc-list">
                        <li>Чистый Linux-сервер на Debian или Ubuntu.</li>
                        <li>Запуск под `root` или через `sudo bash`.</li>
                        <li>Во время установки задается только домен.</li>
                        <li>Если домен не введен, CRM ставится на текущий публичный IP.</li>
                    </ul>
                </article>

                <article class="doc-card">
                    <h3>Что делает инсталлятор</h3>
                    <ul class="doc-list">
                        <li>Скачивает актуальный релиз CRM25 из центра обновлений.</li>
                        <li>Ставит PHP 8.4, Nginx, PHP-FPM, MySQL, Redis, Supervisor, cron и Node.js.</li>
                        <li>Создает базу MySQL, пользователя БД и защищает Redis.</li>
                        <li>Настраивает `.env`, миграции, очереди, realtime и frontend.</li>
                        <li>Если релиз уже содержит собранный `public/build`, серверная сборка frontend пропускается автоматически.</li>
                    </ul>
                </article>

                <article class="doc-card">
                    <h3>После установки</h3>
                    <ul class="doc-list">
                        <li>URL CRM выводится в консоль по завершении установки.</li>
                        <li>Учетные данные администратора сохраняются в <code>/root/crm25-install-credentials.txt</code>.</li>
                        <li>Полный лог установки сохраняется в <code>/var/log/crm25-installer.log</code>.</li>
                        <li>При домене установщик пытается выпустить Let's Encrypt сертификат.</li>
                        <li>Если сертификат не выпустился, установка завершается по HTTP без остановки процесса.</li>
                    </ul>
                </article>
            </div>

            <div class="docs-grid">
                <article class="doc-card">
                    <h3>Команды запуска</h3>
                    <p>Основной сценарий использует установщик напрямую с сервера обновлений. Дополнительно доступны режимы для локального файла и без запроса домена.</p>
                    <div class="code-stack">
                        <code><?= update_center_escape($installCommand) ?></code>
                        <code>sudo CRM25_DOMAIN=crm.example.com bash scripts/install-crm25.sh</code>
                        <code>sudo CRM25_INSTALL_FORCE=1 bash scripts/install-crm25.sh</code>
                    </div>
                </article>

                <article class="doc-card">
                    <h3>Требования и ограничения</h3>
                    <ul class="doc-list">
                        <li>Нужен доступ сервера в интернет для `apt`, `npm` и `update.crm25.webnet.kz`.</li>
                        <li>Если домен не указывает на сервер, HTTPS не выпустится автоматически.</li>
                        <li>Инсталлятор рассчитан на первичную установку на свободный сервер.</li>
                        <li>Для принудительной переустановки в занятый каталог используйте `CRM25_INSTALL_FORCE=1`.</li>
                    </ul>
                </article>
            </div>
        </section>

        <section class="panel">
            <div style="display:flex; justify-content:space-between; gap:16px; align-items:end; flex-wrap:wrap;">
                <div>
                    <h2>История релизов</h2>
                    <p>Каждый релиз доступен как ZIP-пакет и как машинно-читаемая запись в feed.</p>
                </div>
                <div><code>GET <?= update_center_escape($feedExample) ?></code></div>
            </div>
            <div class="grid">
                <?php foreach ($releases as $release): ?>
                    <article class="release-card">
                        <div style="display:flex; justify-content:space-between; gap:12px; align-items:center;">
                            <strong style="font-size:22px;"><?= update_center_escape($release['version']) ?></strong>
                            <span class="channel"><?= update_center_escape($release['channel']) ?></span>
                        </div>
                        <p style="margin-top:10px;"><?= nl2br(update_center_escape($release['notes'] !== '' ? $release['notes'] : 'Описание релиза не заполнено.')) ?></p>
                        <div style="display:grid; gap:8px; margin-top:16px;">
                            <span><strong>Build:</strong> <?= update_center_escape($release['build'] !== '' ? $release['build'] : 'n/a') ?></span>
                            <span><strong>Product:</strong> <?= update_center_escape($release['product']) ?></span>
                            <span><strong>Published:</strong> <?= update_center_escape($release['published_at']) ?></span>
                            <span><strong>File:</strong> <?= update_center_escape($release['original_file_name']) ?></span>
                            <span><strong>SHA256:</strong> <code><?= update_center_escape($release['checksum_sha256']) ?></code></span>
                        </div>
                        <div style="margin-top:16px;">
                            <a class="cta" href="<?= update_center_escape($release['download_url']) ?>">Скачать ZIP</a>
                        </div>
                    </article>
                <?php endforeach; ?>
                <?php if ($releases === []): ?>
                    <article class="release-card">
                        <strong>Релизы пока не опубликованы</strong>
                        <p style="margin-top:10px;">Откройте <code>/admin</code>, загрузите первый ZIP-пакет и feed станет доступен сразу.</p>
                    </article>
                <?php endif; ?>
            </div>
        </section>
    </main>
    <script>
        (() => {
            const copyButton = document.querySelector('[data-copy-command]');
            const feedback = document.querySelector('[data-copy-feedback]');

            if (!(copyButton instanceof HTMLButtonElement)) {
                return;
            }

            const setFeedback = (message) => {
                if (feedback instanceof HTMLElement) {
                    feedback.textContent = message;
                }
            };

            copyButton.addEventListener('click', async () => {
                const command = String(copyButton.dataset.copyCommand || '').trim();
                if (command === '') {
                    return;
                }

                try {
                    if (navigator.clipboard && typeof navigator.clipboard.writeText === 'function') {
                        await navigator.clipboard.writeText(command);
                    } else {
                        const field = document.createElement('textarea');
                        field.value = command;
                        field.setAttribute('readonly', 'readonly');
                        field.style.position = 'absolute';
                        field.style.left = '-9999px';
                        document.body.appendChild(field);
                        field.select();
                        document.execCommand('copy');
                        document.body.removeChild(field);
                    }

                    setFeedback('Команда скопирована. Вставьте ее в терминал сервера и запустите.');
                } catch (error) {
                    setFeedback('Не удалось скопировать автоматически. Скопируйте команду вручную из блока выше.');
                }
            });
        })();
    </script>
</body>
</html>
    <?php

    return (string) ob_get_clean();
}

/**
 * @param  array<int, array<string, mixed>>  $releases
 */
function update_center_render_admin(array $config, array $releases): string
{
    $createdId = isset($_GET['created']) ? (int) $_GET['created'] : null;
    $token = (string) $_SESSION['update_center_csrf'];

    ob_start();
    ?>
<!doctype html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title><?= update_center_escape($config['app_name']) ?> Admin</title>
    <style>
        body {
            margin: 0;
            font-family: "Segoe UI", Arial, sans-serif;
            background: linear-gradient(180deg, #081121 0%, #10233f 100%);
            color: #eaf1fb;
        }
        * { box-sizing: border-box; }
        .wrap { max-width: 1240px; margin: 0 auto; padding: 28px 20px 48px; }
        .card {
            background: rgba(255,255,255,0.08);
            border: 1px solid rgba(255,255,255,0.12);
            border-radius: 24px;
            padding: 22px;
            backdrop-filter: blur(12px);
            box-shadow: 0 18px 50px rgba(0,0,0,0.2);
        }
        .grid { display: grid; gap: 20px; grid-template-columns: minmax(320px, 440px) minmax(0, 1fr); }
        label { display: block; font-size: 12px; text-transform: uppercase; letter-spacing: .08em; color: #a8bddc; margin-bottom: 8px; }
        input, textarea, select {
            width: 100%;
            border: 1px solid rgba(255,255,255,0.18);
            background: rgba(4,14,28,0.45);
            color: #fff;
            border-radius: 14px;
            padding: 12px 14px;
            font: inherit;
        }
        textarea { min-height: 140px; resize: vertical; }
        button {
            border: 0;
            border-radius: 16px;
            background: linear-gradient(135deg, #42a5ff 0%, #1769ff 100%);
            color: #fff;
            font: inherit;
            font-weight: 700;
            padding: 14px 18px;
            cursor: pointer;
        }
        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 12px 10px; border-bottom: 1px solid rgba(255,255,255,0.08); text-align: left; vertical-align: top; }
        th { color: #a8bddc; font-size: 12px; text-transform: uppercase; letter-spacing: .08em; }
        code { background: rgba(255,255,255,0.08); border-radius: 8px; padding: 3px 6px; }
        .pill { display: inline-flex; border-radius: 999px; padding: 6px 10px; font-size: 12px; font-weight: 700; background: rgba(66,165,255,0.18); color: #9cd0ff; text-transform: uppercase; }
        .ok { margin-bottom: 16px; padding: 12px 14px; border-radius: 14px; background: rgba(32, 201, 151, 0.14); border: 1px solid rgba(32, 201, 151, 0.28); color: #9cf0c9; }
        .muted { color: #a8bddc; }
        @media (max-width: 980px) { .grid { grid-template-columns: 1fr; } }
    </style>
</head>
<body>
    <main class="wrap">
        <div style="display:flex; justify-content:space-between; gap:16px; align-items:center; flex-wrap:wrap; margin-bottom:20px;">
            <div>
                <h1 style="margin:0 0 6px;">CRM25 Update Center</h1>
                <div class="muted">Admin upload panel for release packages and feed metadata.</div>
            </div>
            <div class="muted">Feed: <code><?= update_center_escape(rtrim($config['base_url'], '/').'/api/v1/crm25/updates') ?></code></div>
        </div>

        <?php if ($createdId !== null): ?>
            <div class="ok">Release #<?= $createdId ?> has been published successfully.</div>
        <?php endif; ?>

        <div class="grid">
            <section class="card">
                <h2 style="margin-top:0;">Upload release</h2>
                <form method="post" action="/admin/releases" enctype="multipart/form-data">
                    <input type="hidden" name="_token" value="<?= update_center_escape($token) ?>">
                    <div style="display:grid; gap:16px;">
                        <div>
                            <label for="product">Product</label>
                            <input id="product" name="product" type="text" value="<?= update_center_escape($config['default_product']) ?>" required>
                        </div>
                        <div>
                            <label for="version">Version</label>
                            <input id="version" name="version" type="text" placeholder="25.0.1" required>
                        </div>
                        <div>
                            <label for="build">Build</label>
                            <input id="build" name="build" type="text" placeholder="250001">
                        </div>
                        <div>
                            <label for="channel">Channel</label>
                            <select id="channel" name="channel">
                                <option value="stable">Stable</option>
                                <option value="beta">Beta</option>
                                <option value="nightly">Nightly</option>
                            </select>
                        </div>
                        <div>
                            <label for="notes">Release notes</label>
                            <textarea id="notes" name="notes" placeholder="Describe fixes, new features, migration notes."></textarea>
                        </div>
                        <div>
                            <label for="package">ZIP package</label>
                            <input id="package" name="package" type="file" accept=".zip" required>
                        </div>
                        <label style="display:flex; align-items:center; gap:10px; text-transform:none; letter-spacing:0;">
                            <input style="width:auto;" type="checkbox" name="is_active" value="1" checked>
                            Publish this release immediately
                        </label>
                        <button type="submit">Upload and publish</button>
                    </div>
                </form>
            </section>

            <section class="card">
                <h2 style="margin-top:0;">Release history</h2>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Version</th>
                            <th>Channel</th>
                            <th>File</th>
                            <th>Downloads</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($releases as $release): ?>
                            <tr>
                                <td>#<?= (int) $release['id'] ?></td>
                                <td>
                                    <strong><?= update_center_escape($release['version']) ?></strong><br>
                                    <span class="muted"><?= update_center_escape($release['published_at']) ?></span>
                                </td>
                                <td><span class="pill"><?= update_center_escape($release['channel']) ?></span></td>
                                <td>
                                    <a style="color:#9cd0ff;" href="<?= update_center_escape($release['download_url']) ?>"><?= update_center_escape($release['original_file_name']) ?></a><br>
                                    <span class="muted"><code><?= update_center_escape($release['checksum_sha256']) ?></code></span>
                                </td>
                                <td><?= (int) $release['download_count'] ?></td>
                                <td><?= $release['is_active'] ? 'Active' : 'Disabled' ?></td>
                            </tr>
                        <?php endforeach; ?>
                        <?php if ($releases === []): ?>
                            <tr>
                                <td colspan="6" class="muted">No releases uploaded yet.</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </section>
        </div>
    </main>
</body>
</html>
    <?php

    return (string) ob_get_clean();
}
