<?php

namespace Tests\Feature;

use App\Models\OrganizationCompany;
use App\Models\OrganizationSetting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class UserManagementTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_can_create_user_with_permissions_and_company(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        OrganizationSetting::query()->create([
            'group_companies_enabled' => true,
        ]);

        $parent = OrganizationCompany::factory()->create([
            'name' => 'Holding',
        ]);
        $child = OrganizationCompany::factory()->create([
            'name' => 'Branch',
            'parent_id' => $parent->id,
        ]);

        $payload = [
            'name' => 'New Member',
            'email' => 'member@example.test',
            'password' => 'SecurePass123',
            'password_confirmation' => 'SecurePass123',
            'role' => 'user',
            'organization_company_id' => $child->id,
            'permissions' => [
                'tasks' => [
                    'read' => 1,
                    'create' => 1,
                    'update' => 0,
                    'delete' => 0,
                ],
            ],
        ];

        $this->actingAs($admin)
            ->post(route('profile.users.store'), $payload)
            ->assertRedirect(route('profile.edit', ['section' => 'users']));

        $this->assertDatabaseHas('users', [
            'email' => 'member@example.test',
            'organization_company_id' => $child->id,
            'role' => 'user',
        ]);

        $created = User::query()->where('email', 'member@example.test')->firstOrFail();
        $this->assertTrue((bool) ($created->permissions['tasks.read'] ?? false));
        $this->assertTrue((bool) ($created->permissions['tasks.create'] ?? false));
        $this->assertFalse((bool) ($created->permissions['tasks.delete'] ?? true));
    }

    public function test_admin_can_download_template_and_import_users(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $response = $this->actingAs($admin)
            ->get(route('profile.users.template'));

        $response->assertOk();
        $response->assertHeader('Content-Type', 'text/csv; charset=UTF-8');

        $csv = (string) $response->getContent();
        $this->assertStringContainsString('name,email,password', $csv);

        $payload = "name,email,password,role\n".
            "Jane Doe,jane@example.test,SecurePass123,user\n";

        $file = \Illuminate\Http\UploadedFile::fake()->createWithContent('users.csv', $payload);

        $this->actingAs($admin)
            ->post(route('profile.users.import'), [
                'import_file' => $file,
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'users']));

        $this->assertDatabaseHas('users', [
            'email' => 'jane@example.test',
        ]);
    }
}
