<?php

namespace Tests\Feature;

use App\Models\Disk;
use App\Models\Pipeline;
use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class SectionAccessTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_sees_section_access_gear_in_supported_sections(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $this->actingAs($admin)
            ->get(route('tasks.index'))
            ->assertOk()
            ->assertSee('Manage section access');

        $this->actingAs($admin)
            ->get(route('pipelines.index'))
            ->assertOk()
            ->assertSee('Manage section access');

        $this->actingAs($admin)
            ->get(route('disks.index'))
            ->assertOk()
            ->assertSee('Manage section access');

        $this->actingAs($admin)
            ->get(route('onec.index'))
            ->assertOk()
            ->assertSee('Manage section access');

        $this->actingAs($admin)
            ->get(route('hr.index'))
            ->assertOk()
            ->assertSee('Manage section access');
    }

    public function test_admin_can_update_tasks_section_permissions_for_user(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['role' => 'user']);

        $this->actingAs($admin)
            ->patch(route('section-access.update', ['entity' => 'tasks']), [
                'user_id' => $member->id,
                'permissions' => [
                    'read' => 1,
                    'create' => 1,
                    'update' => 0,
                    'delete' => 0,
                ],
            ])
            ->assertRedirect();

        $member->refresh();

        $this->assertTrue((bool) ($member->permissions['tasks.read'] ?? false));
        $this->assertTrue((bool) ($member->permissions['tasks.create'] ?? false));
        $this->assertFalse((bool) ($member->permissions['tasks.update'] ?? true));
        $this->assertFalse((bool) ($member->permissions['tasks.delete'] ?? true));
    }

    public function test_task_creator_can_update_tasks_section_permissions(): void
    {
        $actor = User::factory()->create(['role' => 'user']);
        $member = User::factory()->create(['role' => 'user']);

        Task::factory()->create([
            'creator_id' => $actor->id,
            'assignee_id' => $actor->id,
        ]);

        $this->actingAs($actor)
            ->patch(route('section-access.update', ['entity' => 'tasks']), [
                'user_id' => $member->id,
                'permissions' => [
                    'read' => 1,
                    'create' => 0,
                    'update' => 0,
                    'delete' => 0,
                ],
            ])
            ->assertRedirect();

        $this->assertTrue((bool) ($member->fresh()?->permissions['tasks.read'] ?? false));
    }

    public function test_pipeline_creator_can_update_pipelines_section_permissions(): void
    {
        $actor = User::factory()->create(['role' => 'user']);
        $member = User::factory()->create(['role' => 'user']);

        Pipeline::factory()->create([
            'creator_id' => $actor->id,
        ]);

        $this->actingAs($actor)
            ->patch(route('section-access.update', ['entity' => 'pipelines']), [
                'user_id' => $member->id,
                'permissions' => [
                    'read' => 1,
                    'create' => 1,
                    'update' => 1,
                    'delete' => 0,
                ],
            ])
            ->assertRedirect();

        $member->refresh();

        $this->assertTrue((bool) ($member->permissions['pipelines.read'] ?? false));
        $this->assertTrue((bool) ($member->permissions['pipelines.create'] ?? false));
        $this->assertTrue((bool) ($member->permissions['pipelines.update'] ?? false));
        $this->assertFalse((bool) ($member->permissions['pipelines.delete'] ?? true));
    }

    public function test_disk_owner_can_update_disks_section_permissions(): void
    {
        $actor = User::factory()->create(['role' => 'user']);
        $member = User::factory()->create(['role' => 'user']);

        Disk::factory()->create([
            'owner_id' => $actor->id,
        ]);

        $this->actingAs($actor)
            ->patch(route('section-access.update', ['entity' => 'disks']), [
                'user_id' => $member->id,
                'permissions' => [
                    'read' => 1,
                    'create' => 1,
                    'update' => 1,
                    'delete' => 1,
                ],
            ])
            ->assertRedirect();

        $member->refresh();

        $this->assertTrue((bool) ($member->permissions['disks.read'] ?? false));
        $this->assertTrue((bool) ($member->permissions['disks.create'] ?? false));
        $this->assertTrue((bool) ($member->permissions['disks.update'] ?? false));
        $this->assertTrue((bool) ($member->permissions['disks.delete'] ?? false));
    }

    public function test_non_admin_without_created_section_entities_cannot_manage_section_access(): void
    {
        $actor = User::factory()->create(['role' => 'user']);
        $member = User::factory()->create(['role' => 'user']);

        $this->actingAs($actor)
            ->get(route('tasks.index'))
            ->assertOk()
            ->assertDontSee('Manage section access');

        $this->actingAs($actor)
            ->patch(route('section-access.update', ['entity' => 'tasks']), [
                'user_id' => $member->id,
                'permissions' => [
                    'read' => 1,
                ],
            ])
            ->assertForbidden();
    }
}
