<?php

namespace Tests\Feature;

use App\Models\ChatMessage;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Symfony\Component\Process\Process;
use Tests\TestCase;

class RightRailUiTest extends TestCase
{
    use RefreshDatabase;

    public function test_right_rail_with_notifications_and_chats_is_rendered(): void
    {
        $user = User::factory()->create(['role' => 'admin']);

        $this->actingAs($user)
            ->get(route('dashboard'))
            ->assertOk()
            ->assertSee(route('dashboard'))
            ->assertSee(config('app.name', 'CRM'))
            ->assertSee('Notifications')
            ->assertSee('Chats');
    }

    public function test_profile_photo_is_rendered_in_right_rail_when_set(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
            'profile_photo_path' => 'profile-photos/test-avatar.jpg',
            'profile_photo_focus_x' => 34,
            'profile_photo_focus_y' => 67,
            'profile_photo_zoom' => 1.45,
        ]);

        $this->actingAs($user)
            ->get(route('dashboard'))
            ->assertOk()
            ->assertSee('profile-photos/test-avatar.jpg')
            ->assertSee('object-position: center center; transform: scale(1.45);', false);
    }

    public function test_right_rail_shows_chat_list_and_hides_micro_chat_composer(): void
    {
        $user = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['name' => 'Ivan Petrov']);

        ChatMessage::query()->create([
            'sender_id' => $member->id,
            'recipient_id' => $user->id,
            'body' => 'Test message in chat',
        ]);

        $this->actingAs($user)
            ->get(route('dashboard'))
            ->assertOk()
            ->assertSee('Ivan Petrov')
            ->assertSee('Opens in sidebar')
            ->assertDontSee('Enter message...');
    }

    public function test_right_rail_marks_current_user_as_online_with_green_status_dot(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
            'last_seen_at' => null,
        ]);

        $this->actingAs($user)
            ->get(route('dashboard'))
            ->assertOk()
            ->assertSee('bg-emerald-400', false);

        $this->assertNotNull($user->fresh()?->last_seen_at);
    }

    public function test_right_rail_inline_script_has_valid_javascript_syntax(): void
    {
        $blade = file_get_contents(resource_path('views/layouts/right-rail.blade.php'));
        $this->assertIsString($blade);

        preg_match('/<script>\s*(.*?)\s*<\/script>/s', $blade, $matches);
        $script = $matches[1] ?? null;

        $this->assertIsString($script);
        $this->assertNotSame('', trim($script));

        $tempFile = tempnam(sys_get_temp_dir(), 'right-rail-js-');
        $this->assertNotFalse($tempFile);

        file_put_contents($tempFile, $script);

        $process = new Process(['node', '--check', $tempFile]);
        $process->run();

        @unlink($tempFile);

        $this->assertTrue(
            $process->isSuccessful(),
            trim($process->getErrorOutput()) ?: trim($process->getOutput()) ?: 'Node syntax check failed.'
        );
    }
}
