<?php

namespace Tests\Feature;

use App\Models\RemoteAccessRequest;
use App\Models\User;
use App\Support\RightRailManager;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Schema;
use Tests\TestCase;

class RemoteAccessRequestFeatureTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_can_request_remote_access_from_user_profile(): void
    {
        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create(['role' => 'admin']);

        $this->actingAs($requester)
            ->post(route('users.remote-access-requests.store', ['member' => $recipient]), [
                'redirect_to' => route('users.show', ['member' => $recipient]),
            ])
            ->assertRedirect(route('users.show', ['member' => $recipient]));

        $request = RemoteAccessRequest::query()->first();

        $this->assertNotNull($request);
        $this->assertSame($requester->id, $request->requester_id);
        $this->assertSame($recipient->id, $request->recipient_id);
        $this->assertSame('requested', $request->status);
    }

    public function test_recipient_can_approve_remote_access_request_from_profile(): void
    {
        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create(['role' => 'admin']);

        $request = RemoteAccessRequest::query()->create([
            'requester_id' => $requester->id,
            'recipient_id' => $recipient->id,
            'status' => 'requested',
            'requested_at' => now(),
        ]);

        $this->actingAs($recipient)
            ->patch(route('profile.remote-access-requests.update', $request), [
                'status' => 'approved',
                'response_message' => 'AnyDesk ID: 555 123 777',
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'profile']).'#remote-access-requests');

        $request->refresh();

        $this->assertSame('approved', $request->status);
        $this->assertSame('AnyDesk ID: 555 123 777', $request->response_message);
        $this->assertNotNull($request->resolved_at);
    }

    public function test_profile_page_renders_remote_access_request_lists(): void
    {
        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create(['role' => 'admin']);

        RemoteAccessRequest::query()->create([
            'requester_id' => $requester->id,
            'recipient_id' => $recipient->id,
            'status' => 'requested',
            'requested_at' => now(),
        ]);

        $this->actingAs($recipient)
            ->get(route('profile.edit', ['section' => 'profile']))
            ->assertOk()
            ->assertSee('Remote access requests')
            ->assertSee('Incoming requests')
            ->assertSee('Outgoing requests')
            ->assertSee($requester->name);
    }

    public function test_right_rail_shows_remote_access_notifications_for_recipient_and_requester_result(): void
    {
        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create(['role' => 'admin']);

        $request = RemoteAccessRequest::query()->create([
            'requester_id' => $requester->id,
            'recipient_id' => $recipient->id,
            'status' => 'requested',
            'requested_at' => now()->subMinute(),
        ]);

        $manager = app(RightRailManager::class);
        $recipientNotification = collect($manager->data($recipient)['notifications'])
            ->first(fn (array $item): bool => str_starts_with((string) ($item['key'] ?? ''), 'remote-access-request-'.$request->id.'-'));

        $this->assertIsArray($recipientNotification);
        $this->assertSame('Remote access request', $recipientNotification['title'] ?? null);

        $request->forceFill([
            'status' => 'approved',
            'response_message' => 'Approved',
            'resolved_at' => now(),
        ])->save();

        $requesterNotification = collect($manager->data($requester)['notifications'])
            ->first(fn (array $item): bool => str_starts_with((string) ($item['key'] ?? ''), 'remote-access-result-'.$request->id.'-approved-'));

        $this->assertIsArray($requesterNotification);
        $this->assertSame('Remote access approved', $requesterNotification['title'] ?? null);
    }

    public function test_opening_remote_access_notification_redirects_to_profile_requests_block(): void
    {
        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create(['role' => 'admin']);

        $request = RemoteAccessRequest::query()->create([
            'requester_id' => $requester->id,
            'recipient_id' => $recipient->id,
            'status' => 'requested',
            'requested_at' => now(),
        ]);

        $this->actingAs($recipient)
            ->get(route('notifications.open', ['key' => 'remote-access-request-'.$request->id.'-'.$request->requested_at?->timestamp]))
            ->assertRedirect(route('profile.edit', ['section' => 'profile']).'#remote-access-requests');
    }

    public function test_recipient_can_approve_with_profile_connection_and_requester_can_open_session_page(): void
    {
        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create([
            'role' => 'admin',
            'remote_access_provider' => 'anydesk',
            'remote_access_address' => '555 123 777',
            'remote_access_password' => 'secret-42',
            'remote_access_launch_url' => 'anydesk:555123777',
            'remote_access_notes' => 'PC is online until 19:00',
        ]);

        $request = RemoteAccessRequest::query()->create([
            'requester_id' => $requester->id,
            'recipient_id' => $recipient->id,
            'status' => 'requested',
            'requested_at' => now(),
        ]);

        $this->actingAs($recipient)
            ->patch(route('profile.remote-access-requests.update', $request), [
                'status' => 'approved',
                'response_message' => '',
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'profile']).'#remote-access-requests');

        $request->refresh();

        $this->assertSame('approved', $request->status);
        $this->assertSame('anydesk', data_get($request->connection_payload, 'provider'));
        $this->assertSame('555 123 777', data_get($request->connection_payload, 'address'));
        $this->assertStringContainsString('555 123 777', (string) $request->response_message);

        $this->actingAs($requester)
            ->get(route('remote-access-requests.show', $request))
            ->assertOk()
            ->assertSee('Open in app')
            ->assertSee('555 123 777')
            ->assertSee('secret-42');
    }

    public function test_profile_update_auto_fills_browser_link_for_host_based_remote_access(): void
    {
        $user = User::factory()->create(['role' => 'admin']);

        $this->actingAs($user)
            ->patch(route('profile.update'), [
                'name' => $user->name,
                'email' => $user->email,
                'remote_access_provider' => 'other',
                'remote_access_address' => 'gateway.example.com/session/24',
                'remote_access_password' => 'browser-pass',
                'remote_access_launch_url' => '',
                'remote_access_web_url' => '',
                'remote_access_notes' => 'Open in browser only',
            ])
            ->assertRedirect(route('profile.edit'));

        $user->refresh();

        $this->assertSame('https://gateway.example.com/session/24', $user->remote_access_web_url);
        $this->assertNull($user->remote_access_launch_url);
    }

    public function test_requester_sees_browser_open_button_when_approved_connection_has_web_url(): void
    {
        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create([
            'role' => 'admin',
            'remote_access_provider' => 'other',
            'remote_access_address' => 'gateway.example.com/session/24',
            'remote_access_password' => 'browser-pass',
            'remote_access_notes' => 'Browser gateway',
        ]);

        $request = RemoteAccessRequest::query()->create([
            'requester_id' => $requester->id,
            'recipient_id' => $recipient->id,
            'status' => 'requested',
            'requested_at' => now(),
        ]);

        $this->actingAs($recipient)
            ->patch(route('profile.remote-access-requests.update', $request), [
                'status' => 'approved',
                'response_message' => '',
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'profile']).'#remote-access-requests');

        $request->refresh();

        $this->assertSame('https://gateway.example.com/session/24', data_get($request->connection_payload, 'web_url'));

        $this->actingAs($requester)
            ->get(route('remote-access-requests.show', $request))
            ->assertOk()
            ->assertSee('Open in browser')
            ->assertSee('https://gateway.example.com/session/24');
    }

    public function test_requester_sees_generated_browser_gateway_link_for_rdp_profile(): void
    {
        config()->set('remote_browser_gateway.enabled', true);
        config()->set('remote_browser_gateway.base_url', 'https://update.crm25.webnet.kz/remote-browser/');
        config()->set('remote_browser_gateway.secret_key', '0123456789abcdef0123456789abcdef');

        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create([
            'role' => 'admin',
            'remote_access_provider' => 'rdp',
            'remote_access_address' => '10.10.10.20',
            'remote_access_username' => 'DOMAIN\\alice',
            'remote_access_port' => 3389,
            'remote_access_password' => 'rdp-secret',
            'remote_access_notes' => 'Use browser gateway',
        ]);

        $request = RemoteAccessRequest::query()->create([
            'requester_id' => $requester->id,
            'recipient_id' => $recipient->id,
            'status' => 'requested',
            'requested_at' => now(),
        ]);

        $this->actingAs($recipient)
            ->patch(route('profile.remote-access-requests.update', $request), [
                'status' => 'approved',
                'response_message' => '',
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'profile']).'#remote-access-requests');

        $this->actingAs($requester)
            ->get(route('remote-access-requests.show', $request))
            ->assertOk()
            ->assertSee('Open in browser')
            ->assertSee('DOMAIN\\alice')
            ->assertSee('10.10.10.20')
            ->assertSee('data=');
    }

    public function test_requester_opening_approved_remote_access_notification_redirects_to_session_page(): void
    {
        $requester = User::factory()->create(['role' => 'admin']);
        $recipient = User::factory()->create(['role' => 'admin']);

        $request = RemoteAccessRequest::query()->create([
            'requester_id' => $requester->id,
            'recipient_id' => $recipient->id,
            'status' => 'approved',
            'response_message' => 'AnyDesk ID: 555 123 777',
            'connection_payload' => [
                'provider' => 'anydesk',
                'address' => '555 123 777',
            ],
            'requested_at' => now()->subMinute(),
            'resolved_at' => now(),
        ]);

        $this->actingAs($requester)
            ->get(route('notifications.open', ['key' => 'remote-access-result-'.$request->id.'-approved-'.$request->resolved_at?->timestamp]))
            ->assertRedirect(route('remote-access-requests.show', $request));
    }

    public function test_user_profile_page_does_not_crash_when_remote_access_table_is_missing(): void
    {
        $viewer = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['role' => 'admin']);

        Schema::dropIfExists('remote_access_requests');

        $this->actingAs($viewer)
            ->get(route('users.show', ['member' => $member]))
            ->assertOk()
            ->assertDontSee('Request remote access');
    }
}
