<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProjectAndReportFlowTest extends TestCase
{
    use RefreshDatabase;

    public function test_authenticated_user_can_open_projects_pages(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $user->id,
            'manager_id' => $user->id,
        ]);

        $project->members()->sync([
            $user->id => ['role' => 'owner', 'joined_at' => now()],
        ]);

        $this->actingAs($user)
            ->get(route('projects.index'))
            ->assertOk();

        $this->actingAs($user)
            ->get(route('projects.show', $project))
            ->assertOk();
    }

    public function test_authenticated_user_can_open_projects_kanban_and_preference_is_saved(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
            'preferred_project_view' => 'list',
        ]);

        Project::factory()->create([
            'owner_id' => $user->id,
            'manager_id' => $user->id,
            'status' => 'active',
        ]);

        $this->actingAs($user)
            ->get(route('projects.index', ['view' => 'kanban']))
            ->assertOk()
            ->assertSee('Project Kanban', false)
            ->assertSee('data-project-card', false);

        $this->assertSame('kanban', $user->fresh()->preferred_project_view);
    }

    public function test_project_status_can_be_updated_via_ajax(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $manager->id,
            'manager_id' => $manager->id,
            'status' => 'planned',
            'completed_at' => null,
        ]);

        $project->members()->sync([
            $manager->id => ['role' => 'owner', 'joined_at' => now()],
        ]);

        $this->actingAs($manager)
            ->patchJson(route('projects.update-status', $project), [
                'status' => 'completed',
            ])
            ->assertOk()
            ->assertJsonPath('project.id', $project->id)
            ->assertJsonPath('project.status', 'completed')
            ->assertJsonPath('project.status_label', 'Completed');

        $project->refresh();

        $this->assertSame('completed', $project->status);
        $this->assertNotNull($project->completed_at);
    }

    public function test_project_task_stage_can_be_updated_via_ajax(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $manager->id,
            'manager_id' => $manager->id,
            'status' => 'active',
        ]);

        $project->members()->sync([
            $manager->id => ['role' => 'owner', 'joined_at' => now()],
        ]);

        $stageTodo = $project->stages()->create([
            'name' => 'Backlog',
            'code' => 'backlog_1',
            'sort_order' => 0,
            'is_done' => false,
        ]);

        $stageDone = $project->stages()->create([
            'name' => 'Done',
            'code' => 'done_2',
            'sort_order' => 1,
            'is_done' => true,
        ]);

        $task = Task::create([
            'title' => 'Move in board',
            'project_id' => $project->id,
            'project_stage_id' => $stageTodo->id,
            'creator_id' => $manager->id,
            'assignee_id' => $manager->id,
            'status' => 'todo',
            'priority' => 'high',
        ]);

        $this->actingAs($manager)
            ->patchJson(route('projects.tasks.update-stage', [$project, $task]), [
                'project_stage_id' => $stageDone->id,
            ])
            ->assertOk()
            ->assertJsonPath('task.id', $task->id)
            ->assertJsonPath('task.project_stage_id', $stageDone->id)
            ->assertJsonPath('task.status', 'done');

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'project_stage_id' => $stageDone->id,
            'status' => 'done',
        ]);
    }

    public function test_manager_can_update_project_stage_order_and_colors_from_kanban_endpoint(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $manager->id,
            'manager_id' => $manager->id,
            'status' => 'active',
        ]);

        $project->members()->sync([
            $manager->id => ['role' => 'owner', 'joined_at' => now()],
        ]);

        $firstStage = $project->stages()->create([
            'name' => 'Backlog',
            'code' => 'backlog_1',
            'sort_order' => 0,
            'color' => '#3B82F6',
            'is_done' => false,
        ]);

        $secondStage = $project->stages()->create([
            'name' => 'In progress',
            'code' => 'in_progress_2',
            'sort_order' => 1,
            'color' => '#F59E0B',
            'is_done' => false,
        ]);

        $this->actingAs($manager)
            ->patchJson(route('projects.update-stages', $project), [
                'stages' => [
                    ['id' => $secondStage->id, 'color' => '#123ABC'],
                    ['id' => $firstStage->id, 'color' => '#654321'],
                ],
            ])
            ->assertOk()
            ->assertJsonPath('stages.0.id', $secondStage->id)
            ->assertJsonPath('stages.0.color', '#123ABC')
            ->assertJsonPath('stages.1.id', $firstStage->id)
            ->assertJsonPath('stages.1.color', '#654321');

        $this->assertDatabaseHas('project_stages', [
            'id' => $secondStage->id,
            'sort_order' => 0,
            'color' => '#123ABC',
        ]);

        $this->assertDatabaseHas('project_stages', [
            'id' => $firstStage->id,
            'sort_order' => 1,
            'color' => '#654321',
        ]);
    }

    public function test_manager_can_create_project_stage_via_ajax(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $manager->id,
            'manager_id' => $manager->id,
            'status' => 'active',
        ]);

        $project->members()->sync([
            $manager->id => ['role' => 'owner', 'joined_at' => now()],
        ]);

        $this->actingAs($manager)
            ->postJson(route('projects.stages.store', $project), [
                'name' => 'Client review',
                'color' => '#123ABC',
            ])
            ->assertCreated()
            ->assertJsonPath('stage.name', 'Client review')
            ->assertJsonPath('stage.color', '#123ABC');

        $this->assertDatabaseHas('project_stages', [
            'project_id' => $project->id,
            'name' => 'Client review',
            'color' => '#123ABC',
        ]);
    }

    public function test_manager_can_delete_project_stage_and_move_tasks_to_unassigned_via_ajax(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $manager->id,
            'manager_id' => $manager->id,
            'status' => 'active',
        ]);

        $project->members()->sync([
            $manager->id => ['role' => 'owner', 'joined_at' => now()],
        ]);

        $stage = $project->stages()->create([
            'name' => 'Delete me',
            'code' => 'delete_me_1',
            'sort_order' => 0,
            'color' => '#3B82F6',
            'is_done' => false,
        ]);

        $task = Task::create([
            'title' => 'Task from stage',
            'project_id' => $project->id,
            'project_stage_id' => $stage->id,
            'creator_id' => $manager->id,
            'assignee_id' => $manager->id,
            'status' => 'todo',
            'priority' => 'medium',
        ]);

        $this->actingAs($manager)
            ->deleteJson(route('projects.stages.destroy', [$project, $stage]))
            ->assertOk()
            ->assertJsonPath('deleted_stage_id', $stage->id)
            ->assertJsonPath('moved_task_ids.0', $task->id);

        $this->assertDatabaseMissing('project_stages', [
            'id' => $stage->id,
        ]);

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'project_stage_id' => null,
        ]);
    }

    public function test_sales_cannot_access_reports_but_manager_can(): void
    {
        $sales = User::factory()->create([
            'role' => 'sales',
        ]);

        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $this->actingAs($sales)
            ->get(route('reports.index'))
            ->assertForbidden();

        $this->actingAs($manager)
            ->get(route('reports.index'))
            ->assertOk();
    }
}
