<?php

namespace Tests\Feature;

use App\Models\Product;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProductSectionTest extends TestCase
{
    use RefreshDatabase;

    public function test_manager_can_open_products_index_and_create_product(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $this->actingAs($manager)
            ->get(route('products.index'))
            ->assertOk()
            ->assertSee('Products');

        $this->actingAs($manager)
            ->post(route('products.store'), [
                'name' => 'CRM Product',
                'sku' => 'CRM-001',
                'price' => 499,
                'currency' => 'usd',
                'unit' => 'pcs',
                'stock' => 12,
                'status' => 'active',
                'owner_id' => $manager->id,
            ])
            ->assertRedirect();

        $this->assertDatabaseHas('products', [
            'name' => 'CRM Product',
            'sku' => 'CRM-001',
            'currency' => 'USD',
            'status' => 'active',
        ]);
    }

    public function test_regular_user_cannot_open_products_section_by_default(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
        ]);

        $this->actingAs($user)
            ->get(route('products.index'))
            ->assertForbidden();
    }

    public function test_personalized_permissions_allow_regular_user_to_manage_products(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'products.read' => true,
                'products.create' => true,
                'products.update' => true,
                'products.delete' => true,
            ],
        ]);

        $this->actingAs($user)
            ->post(route('products.store'), [
                'name' => 'Custom access product',
                'sku' => 'CUSTOM-001',
                'price' => 150,
                'currency' => 'EUR',
                'unit' => 'license',
                'stock' => 3,
                'status' => 'active',
                'owner_id' => $user->id,
            ])
            ->assertRedirect();

        $product = Product::query()->where('sku', 'CUSTOM-001')->firstOrFail();

        $this->actingAs($user)
            ->put(route('products.update', $product), [
                'name' => 'Custom access product updated',
                'sku' => 'CUSTOM-001',
                'price' => 175,
                'currency' => 'EUR',
                'unit' => 'license',
                'stock' => 2,
                'status' => 'archived',
                'owner_id' => $user->id,
            ])
            ->assertRedirect(route('products.show', $product));

        $this->assertDatabaseHas('products', [
            'id' => $product->id,
            'name' => 'Custom access product updated',
            'status' => 'archived',
        ]);
    }
}
