<?php

namespace Tests\Feature;

use App\Models\News;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class NewsFeedTest extends TestCase
{
    use RefreshDatabase;

    public function test_news_page_shows_only_news_feed_without_extra_side_blocks(): void
    {
        $user = User::factory()->create();
        News::factory()->create([
            'user_id' => $user->id,
            'title' => 'New company policy',
            'content' => 'We are moving to a unified process of working with tasks.',
        ]);

        $this->actingAs($user)
            ->get(route('news.index'))
            ->assertOk()
            ->assertSee('News feed')
            ->assertSee('New company policy')
            ->assertDontSee('Hot deals')
            ->assertDontSee('Urgent tasks');
    }

    public function test_user_can_publish_news_from_news_section(): void
    {
        $user = User::factory()->create();

        $this->actingAs($user)
            ->post(route('news.store'), [
                'title' => 'Launch of a new release',
                'content' => 'Today we updated our CRM and added a task calendar.',
            ])
            ->assertRedirect(route('news.index'));

        $this->assertDatabaseHas('news', [
            'title' => 'Launch of a new release',
            'user_id' => $user->id,
        ]);

        $this->actingAs($user)
            ->get(route('news.index'))
            ->assertOk()
            ->assertSee('Launch of a new release')
            ->assertSee('Today we updated our CRM and added a task calendar.');
    }

    public function test_news_create_form_is_rendered_for_sidepanel_requests(): void
    {
        $user = User::factory()->create();

        $this->actingAs($user)
            ->get(route('news.create', ['sidepanel' => 1]))
            ->assertOk()
            ->assertSee('data-sidepanel-submit', false)
            ->assertSee('Create news');
    }

    public function test_user_can_publish_news_from_sidepanel_with_json_response(): void
    {
        $user = User::factory()->create();

        $this->actingAs($user)
            ->postJson(route('news.store'), [
                'title' => 'Internal update',
                'content' => 'The new dashboard is ready.',
                'sidepanel' => 1,
            ], [
                'X-Sidepanel' => '1',
            ])
            ->assertOk()
            ->assertJsonPath('reload', true)
            ->assertJsonPath('message', 'The news has been published.');

        $this->assertDatabaseHas('news', [
            'title' => 'Internal update',
            'content' => 'The new dashboard is ready.',
            'user_id' => $user->id,
        ]);
    }
}
