<?php

namespace Tests\Feature;

use App\Models\CrmModule;
use App\Models\Company;
use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Tests\TestCase;
use ZipArchive;

class ModuleManagementTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_sees_modules_settings_tab_and_can_open_it(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $this->actingAs($admin)
            ->get(route('profile.edit'))
            ->assertOk()
            ->assertSee('Modules');

        $this->actingAs($admin)
            ->get(route('profile.edit', ['section' => 'modules']))
            ->assertOk()
            ->assertSee('Modules')
            ->assertSee('Upload module archive')
            ->assertSee('Download scaffold ZIP');
    }

    public function test_non_admin_cannot_install_modules(): void
    {
        $member = User::factory()->create(['role' => 'user']);

        $archivePath = $this->createZipArchive([
            'module.json' => json_encode([
                'name' => 'Blocked module',
                'slug' => 'blocked-module',
                'version' => '1.0.0',
                'hooks' => [],
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
        ]);

        try {
            $upload = new UploadedFile($archivePath, 'blocked-module.zip', 'application/zip', null, true);

            $this->actingAs($member)
                ->post(route('modules.store'), [
                    'archive' => $upload,
                ])
                ->assertForbidden();
        } finally {
            @unlink($archivePath);
        }
    }

    public function test_authenticated_user_can_open_modules_documentation_page(): void
    {
        $user = User::factory()->create();

        $this->actingAs($user)
            ->get(route('docs.modules.page'))
            ->assertOk()
            ->assertSee('Custom Modules Documentation')
            ->assertSee('companies.store')
            ->assertSee('tasks.store')
            ->assertSee('deals.update');
    }

    public function test_admin_can_install_module_and_task_store_hook_modifies_payload(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $archivePath = $this->createZipArchive([
            'module.json' => json_encode([
                'name' => 'Task priority module',
                'slug' => 'task-priority-module',
                'version' => '1.0.0',
                'description' => 'Forces urgent priority on new tasks.',
                'hooks' => [
                    'tasks.store' => 'hooks/tasks-store.php',
                ],
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
            'hooks/tasks-store.php' => <<<'PHP'
<?php
return static function (array $payload): array {
    $payload['priority'] = 'urgent';
    return $payload;
};
PHP,
        ]);

        try {
            $upload = new UploadedFile($archivePath, 'task-priority-module.zip', 'application/zip', null, true);

            $this->actingAs($admin)
                ->post(route('modules.store'), [
                    'archive' => $upload,
                ])
                ->assertRedirect(route('profile.edit', ['section' => 'modules']));

            $this->assertDatabaseHas('crm_modules', [
                'slug' => 'task-priority-module',
                'is_enabled' => true,
            ]);
        } finally {
            @unlink($archivePath);
        }

        $this->actingAs($admin)
            ->post(route('tasks.store'), [
                'title' => 'Task from module test',
                'status' => 'todo',
                'priority' => 'low',
            ])
            ->assertRedirect();

        $task = Task::query()->where('title', 'Task from module test')->latest('id')->first();

        $this->assertNotNull($task);
        $this->assertSame('urgent', $task?->priority);
    }

    public function test_disabling_module_stops_hook_execution(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $archivePath = $this->createZipArchive([
            'module.json' => json_encode([
                'name' => 'Disable module test',
                'slug' => 'disable-module-test',
                'version' => '1.0.0',
                'hooks' => [
                    'tasks.store' => 'hooks/tasks-store.php',
                ],
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
            'hooks/tasks-store.php' => <<<'PHP'
<?php
return static fn (array $payload): array => ['priority' => 'urgent'];
PHP,
        ]);

        try {
            $upload = new UploadedFile($archivePath, 'disable-module-test.zip', 'application/zip', null, true);

            $this->actingAs($admin)
                ->post(route('modules.store'), [
                    'archive' => $upload,
                ])
                ->assertRedirect(route('profile.edit', ['section' => 'modules']));
        } finally {
            @unlink($archivePath);
        }

        $module = CrmModule::query()->where('slug', 'disable-module-test')->firstOrFail();

        $this->actingAs($admin)
            ->patch(route('modules.update', $module), [
                'is_enabled' => 0,
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'modules']));

        $this->actingAs($admin)
            ->post(route('tasks.store'), [
                'title' => 'Task after module disabled',
                'status' => 'todo',
                'priority' => 'low',
            ])
            ->assertRedirect();

        $task = Task::query()->where('title', 'Task after module disabled')->latest('id')->first();

        $this->assertNotNull($task);
        $this->assertSame('low', $task?->priority);
    }

    public function test_company_store_hook_modifies_web_payload(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $archivePath = $this->createZipArchive([
            'module.json' => json_encode([
                'name' => 'Company source module',
                'slug' => 'company-source-module',
                'version' => '1.0.0',
                'hooks' => [
                    'companies.store' => 'hooks/companies-store.php',
                ],
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
            'hooks/companies-store.php' => <<<'PHP'
<?php
return static fn (array $payload): array => ['source' => 'module-web'];
PHP,
        ]);

        try {
            $upload = new UploadedFile($archivePath, 'company-source-module.zip', 'application/zip', null, true);

            $this->actingAs($admin)
                ->post(route('modules.store'), ['archive' => $upload])
                ->assertRedirect(route('profile.edit', ['section' => 'modules']));
        } finally {
            @unlink($archivePath);
        }

        $response = $this->actingAs($admin)
            ->post(route('companies.store'), [
                'name' => 'Hooked Company',
                'status' => 'lead',
                'source' => 'manual',
            ]);

        $response->assertRedirect();

        $company = Company::query()->where('name', 'Hooked Company')->latest('id')->first();

        $this->assertNotNull($company);
        $this->assertSame('module-web', $company?->source);
    }

    public function test_product_store_hook_modifies_api_payload(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $archivePath = $this->createZipArchive([
            'module.json' => json_encode([
                'name' => 'Product status module',
                'slug' => 'product-status-module',
                'version' => '1.0.0',
                'hooks' => [
                    'products.store' => 'hooks/products-store.php',
                ],
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
            'hooks/products-store.php' => <<<'PHP'
<?php
return static fn (array $payload): array => ['status' => 'archived'];
PHP,
        ]);

        try {
            $upload = new UploadedFile($archivePath, 'product-status-module.zip', 'application/zip', null, true);

            $this->actingAs($admin)
                ->post(route('modules.store'), ['archive' => $upload])
                ->assertRedirect(route('profile.edit', ['section' => 'modules']));
        } finally {
            @unlink($archivePath);
        }

        $token = $admin->createToken('products-with-hooks', [
            'products.read',
            'products.create',
        ])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/products', [
                'name' => 'Hooked API product',
                'currency' => 'USD',
                'unit' => 'pcs',
                'status' => 'active',
            ])
            ->assertCreated()
            ->assertJsonPath('data.status', 'archived');
    }

    /**
     * @param  array<string, string>  $files
     */
    private function createZipArchive(array $files): string
    {
        $tempPath = tempnam(sys_get_temp_dir(), 'crm-module-');
        if (! is_string($tempPath) || $tempPath === '') {
            throw new \RuntimeException('Unable to create temporary file for module archive.');
        }

        @unlink($tempPath);
        $zipPath = $tempPath.'.zip';

        $zip = new ZipArchive();
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new \RuntimeException('Unable to create module ZIP archive.');
        }

        foreach ($files as $path => $contents) {
            $zip->addFromString($path, $contents);
        }

        $zip->close();

        return $zipPath;
    }
}
