<?php

namespace Tests\Feature;

use App\Models\OrganizationCompany;
use App\Models\OrganizationSetting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class CompanySettingsTest extends TestCase
{
    use RefreshDatabase;

    public function test_admin_can_open_company_settings_window(): void
    {
        $admin = User::factory()->create(['role' => 'admin']);

        $this->actingAs($admin)
            ->get(route('profile.edit'))
            ->assertOk()
            ->assertSee('Company');

        $this->actingAs($admin)
            ->get(route('profile.edit', ['section' => 'company']))
            ->assertOk()
            ->assertSee('Company')
            ->assertSee('Group of companies');
    }

    public function test_non_admin_cannot_update_company_settings(): void
    {
        $member = User::factory()->create(['role' => 'user']);

        $this->actingAs($member)
            ->patch(route('profile.company-settings.update'), [
                'group_companies_enabled' => 1,
                'companies' => [
                    ['name' => 'Blocked Company'],
                ],
            ])
            ->assertForbidden();
    }

    public function test_admin_can_manage_companies_and_user_can_select_company_in_profile(): void
    {
        Storage::fake('public');
        $admin = User::factory()->create(['role' => 'admin']);

        $this->actingAs($admin)
            ->patch(route('profile.company-settings.update'), [
                'group_companies_enabled' => 1,
                'companies' => [
                    [
                        'name' => 'CRM Holding',
                        'requisites' => 'TIN 123456789',
                        'address' => 'Main street 1',
                        'phones' => "+1 555 100\n+1 555 200",
                        'logo' => UploadedFile::fake()->image('holding-logo.png'),
                    ],
                    [
                        'name' => 'CRM Branch',
                        'parent_index' => 0,
                        'requisites' => 'TIN 987654321',
                        'address' => 'Second street 5',
                        'phones' => '+1 555 300',
                    ],
                ],
            ])
            ->assertRedirect(route('profile.edit', ['section' => 'company']));

        $this->assertDatabaseHas('organization_settings', [
            'group_companies_enabled' => 1,
        ]);
        $this->assertDatabaseHas('organization_companies', [
            'name' => 'CRM Holding',
        ]);
        $this->assertDatabaseHas('organization_companies', [
            'name' => 'CRM Branch',
        ]);

        $holding = OrganizationCompany::query()->where('name', 'CRM Holding')->firstOrFail();
        $branch = OrganizationCompany::query()->where('name', 'CRM Branch')->firstOrFail();

        $this->assertNotNull($holding->logo_path);
        Storage::disk('public')->assertExists((string) $holding->logo_path);
        $this->assertSame($holding->id, $branch->parent_id);

        $member = User::factory()->create(['role' => 'user']);

        $this->actingAs($member)
            ->get(route('profile.edit'))
            ->assertOk()
            ->assertSee('id="organization_company_id"', false)
            ->assertSee('CRM Holding / CRM Branch');

        $this->actingAs($member)
            ->patch(route('profile.update'), [
                'name' => $member->name,
                'email' => $member->email,
                'organization_company_id' => $branch->id,
            ])
            ->assertSessionHasNoErrors()
            ->assertRedirect(route('profile.edit'));

        $this->assertSame($branch->id, $member->refresh()->organization_company_id);
    }

    public function test_user_cannot_submit_company_id_when_group_mode_is_disabled(): void
    {
        OrganizationSetting::query()->create([
            'group_companies_enabled' => false,
        ]);
        $company = OrganizationCompany::factory()->create();
        $member = User::factory()->create(['role' => 'user']);

        $this->actingAs($member)
            ->patch(route('profile.update'), [
                'name' => $member->name,
                'email' => $member->email,
                'organization_company_id' => $company->id,
            ])
            ->assertSessionHasErrors(['organization_company_id']);
    }
}
