<?php

namespace Tests\Feature;

use App\Models\Company;
use App\Models\Contact;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class CompanyContactAvatarTest extends TestCase
{
    use RefreshDatabase;

    public function test_company_avatar_can_be_uploaded_displayed_and_removed(): void
    {
        Storage::fake('public');

        $user = User::factory()->create(['role' => 'admin']);

        $this->actingAs($user)
            ->post(route('companies.store'), [
                'name' => 'Avatar Company',
                'status' => 'lead',
                'source' => 'manual',
                'owner_id' => $user->id,
                'avatar' => UploadedFile::fake()->image('company-avatar.jpg', 320, 320),
            ])
            ->assertRedirect();

        /** @var Company $company */
        $company = Company::query()->latest('id')->firstOrFail();
        $this->assertNotNull($company->avatar_path);
        Storage::disk('public')->assertExists((string) $company->avatar_path);

        $this->actingAs($user)
            ->get(route('companies.show', $company))
            ->assertOk()
            ->assertSee((string) $company->avatar_path, false);

        $oldAvatarPath = (string) $company->avatar_path;

        $this->actingAs($user)
            ->put(route('companies.update', $company), [
                'name' => 'Avatar Company',
                'status' => 'lead',
                'source' => 'manual',
                'owner_id' => $user->id,
                'remove_avatar' => 1,
            ])
            ->assertRedirect(route('companies.show', $company));

        $company->refresh();
        $this->assertNull($company->avatar_path);
        Storage::disk('public')->assertMissing($oldAvatarPath);
    }

    public function test_contact_avatar_can_be_replaced_and_visible_in_views(): void
    {
        Storage::fake('public');

        $user = User::factory()->create(['role' => 'admin']);

        $this->actingAs($user)
            ->post(route('contacts.store'), [
                'first_name' => 'John',
                'last_name' => 'Avatar',
                'source' => 'manual',
                'owner_id' => $user->id,
                'avatar' => UploadedFile::fake()->image('contact-avatar-1.jpg', 300, 300),
            ])
            ->assertRedirect();

        /** @var Contact $contact */
        $contact = Contact::query()->latest('id')->firstOrFail();
        $oldAvatarPath = (string) $contact->avatar_path;
        $this->assertNotSame('', $oldAvatarPath);
        Storage::disk('public')->assertExists($oldAvatarPath);

        $this->actingAs($user)
            ->put(route('contacts.update', $contact), [
                'first_name' => 'John',
                'last_name' => 'Avatar',
                'source' => 'manual',
                'owner_id' => $user->id,
                'avatar' => UploadedFile::fake()->image('contact-avatar-2.jpg', 300, 300),
            ])
            ->assertRedirect(route('contacts.show', $contact));

        $contact->refresh();
        $this->assertNotNull($contact->avatar_path);
        $this->assertNotSame($oldAvatarPath, (string) $contact->avatar_path);
        Storage::disk('public')->assertMissing($oldAvatarPath);
        Storage::disk('public')->assertExists((string) $contact->avatar_path);

        $this->actingAs($user)
            ->get(route('contacts.index'))
            ->assertOk()
            ->assertSee((string) $contact->avatar_path, false);

        $this->actingAs($user)
            ->get(route('contacts.show', $contact).'?sidepanel=1')
            ->assertOk()
            ->assertSee((string) $contact->avatar_path, false);
    }
}
