<?php

namespace Tests\Feature;

use App\Models\ChatMessage;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ChatSidebarTest extends TestCase
{
    use RefreshDatabase;

    public function test_chat_sidebar_renders_full_chat_list_in_sidepanel_mode(): void
    {
        $user = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['name' => 'Ivan Petrov']);
        $anotherMember = User::factory()->create(['name' => 'Anna Smirnova']);

        ChatMessage::query()->create([
            'sender_id' => $member->id,
            'recipient_id' => $user->id,
            'body' => 'First message'
        ]);

        $this->actingAs($user)
            ->get(route('chat.sidebar', ['member' => $member]).'?sidepanel=1')
            ->assertOk()
            ->assertSee('Chat history')
            ->assertSee('Please enter a message...')
            ->assertSee('Attach from disk')
            ->assertDontSee('✓✓')
            ->assertSee('sidebarMode: true', false)
            ->assertSee('Ivan Petrov')
            ->assertSee('Anna Smirnova');
    }

    public function test_chat_sidebar_can_be_opened_as_full_page(): void
    {
        $user = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['name' => 'Ivan Petrov']);

        $this->actingAs($user)
            ->get(route('chat.sidebar', ['member' => $member]))
            ->assertOk()
            ->assertSee('Chats')
            ->assertSee('Chat history');
    }

    public function test_right_rail_chat_sidebar_urls_are_relative(): void
    {
        $user = User::factory()->create(['role' => 'admin']);
        User::factory()->create(['name' => 'Ivan Petrov']);

        $this->actingAs($user)
            ->get(route('dashboard'))
            ->assertOk()
            ->assertSee('\\/chat\\/conversations\\/__MEMBER__', false)
            ->assertSee('\\/chat\\/sidebar\\/__MEMBER__', false)
            ->assertSee('\\/users\\/__USER__', false)
            ->assertDontSee('http:\\/\\/localhost\\/chat\\/conversations\\/__MEMBER__', false)
            ->assertDontSee('http:\\/\\/localhost\\/chat\\/sidebar\\/__MEMBER__', false)
            ->assertDontSee('http:\\/\\/localhost\\/users\\/__USER__', false);
    }

    public function test_chat_feed_returns_conversations_and_unread_count_for_polling(): void
    {
        $user = User::factory()->create(['role' => 'admin']);
        $member = User::factory()->create(['name' => 'Ivan Petrov']);

        ChatMessage::query()->create([
            'sender_id' => $member->id,
            'recipient_id' => $user->id,
            'body' => 'New unread message',
        ]);

        $this->actingAs($user)
            ->getJson(route('chat.feed'))
            ->assertOk()
            ->assertJsonPath('unread_count', 1)
            ->assertJsonPath('chats.0.id', $member->id)
            ->assertJsonPath('chats.0.unread_count', 1);
    }
}
