<?php

namespace Tests\Feature;

use App\Models\ChatMessage;
use App\Models\Disk;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ChatConversationTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_can_load_conversation_and_mark_incoming_messages_as_read(): void
    {
        $user = User::factory()->create();
        $member = User::factory()->create();

        $incoming = ChatMessage::query()->create([
            'sender_id' => $member->id,
            'recipient_id' => $user->id,
            'body' => 'Hello, this is an incoming message',
        ]);

        ChatMessage::query()->create([
            'sender_id' => $user->id,
            'recipient_id' => $member->id,
            'body' => 'Reply message',
        ]);

        $response = $this->actingAs($user)
            ->getJson(route('chat.conversations.show', ['member' => $member]));

        $response->assertOk()
            ->assertJsonCount(2, 'messages')
            ->assertJsonPath('member.id', $member->id)
            ->assertJsonPath('messages.0.body', 'Hello, this is an incoming message')
            ->assertJsonPath('messages.0.is_own', false)
            ->assertJsonPath('messages.0.is_read', true)
            ->assertJsonPath('messages.1.is_own', true)
            ->assertJsonPath('messages.1.is_read', false);

        $this->assertNotNull($incoming->fresh()?->read_at);
    }

    public function test_user_can_send_message_to_selected_member(): void
    {
        $user = User::factory()->create();
        $member = User::factory()->create();

        $response = $this->actingAs($user)
            ->postJson(route('chat.conversations.store', ['member' => $member]), [
                'body' => 'Good afternoon In touch.',
            ]);

        $response->assertCreated()
            ->assertJsonPath('message.body', 'Good afternoon In touch.')
            ->assertJsonPath('message.is_own', true)
            ->assertJsonPath('message.is_read', false);

        $this->assertDatabaseHas('chat_messages', [
            'sender_id' => $user->id,
            'recipient_id' => $member->id,
            'body' => 'Good afternoon In touch.',
        ]);
    }

    public function test_user_can_send_message_with_disk_link_attachment(): void
    {
        $user = User::factory()->create([
            'permissions' => [
                'disks.read' => true,
            ],
        ]);
        $member = User::factory()->create();
        $disk = Disk::factory()->for($user, 'owner')->create([
            'name' => 'Contract #25',
            'extension' => 'pdf',
        ]);

        $response = $this->actingAs($user)
            ->postJson(route('chat.conversations.store', ['member' => $member]), [
                'disk_id' => $disk->id,
            ]);

        $response->assertCreated()
            ->assertJsonPath('message.is_own', true)
            ->assertJsonPath('message.body', '')
            ->assertJsonPath('message.disk_attachment.id', $disk->id)
            ->assertJsonPath('message.disk_attachment.name', 'Contract #25')
            ->assertJsonPath('message.disk_attachment.icon', 'fa-solid fa-file-pdf')
            ->assertJsonPath('message.disk_attachment.preview_url', route('disks.preview', ['disk' => $disk]));

        $this->assertDatabaseHas('chat_messages', [
            'sender_id' => $user->id,
            'recipient_id' => $member->id,
            'disk_id' => $disk->id,
            'body' => '',
        ]);
    }

    public function test_conversation_payload_includes_disk_attachment(): void
    {
        $user = User::factory()->create();
        $member = User::factory()->create([
            'permissions' => [
                'disks.read' => true,
            ],
        ]);
        $disk = Disk::factory()->for($member, 'owner')->create([
            'name' => 'Specification',
            'extension' => 'pdf',
        ]);

        ChatMessage::query()->create([
            'sender_id' => $member->id,
            'recipient_id' => $user->id,
            'body' => '',
            'disk_id' => $disk->id,
        ]);

        $this->actingAs($user)
            ->getJson(route('chat.conversations.show', ['member' => $member]))
            ->assertOk()
            ->assertJsonPath('messages.0.disk_attachment.id', $disk->id)
            ->assertJsonPath('messages.0.disk_attachment.name', 'Specification')
            ->assertJsonPath('messages.0.preview_text', 'File: Specification');
    }
}
