<?php

namespace Tests\Feature;

use App\Models\Activity;
use App\Models\Deal;
use App\Models\DealStage;
use App\Models\Pipeline;
use App\Models\Task;
use App\Models\User;
use App\Support\MenuManager;
use Carbon\CarbonImmutable;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CalendarFeatureTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_can_open_calendar_page(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $this->actingAs($user)
            ->get(route('calendar.index'))
            ->assertOk()
            ->assertSee('Task calendar')
            ->assertSee('data-task-calendar', false);
    }

    public function test_calendar_core_menu_item_is_not_available_for_regular_user(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
        ]);
        $menuItems = app(MenuManager::class)->sidebarItems($user);

        $this->assertFalse(
            $menuItems->contains(fn (array $item): bool => ($item['key'] ?? null) === 'calendar')
        );
    }

    public function test_user_without_calendar_permission_cannot_open_calendar(): void
    {
        $user = User::factory()->create([
            'role' => 'user',
            'permissions' => [
                'calendar.read' => false,
                'tasks.read' => true,
            ],
        ]);

        $this->actingAs($user)
            ->get(route('calendar.index'))
            ->assertForbidden();

        $this->actingAs($user)
            ->getJson(route('calendar.events'))
            ->assertForbidden();
    }

    public function test_calendar_events_returns_filtered_tasks(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);
        $rangeStart = CarbonImmutable::parse('2026-02-01 00:00:00');
        $rangeEnd = CarbonImmutable::parse('2026-02-28 23:59:59');

        $visibleTask = Task::query()->create([
            'title' => 'Calendar: task todo',
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
            'priority' => 'high',
            'starts_at' => '2026-02-10 10:00:00',
            'due_at' => '2026-02-10 12:00:00',
        ]);

        Task::query()->create([
            'title' => 'Calendar: task done',
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'done',
            'priority' => 'low',
            'starts_at' => '2026-02-11 10:00:00',
            'due_at' => '2026-02-11 11:00:00',
        ]);

        $response = $this->actingAs($user)->getJson(route('calendar.events', [
            'start' => $rangeStart->toIso8601String(),
            'end' => $rangeEnd->toIso8601String(),
            'status' => 'todo',
            'assignee_id' => $user->id,
        ]));

        $response->assertOk()
            ->assertJsonCount(1, 'events')
            ->assertJsonPath('events.0.id', 'task-'.$visibleTask->id)
            ->assertJsonPath('events.0.extendedProps.status', 'todo')
            ->assertJsonPath('events.0.extendedProps.task_id', $visibleTask->id);
    }

    public function test_calendar_can_update_task_timing(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::query()->create([
            'title' => 'Postponement',
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
            'priority' => 'medium',
            'starts_at' => '2026-02-10 09:00:00',
            'due_at' => '2026-02-10 10:00:00',
        ]);

        $newStart = CarbonImmutable::parse('2026-02-12 14:00:00');
        $newEnd = CarbonImmutable::parse('2026-02-12 16:30:00');

        $this->actingAs($user)
            ->patchJson(route('calendar.tasks.timing', $task), [
                'start' => $newStart->toIso8601String(),
                'end' => $newEnd->toIso8601String(),
            ])
            ->assertOk()
            ->assertJsonPath('event.id', 'task-'.$task->id)
            ->assertJsonPath('event.extendedProps.task_id', $task->id);

        $task->refresh();

        $this->assertTrue($task->starts_at?->equalTo($newStart));
        $this->assertTrue($task->due_at?->equalTo($newEnd));
    }

    public function test_due_only_task_receives_non_zero_duration_event(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::query()->create([
            'title' => 'Deadline only',
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
            'priority' => 'medium',
            'starts_at' => null,
            'due_at' => '2026-02-14 18:00:00',
        ]);

        $response = $this->actingAs($user)->getJson(route('calendar.events', [
            'start' => '2026-02-01T00:00:00+00:00',
            'end' => '2026-02-28T23:59:59+00:00',
        ]));

        $response->assertOk();

        $events = $response->json('events');
        $event = collect($events)->firstWhere('id', 'task-'.$task->id);

        $this->assertIsArray($event);
        $this->assertNotSame($event['start'], $event['end']);
    }

    public function test_calendar_returns_tasks_activities_and_deal_events(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $task = Task::query()->create([
            'title' => 'Calendar mixed: task',
            'creator_id' => $user->id,
            'assignee_id' => $user->id,
            'status' => 'todo',
            'priority' => 'medium',
            'starts_at' => '2026-02-15 09:00:00',
            'due_at' => '2026-02-15 10:00:00',
        ]);

        $pipeline = Pipeline::factory()->create([
            'is_default' => false,
        ]);
        $stage = DealStage::factory()->create([
            'pipeline_id' => $pipeline->id,
            'sort_order' => 1,
        ]);
        $deal = Deal::factory()->create([
            'pipeline_id' => $pipeline->id,
            'stage_id' => $stage->id,
            'owner_id' => $user->id,
            'status' => 'open',
            'expected_close_at' => '2026-02-16',
        ]);

        $activity = Activity::factory()->create([
            'deal_id' => $deal->id,
            'user_id' => $user->id,
            'occurred_at' => '2026-02-17 11:30:00',
            'subject' => 'Calendar mixed: activity',
        ]);

        $response = $this->actingAs($user)->getJson(route('calendar.events', [
            'start' => '2026-02-01T00:00:00+00:00',
            'end' => '2026-02-28T23:59:59+00:00',
        ]));

        $response->assertOk();
        $events = collect($response->json('events'));

        $this->assertTrue($events->contains(fn (array $event): bool => ($event['id'] ?? null) === 'task-'.$task->id));
        $this->assertTrue($events->contains(fn (array $event): bool => ($event['id'] ?? null) === 'activity-'.$activity->id));
        $this->assertTrue($events->contains(fn (array $event): bool => ($event['id'] ?? null) === 'deal-'.$deal->id));
    }
}
