<?php

namespace Tests\Feature\Auth;

use App\Models\User;
use App\Support\TwoFactorAuthenticator;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Carbon;
use Tests\TestCase;

class TwoFactorAuthenticationTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_with_enabled_two_factor_is_redirected_to_challenge_after_password_login(): void
    {
        Carbon::setTestNow('2026-02-27 12:00:00');

        try {
            $twoFactorAuthenticator = app(TwoFactorAuthenticator::class);
            $secret = $twoFactorAuthenticator->generateSecret();

            $user = User::factory()->create([
                'password' => 'password',
                'two_factor_secret' => $secret,
                'two_factor_recovery_codes' => ['ABCD-EFGH'],
                'two_factor_confirmed_at' => now(),
            ]);

            $response = $this->post('/login', [
                'email' => $user->email,
                'password' => 'password',
            ]);

            $this->assertGuest();
            $response->assertRedirect(route('two-factor.challenge', absolute: false));
            $response->assertSessionHas('two_factor_login.user_id', $user->id);
        } finally {
            Carbon::setTestNow();
        }
    }

    public function test_user_can_complete_two_factor_challenge_with_totp_code(): void
    {
        Carbon::setTestNow('2026-02-27 12:00:00');

        try {
            $twoFactorAuthenticator = app(TwoFactorAuthenticator::class);
            $secret = $twoFactorAuthenticator->generateSecret();

            $user = User::factory()->create([
                'password' => 'password',
                'two_factor_secret' => $secret,
                'two_factor_recovery_codes' => ['QWER-TYUI'],
                'two_factor_confirmed_at' => now(),
            ]);

            $this->post('/login', [
                'email' => $user->email,
                'password' => 'password',
            ])->assertRedirect(route('two-factor.challenge', absolute: false));

            $code = $twoFactorAuthenticator->currentCode($secret);

            $response = $this->post('/two-factor-challenge', [
                'code' => $code,
            ]);

            $this->assertAuthenticatedAs($user);
            $response->assertRedirect(route('dashboard', absolute: false));
        } finally {
            Carbon::setTestNow();
        }
    }

    public function test_recovery_code_can_be_used_only_once(): void
    {
        Carbon::setTestNow('2026-02-27 12:00:00');

        try {
            $twoFactorAuthenticator = app(TwoFactorAuthenticator::class);
            $secret = $twoFactorAuthenticator->generateSecret();

            $user = User::factory()->create([
                'password' => 'password',
                'two_factor_secret' => $secret,
                'two_factor_recovery_codes' => ['ABCD-EFGH'],
                'two_factor_confirmed_at' => now(),
            ]);

            $this->post('/login', [
                'email' => $user->email,
                'password' => 'password',
            ]);

            $this->post('/two-factor-challenge', [
                'code' => 'ABCD-EFGH',
            ])->assertRedirect(route('dashboard', absolute: false));

            $user->refresh();
            $this->assertSame([], $user->two_factor_recovery_codes);

            $this->post('/logout')->assertRedirect('/');

            $this->post('/login', [
                'email' => $user->email,
                'password' => 'password',
            ])->assertRedirect(route('two-factor.challenge', absolute: false));

            $this->post('/two-factor-challenge', [
                'code' => 'ABCD-EFGH',
            ])->assertSessionHasErrors('code');

            $this->assertGuest();
        } finally {
            Carbon::setTestNow();
        }
    }
}
