<?php

namespace Tests\Feature\Auth;

use App\Models\MailServiceSetting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class RegistrationTest extends TestCase
{
    use RefreshDatabase;

    public function test_registration_screen_can_be_rendered(): void
    {
        $response = $this->get('/register');

        $response->assertStatus(200);
    }

    public function test_new_users_can_register(): void
    {
        $response = $this->post('/register', [
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ]);

        $this->assertAuthenticated();
        $response->assertRedirect(route('dashboard', absolute: false));
    }

    public function test_registration_auto_creates_mailbox_when_enabled(): void
    {
        MailServiceSetting::query()->create([
            'provider' => 'manual',
            'is_active' => true,
            'domain' => 'mail.crm.local',
            'auto_provision_on_registration' => true,
            'auto_provision_on_user_create' => true,
            'default_status' => 'active',
            'default_quota_mb' => 1024,
        ]);

        $this->post('/register', [
            'name' => 'Mailbox User',
            'email' => 'mailbox.user@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ])->assertRedirect(route('dashboard', absolute: false));

        $user = User::query()->where('email', 'mailbox.user@example.com')->firstOrFail();

        $this->assertDatabaseHas('mailboxes', [
            'user_id' => $user->id,
            'domain' => 'mail.crm.local',
            'is_primary' => true,
        ]);
    }

    public function test_registration_skips_mailbox_when_auto_provision_is_disabled(): void
    {
        MailServiceSetting::query()->create([
            'provider' => 'manual',
            'is_active' => true,
            'domain' => 'mail.crm.local',
            'auto_provision_on_registration' => false,
            'auto_provision_on_user_create' => true,
            'default_status' => 'active',
            'default_quota_mb' => 1024,
        ]);

        $this->post('/register', [
            'name' => 'No Mailbox User',
            'email' => 'nomail.user@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ])->assertRedirect(route('dashboard', absolute: false));

        $user = User::query()->where('email', 'nomail.user@example.com')->firstOrFail();

        $this->assertDatabaseMissing('mailboxes', [
            'user_id' => $user->id,
        ]);
    }
}
