<?php

namespace Tests\Feature\Api;

use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class WarehouseApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_read_only_warehouse_token_can_list_warehouses_but_cannot_create(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $warehouse = Warehouse::factory()->create([
            'manager_id' => $user->id,
            'name' => 'Readonly warehouse',
        ]);

        $token = $user->createToken('warehouses-read', ['warehouses.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/warehouses')
            ->assertOk()
            ->assertJsonPath('data.0.id', $warehouse->id);

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/warehouses', [
                'name' => 'Should fail',
                'status' => 'active',
            ])
            ->assertForbidden();

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/warehouses/'.$warehouse->id.'/addresses', [
                'code' => 'A-01-01',
                'x' => 1,
                'y' => 1,
                'status' => 'free',
            ])
            ->assertForbidden();
    }

    public function test_warehouse_crud_token_can_create_update_and_delete_warehouse(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $token = $user->createToken('warehouses-crud', [
            'warehouses.read',
            'warehouses.create',
            'warehouses.update',
            'warehouses.delete',
        ])->plainTextToken;

        $createResponse = $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/warehouses', [
                'name' => 'API Warehouse',
                'code' => 'WH-API-01',
                'location' => 'East district',
                'capacity' => 5000,
                'status' => 'active',
                'manager_id' => $user->id,
            ]);

        $createResponse
            ->assertCreated()
            ->assertJsonPath('data.name', 'API Warehouse')
            ->assertJsonPath('data.status', 'active');

        $warehouseId = (int) $createResponse->json('data.id');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/warehouses/'.$warehouseId, [
                'name' => 'API Warehouse Updated',
                'code' => 'WH-API-01',
                'location' => 'West district',
                'capacity' => 4200,
                'status' => 'inactive',
                'manager_id' => $user->id,
            ])
            ->assertOk()
            ->assertJsonPath('data.name', 'API Warehouse Updated')
            ->assertJsonPath('data.status', 'inactive');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->deleteJson('/api/v1/warehouses/'.$warehouseId)
            ->assertNoContent();

        $this->assertDatabaseMissing('warehouses', [
            'id' => $warehouseId,
        ]);
    }

    public function test_warehouse_api_can_manage_address_storage_map(): void
    {
        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $warehouse = Warehouse::factory()->create([
            'manager_id' => $user->id,
            'map_rows' => 4,
            'map_columns' => 4,
        ]);

        $token = $user->createToken('warehouses-map', [
            'warehouses.read',
            'warehouses.update',
        ])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/warehouses/'.$warehouse->id.'/map', [
                'map_rows' => 5,
                'map_columns' => 6,
            ])
            ->assertOk()
            ->assertJsonPath('data.map_rows', 5)
            ->assertJsonPath('data.map_columns', 6);

        $createAddressResponse = $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/warehouses/'.$warehouse->id.'/addresses', [
                'code' => 'A-01-01',
                'zone' => 'Main',
                'aisle' => '1',
                'rack' => '1',
                'shelf' => '1',
                'cell' => '1',
                'x' => 1,
                'y' => 1,
                'color' => '#10B981',
                'capacity' => 120,
                'current_load' => 50,
                'status' => 'free',
                'note' => 'Primary bin',
            ]);

        $createAddressResponse
            ->assertCreated()
            ->assertJsonPath('data.code', 'A-01-01')
            ->assertJsonPath('data.x', 1)
            ->assertJsonPath('data.y', 1);

        $addressId = (int) $createAddressResponse->json('data.id');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/warehouses/'.$warehouse->id.'/addresses')
            ->assertOk()
            ->assertJsonPath('data.0.id', $addressId);

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/warehouses/'.$warehouse->id.'/addresses/'.$addressId, [
                'code' => 'A-01-01',
                'zone' => 'Main',
                'aisle' => '1',
                'rack' => '1',
                'shelf' => '2',
                'cell' => '9',
                'x' => 1,
                'y' => 1,
                'color' => '#F97316',
                'capacity' => 120,
                'current_load' => 60,
                'status' => 'reserved',
                'note' => 'Reserved',
            ])
            ->assertOk()
            ->assertJsonPath('data.status', 'reserved')
            ->assertJsonPath('data.cell', '9');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->deleteJson('/api/v1/warehouses/'.$warehouse->id.'/addresses/'.$addressId)
            ->assertNoContent();

        $this->assertDatabaseMissing('warehouse_addresses', [
            'id' => $addressId,
        ]);
    }
}
