<?php

namespace Tests\Feature\Api;

use App\Models\UpdateSetting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Tests\TestCase;
use Tests\Concerns\BuildsUpdatePackages;

class UpdatesApiTest extends TestCase
{
    use RefreshDatabase;
    use BuildsUpdatePackages;

    public function test_update_settings_can_be_read_with_updates_read_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        UpdateSetting::query()->create([
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '25.0.0',
            'channel' => 'stable',
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ]);

        $token = $user->createToken('updates-read', ['updates.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/updates/settings')
            ->assertOk()
            ->assertJsonPath('data.feed_url', 'https://update.crm25.webnet.kz')
            ->assertJsonPath('data.channel', 'stable');
    }

    public function test_update_settings_can_be_updated_with_updates_update_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        UpdateSetting::query()->create([
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '25.0.0',
            'channel' => 'stable',
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ]);

        $token = $user->createToken('updates-update', ['updates.read', 'updates.update'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/updates/settings', [
                'feed_url' => 'update.crm25.webnet.kz',
                'current_version' => '25.1.0',
                'channel' => 'beta',
                'is_active' => true,
                'auto_check_enabled' => true,
                'check_interval_minutes' => 120,
                'request_timeout_seconds' => 6,
                'verify_tls' => true,
                'meta' => ['region' => 'kz'],
            ])
            ->assertOk()
            ->assertJsonPath('data.feed_url', 'https://update.crm25.webnet.kz')
            ->assertJsonPath('data.current_version', '25.1.0')
            ->assertJsonPath('data.channel', 'beta')
            ->assertJsonPath('data.meta.region', 'kz');
    }

    public function test_update_check_can_be_triggered_with_updates_create_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        UpdateSetting::query()->create([
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '25.0.0',
            'channel' => 'stable',
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ]);

        Http::fake([
            'https://update.crm25.webnet.kz/api/v1/crm25/updates*' => Http::response([
                'version' => '25.3.0',
                'build' => '250300',
                'is_update_available' => true,
                'notes' => 'Release available.',
            ], 200),
            '*' => Http::response(['message' => 'not used'], 404),
        ]);

        $token = $user->createToken('updates-check', ['updates.read', 'updates.create'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/updates/checks')
            ->assertCreated()
            ->assertJsonPath('data.status', 'success')
            ->assertJsonPath('data.remote_version', '25.3.0')
            ->assertJsonPath('data.is_update_available', true);

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/updates/checks')
            ->assertOk()
            ->assertJsonCount(1, 'data');
    }

    public function test_update_check_is_forbidden_without_updates_create_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        $token = $user->createToken('updates-read-only', ['updates.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/updates/checks')
            ->assertForbidden();
    }

    public function test_update_installation_can_be_triggered_and_listed(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        UpdateSetting::query()->create([
            'feed_url' => 'https://update.crm25.webnet.kz',
            'current_version' => '1.001',
            'channel' => 'stable',
            'is_active' => true,
            'auto_check_enabled' => true,
            'check_interval_minutes' => 720,
            'request_timeout_seconds' => 8,
            'verify_tls' => true,
        ]);

        $targetPath = $this->createUpdateTargetFixture();
        $package = $this->createUpdatePackageFixture([
            'artisan' => "#!/usr/bin/env php\n<?php echo 'updated';\n",
            'config/app.php' => "<?php\n\nreturn ['name' => 'CRM25'];\n",
            'resources/api-update.txt' => 'api update',
        ]);

        config()->set('updates.target_path', $targetPath);
        config()->set('updates.storage_root', 'updates-testing-api');
        config()->set('updates.run_post_install_commands', false);
        config()->set('updates.backup_enabled', true);

        $this->fakeAvailableUpdate($package['path'], $package['checksum']);

        $token = $user->createToken('updates-install', ['updates.read', 'updates.create'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/updates/installations')
            ->assertCreated()
            ->assertJsonPath('data.status', 'installed')
            ->assertJsonPath('data.target_version', '1.002');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/updates/installations')
            ->assertOk()
            ->assertJsonCount(1, 'data');

        $this->assertSame('1.002', UpdateSetting::query()->value('current_version'));
        $this->assertFileExists($targetPath.'/resources/api-update.txt');
        $this->assertFileExists($targetPath.'/storage/app/crm-version');
        $this->assertSame('1.002', trim((string) File::get($targetPath.'/storage/app/crm-version')));

        File::deleteDirectory($targetPath);
        @unlink($package['path']);
    }
}
