<?php

namespace Tests\Feature\Api;

use App\Models\Project;
use App\Models\Task;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProjectTaskStageApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_api_can_move_project_task_between_stages(): void
    {
        $manager = User::factory()->create([
            'role' => 'manager',
        ]);

        $project = Project::factory()->create([
            'owner_id' => $manager->id,
            'manager_id' => $manager->id,
            'status' => 'active',
        ]);

        $project->members()->sync([
            $manager->id => ['role' => 'owner', 'joined_at' => now()],
        ]);

        $stageTodo = $project->stages()->create([
            'name' => 'Backlog',
            'code' => 'backlog_1',
            'sort_order' => 0,
            'is_done' => false,
        ]);

        $stageDone = $project->stages()->create([
            'name' => 'Done',
            'code' => 'done_2',
            'sort_order' => 1,
            'is_done' => true,
        ]);

        $task = Task::create([
            'title' => 'API move',
            'project_id' => $project->id,
            'project_stage_id' => $stageTodo->id,
            'creator_id' => $manager->id,
            'assignee_id' => $manager->id,
            'status' => 'todo',
            'priority' => 'medium',
        ]);

        $token = $manager->createToken('mobile-app')->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/projects/'.$project->id.'/tasks/'.$task->id.'/stage', [
                'project_stage_id' => $stageDone->id,
            ])
            ->assertOk()
            ->assertJsonPath('data.id', $task->id)
            ->assertJsonPath('data.project_stage.id', $stageDone->id)
            ->assertJsonPath('data.status', 'done');

        $this->assertDatabaseHas('tasks', [
            'id' => $task->id,
            'project_stage_id' => $stageDone->id,
            'status' => 'done',
        ]);
    }
}
