<?php

namespace Tests\Feature\Api;

use App\Models\OneCExchange;
use App\Models\OneCSetting;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class OneCApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_one_c_settings_can_be_read_with_one_c_read_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        OneCSetting::query()->create([
            'provider' => '1c',
            'is_active' => true,
            'exchange_mode' => 'bidirectional',
            'sync_interval_minutes' => 20,
        ]);

        $token = $user->createToken('onec-read', ['onec.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/onec/settings')
            ->assertOk()
            ->assertJsonPath('data.provider', '1c')
            ->assertJsonPath('data.exchange_mode', 'bidirectional');
    }

    public function test_one_c_exchange_crud_works_with_one_c_abilities(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        $token = $user->createToken('onec-full', [
            'onec.read',
            'onec.create',
            'onec.update',
            'onec.delete',
        ])->plainTextToken;

        $createResponse = $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/onec/exchanges', [
                'external_id' => 'onec-api-001',
                'direction' => 'outbound',
                'entity' => 'companies',
                'action' => 'upsert',
                'status' => 'queued',
                'payload' => ['company_id' => 17],
                'response' => ['status' => 'queued'],
                'initiated_by_user_id' => $user->id,
            ])
            ->assertCreated()
            ->assertJsonPath('data.external_id', 'onec-api-001')
            ->assertJsonPath('data.entity', 'companies');

        $exchangeId = (int) $createResponse->json('data.id');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/onec/exchanges/'.$exchangeId, [
                'external_id' => 'onec-api-001',
                'direction' => 'outbound',
                'entity' => 'companies',
                'action' => 'update',
                'status' => 'processing',
                'payload' => ['company_id' => 17, 'step' => 'sync'],
                'response' => ['status' => 'processing'],
                'initiated_by_user_id' => $user->id,
            ])
            ->assertOk()
            ->assertJsonPath('data.action', 'update')
            ->assertJsonPath('data.status', 'processing');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->patchJson('/api/v1/onec/exchanges/'.$exchangeId.'/retry')
            ->assertOk()
            ->assertJsonPath('data.status', 'queued');

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->deleteJson('/api/v1/onec/exchanges/'.$exchangeId)
            ->assertNoContent();

        $this->assertDatabaseMissing('one_c_exchanges', [
            'id' => $exchangeId,
        ]);
    }

    public function test_one_c_exchange_create_is_forbidden_without_one_c_create_ability(): void
    {
        $user = User::factory()->create([
            'role' => 'admin',
        ]);

        $token = $user->createToken('onec-read-only', ['onec.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->postJson('/api/v1/onec/exchanges', [
                'direction' => 'outbound',
                'entity' => 'deals',
                'action' => 'upsert',
                'status' => 'queued',
            ])
            ->assertForbidden();

        $this->assertSame(0, OneCExchange::query()->count());
    }
}

