<?php

namespace Tests\Feature\Api;

use App\Models\Disk;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class DiskApiTest extends TestCase
{
    use RefreshDatabase;

    public function test_read_only_disk_token_can_list_files_but_cannot_upload(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $storedPath = UploadedFile::fake()
            ->create('readme.txt', 4, 'text/plain')
            ->store('disk/shared', 'public');

        $disk = Disk::query()->create([
            'name' => 'readme',
            'original_name' => 'readme.txt',
            'storage_path' => $storedPath,
            'folder' => 'shared',
            'mime_type' => 'text/plain',
            'extension' => 'txt',
            'size' => 4096,
            'description' => null,
            'is_public' => false,
            'owner_id' => $user->id,
        ]);

        $token = $user->createToken('disk-read-only', ['disks.read'])->plainTextToken;

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->getJson('/api/v1/disks')
            ->assertOk()
            ->assertJsonPath('data.0.id', $disk->id);

        $this->withHeaders([
            'Authorization' => 'Bearer '.$token,
            'Accept' => 'application/json',
        ])->post('/api/v1/disks', [
            'file' => UploadedFile::fake()->create('new.txt', 8, 'text/plain'),
            'name' => 'new',
        ])->assertForbidden();
    }

    public function test_disk_token_with_crud_abilities_can_upload_update_and_delete_file(): void
    {
        Storage::fake('public');

        $user = User::factory()->create([
            'role' => 'manager',
        ]);

        $token = $user->createToken('disk-crud', [
            'disks.read',
            'disks.create',
            'disks.update',
            'disks.delete',
        ])->plainTextToken;

        $createResponse = $this->withHeaders([
            'Authorization' => 'Bearer '.$token,
            'Accept' => 'application/json',
        ])->post('/api/v1/disks', [
            'file' => UploadedFile::fake()->create('scope.docx', 32, 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'),
            'name' => 'Scope document',
            'folder' => 'sales',
            'description' => 'Scope file',
            'is_public' => 1,
        ]);

        $createResponse
            ->assertCreated()
            ->assertJsonPath('data.name', 'Scope document')
            ->assertJsonPath('data.folder', 'sales');

        $diskId = (int) $createResponse->json('data.id');
        $this->assertGreaterThan(0, $diskId);

        $disk = Disk::query()->findOrFail($diskId);
        Storage::disk('public')->assertExists($disk->storage_path);

        $this->withHeaders([
            'Authorization' => 'Bearer '.$token,
            'Accept' => 'application/json',
        ])->patch('/api/v1/disks/'.$diskId, [
            'name' => 'Updated scope',
            'folder' => 'sales/2026',
            'is_public' => 0,
        ])->assertOk()
            ->assertJsonPath('data.name', 'Updated scope')
            ->assertJsonPath('data.folder', 'sales/2026')
            ->assertJsonPath('data.is_public', false);

        $this->withHeader('Authorization', 'Bearer '.$token)
            ->deleteJson('/api/v1/disks/'.$diskId)
            ->assertNoContent();

        $this->assertDatabaseMissing('disks', [
            'id' => $diskId,
        ]);
        Storage::disk('public')->assertMissing($disk->storage_path);
    }
}
